/*
$Id: OperatorBase.hh 1958 2008-08-12 00:06:30Z jisuzuki $
*/


#ifndef OPERATORBASE_HH
#define OPERATORBASE_HH
#include "Header.hh"


//!The base class for constructing calculating operators.
/*!
  This class is the template base class for constructing calculating operators.
  The child class of this class should only have functionalities of some
  procedures for data-converter, analysis etc.

  Some source data-objects are inputed into "InputVec" with SetInput( T1 ).
  Input source data-objects will be deleted by the destructor of this class.
  The contents of input source data-objects are extracted and analyzed
  by the child class of this class.
  The output object should be constructed in the child class
  and the output objects are installed into "OutputVec" with "SetOutput(T2*)".
  The output objects will be deleted by the destructor,
  but you should see the following comments.


  The results of the operation can be obtained
  from "T2 Put(UInt4)".
  "Put(UInt4)" will return a data object, and the return objects can install
  data containers and the other operators.
  The original data objects held by OutputVec will be deleted by the destructor
  of this class, but "T2 Put(UInt4)" returns the copy of the results.
 */


template < class T1, class T2 >
class OperatorBase
{
private:
  std::vector< T1* > InputVec;
  std::vector< T2* > OutputVec;
  std::vector<bool> IsCopiedInput;

protected:
  T1 *PutInput( UInt4 index=0 ){ return InputVec[index]; }
  /*!<
    Return the pointer of input data object assigned to "index".
    This method will be called from the child class.
  */

  void SetOutput( T2* OutputObj );
  /*!<
    Set the pointer of output data objects,
    these object will be deleted by the destructor.
    This method will be called from the child class.
   */

public:
  OperatorBase();
  /*!<Constructor. */

  virtual ~OperatorBase();
  /*!< Destructor.
    Delete all data objects, output objects.
    See the comments of "Put(Int4)" and "Duplicate(Int4)".
   */

  void SetInput( T1 InputObj );
  void SetInputP( T1 *InputObj );
  /*!<
    Set input object. The index number is given automatically.
    The input data objects will be copied to dynamical data objects
    and the copy will be installed to the data storage.
    The objects will be deleted by the destructor.
   */

  T2 Put( UInt4 index=0 );
  /*!<
    Return the result data-object assigned to "index".
    If T2-type data container is not installed with "SetOutput(T2*)",
    a T2-type data container will be constructed in this method
    and returned.
    If the value of the argument is too large,
    this method will be return a T2-type data container whose content
    is empty.
   */

  UInt4 InputObjNumber() { return (UInt4)(InputVec.size());  }
  /*!< Returns the size of InputVec.
    The return integer express the number of input objects.
   */
  UInt4 OutputObjNumber(){ return (UInt4)(OutputVec.size()); }
  /*!< Returns the size of OutputVec.
    The return integer express the number of output objects.
  */

  void Clear();
  /*!<
    Clear the two data storage completely.
    All output-objects and the copies of input-objects
    are deleted.
   */
  void ClearOutput();
  /*!<
    Clear the output data storage completely.
    All output-objects re deleted.
   */

};

#include "OperatorBase.cc"
#endif
