#include "MlfMatrixSlicer.hh"


//////////////// Constructor /////////////////////
MlfMatrixSlicer::
MlfMatrixSlicer() :
_valid(false), _datType(-1),_sliced(false), _plane_Xkey("X"),_plane_Ykey("Y"),_line_Xkey("X"),_line_Ykey("Y")
{
    _Initialize();
}


///////////// Constructor with Matrix data //////////////////////
MlfMatrixSlicer::
MlfMatrixSlicer( ElementContainerMatrix *ecm ) :
_valid(false), _datType(0),_sliced(false), _plane_Xkey("X"),_plane_Ykey("Y"),_line_Xkey("X"),_line_Ykey("Y")
{
    _ecm = ecm;
    //std::cout << "@@@ The data is an ElementContainerMatrix." << std::endl;
    _Initialize();
}

///////////// Constructor with Array data //////////////////////
MlfMatrixSlicer::
MlfMatrixSlicer( ElementContainerArray *eca ) :
_valid(false), _datType(1),_sliced(false),_plane_Xkey("X"),_plane_Ykey("Y"),_line_Xkey("X"),_line_Ykey("Y")
{
    _eca = eca;
    //std::cout << "@@@ The data is an ElementContainerArray." << std::endl;
    _Initialize();
}

////////////// Destructor ///////////////////////
MlfMatrixSlicer::
~MlfMatrixSlicer()
{
    if (_sliced ){
        delete _plane;
    }
}

///////////// Initialize /////////////////
void MlfMatrixSlicer::
_Initialize(){
    _data_Keys.resize(3,"");
    std::pair<Int4,Int4> tmp;
    tmp.first=-1;
    tmp.second=-1;
    _data_Keys_Info.resize(3,tmp);
    _isAverage = true;
    _st = new StringTools();
}
void MlfMatrixSlicer::
SetTarget( ElementContainerMatrix* ecm ){
    _ecm = ecm;
}
void MlfMatrixSlicer::
SetTarget( ElementContainerArray* eca ){
    _eca = eca;
}
///////////// Specify keys for 3 axes //////////////////////
bool MlfMatrixSlicer::
SetAxes( std::string keyX, std::string keyY, std::string keyZ )
{
    _valid = false;
    ElementContainer* ec;
    HeaderBase* hh_eca;
    HeaderBase* hh_ec;
    if (_datType == 0){
        // Matrix
        ec = _ecm->PutPointer(0)->PutPointer(0);
        hh_eca = _ecm->PutPointer(0)->PutHeaderPointer();
        hh_ec = ec->PutHeaderPointer();
    }
    else if (_datType == 1){
        // Array
        ec = _eca->PutPointer(0);
    }
    else if (_datType == 2){
        // Matrix in DetectMap type
        ec = _ecm->PutPointer(0)->PutPointer(0);
        hh_eca = _ecm->PutPointer(0)->PutHeaderPointer();
        hh_ec = ec->PutHeaderPointer();
    }
    else{
        // No data
        return false;
    }
    // Check if all keys exist
    _data_Keys[0] = keyX;
    if (keyX!=""){
        Int8 tmp_xi = ec->PutIndexNumber(keyX);
        if (tmp_xi >=0){
            _xi = (UInt4)tmp_xi;
            _data_Keys_Info[0].first = 0;
            _data_Keys_Info[0].second = 0;
        }else{
            if (hh_eca->CheckKey( keyX )==1){
                _data_Keys_Info[0].first = 1;
                _data_Keys_Info[0].second = hh_eca->PutKeyLocation( keyX );
            }else{
                if (hh_ec->CheckKey( keyX )==1){
                    _data_Keys_Info[0].first = 2;
                    _data_Keys_Info[0].second = hh_ec->PutKeyLocation( keyX );
                }else{
                    _data_Keys[0] = "";
                    _data_Keys_Info[0].second = -1;
                    std::cout << "MlfMatrixSlicer::SetAxes > KeyX(" << keyX << ") is not exist. Use index No" << std::endl;
                }
            }
        }
    }

    _data_Keys[1] = keyY;
    if (keyY!=""){
        //std::cout << "keyY="+keyY<<std::endl;
        Int8 tmp_yi = ec->PutIndexNumber(keyY);
        if (tmp_yi >=0){
            _yi = (UInt4)tmp_yi;
            _data_Keys_Info[1].first = 0;
            _data_Keys_Info[1].second = 0;
        }else{
            if (hh_eca->CheckKey( keyY )==1){
                _data_Keys_Info[1].first = 1;
                _data_Keys_Info[1].second = hh_eca->PutKeyLocation( keyY );
            }else{
                if (hh_ec->CheckKey( keyY )==1){
                    _data_Keys_Info[1].first = 2;
                    _data_Keys_Info[1].second = hh_ec->PutKeyLocation( keyY );
                }else{
                    _data_Keys[1] = "";
                    _data_Keys_Info[1].second = -1;
                    std::cout << "MlfMatrixSlicer::SetAxes > KeyY(" << keyY << ") is not exist. Use index No" << std::endl;
                }
            }
        }
    }

    _data_Keys[2] = keyZ;
    if (keyZ!=""){
        Int8 tmp_zi = ec->PutIndexNumber(keyZ);
        if (tmp_zi >=0){
            _zi = (UInt4)tmp_zi;
            _data_Keys_Info[2].first = 0;
            _data_Keys_Info[2].second = 0;
        }else{
            if (hh_eca->CheckKey( keyZ )==1){
                _data_Keys_Info[2].first = 1;
                _data_Keys_Info[2].second = hh_eca->PutKeyLocation( keyZ );
            }else{
                if (hh_ec->CheckKey( keyZ )==1){
                    _data_Keys_Info[2].first = 2;
                    _data_Keys_Info[2].second = hh_ec->PutKeyLocation( keyZ );
                }else{
                    _data_Keys[2] = "";
                    _data_Keys_Info[2].second = -1;
                    std::cout << "MlfMatrixSlicer::SetAxes > KeyZ(" << keyZ << ") is not exist. Use index No" << std::endl;
                }
            }
        }
    }
    /*
    std::cout << "_data_Keys[0] = " << _data_Keys[0] <<std::endl;
    std::cout << "_data_Keys_Info[0].first=" << _data_Keys_Info[0].first << std::endl;
    std::cout << "_data_Keys_Info[0].second=" << _data_Keys_Info[0].second << std::endl;
    std::cout << "_data_Keys[1] = " << _data_Keys[1] <<std::endl;
    std::cout << "_data_Keys_Info[1].first=" << _data_Keys_Info[1].first << std::endl;
    std::cout << "_data_Keys_Info[1].second=" << _data_Keys_Info[1].second << std::endl;
    std::cout << "_data_Keys[2] = " << _data_Keys[2] <<std::endl;
    std::cout << "_data_Keys_Info[2].first=" << _data_Keys_Info[2].first << std::endl;
    std::cout << "_data_Keys_Info[2].second=" << _data_Keys_Info[2].second << std::endl;
    */
    // KeyX is std::vector in EC, KeyY is std::vector in EC, KeyZ is std::vector in EC
    if ((_data_Keys_Info[0].first==0)&&(_data_Keys_Info[1].first==0)&&(_data_Keys_Info[2].first==0)){
        _valid = true;
    // KeyX is std::vector in EC, KeyY and KeyZ are in Header
    }else if ((_data_Keys_Info[0].first==0)&&(_data_Keys_Info[1].first!=0)&&(_data_Keys_Info[2].first!=0)){
        // one of KeyY or KeyZ is in Header of EC, the other is in Header of ECA
        if (_data_Keys_Info[1].first!=_data_Keys_Info[2].first){
            _datType = 2;
            _valid = true;
        }
        if ((_data_Keys_Info[1].first==1)&&(_data_Keys[2]=="")) _data_Keys_Info[2].first=2;
        else if ((_data_Keys_Info[1].first==2)&&(_data_Keys[2]=="")) _data_Keys_Info[2].first=1;
        else if ((_data_Keys_Info[2].first==1)&&(_data_Keys[1]=="")) _data_Keys_Info[1].first=2;
        else if ((_data_Keys_Info[2].first==2)&&(_data_Keys[1]=="")) _data_Keys_Info[1].first=1;
    }else if ((_data_Keys_Info[0].first!=0)&&(_data_Keys_Info[1].first==0)&&(_data_Keys_Info[2].first!=0)){
        if (_data_Keys_Info[0].first!=_data_Keys_Info[2].first){
            _datType = 2;
            _valid = true;
        }
        if ((_data_Keys_Info[0].first==1)&&(_data_Keys[2]=="")) _data_Keys_Info[2].first=2;
        else if ((_data_Keys_Info[0].first==2)&&(_data_Keys[2]=="")) _data_Keys_Info[2].first=1;
        else if ((_data_Keys_Info[2].first==1)&&(_data_Keys[0]=="")) _data_Keys_Info[0].first=2;
        else if ((_data_Keys_Info[2].first==2)&&(_data_Keys[0]=="")) _data_Keys_Info[0].first=1;
    // KeyX ans KeyY are in Header , KeyZ is std::vector
    }else if ((_data_Keys_Info[0].first!=0)&&(_data_Keys_Info[1].first!=0)&&(_data_Keys_Info[2].first==0)){
        // KeyX=KeyY=""
        if ((_data_Keys_Info[0].first==-1)&&(_data_Keys_Info[1].first==-1)){
            // In this pattern, KeyX is index of ECA, KeyY is index of EC
            _data_Keys_Info[0].first = 1;
            _data_Keys_Info[1].first = 2;
            _datType = 2;
            _valid = true;
        }
        if (_data_Keys_Info[0].first!=_data_Keys_Info[1].first){
            _datType = 2;
            _valid = true;
        }
        if ((_data_Keys_Info[0].first==1)&&(_data_Keys[1]=="")) _data_Keys_Info[1].first=2;
        else if ((_data_Keys_Info[0].first==2)&&(_data_Keys[1]=="")) _data_Keys_Info[1].first=1;
        else if ((_data_Keys_Info[1].first==1)&&(_data_Keys[0]=="")) _data_Keys_Info[0].first=2;
        else if ((_data_Keys_Info[1].first==2)&&(_data_Keys[0]=="")) _data_Keys_Info[0].first=1;
    }else{
        std::cerr << "MlfMatrixSlicer::SetAxes > Failed." << std::endl;
    }

    return _valid;
}

//// Slice a plane ////
bool MlfMatrixSlicer::
Slice(ElementContainerArray* ret_eca, std::vector<Double> org, std::vector<Double> ux, std::vector<Double> uy, std::vector<Double> xrange, std::vector<Double> yrange, Double thickness){
    // Is there any plane ?
    if (_sliced ){
        delete _plane;
        _sliced = false;
    }
    // Check if valid data
    if (!_valid){
        if (_datType < 0){
            std::cerr << "MlfMatrixSlicer::Slice > There is no data to slice." << std::endl;
        }
        else{
            std::cerr << "MlfMatrixSlicer::Slice > Keys have not set yet." << std::endl;
        }
        // Return empty array
        return false;
    }

    Double xbin = 0.0;
    Double ybin = 0.0;
    // Check X Range
    if (xrange.size()==3)
        xbin = xrange[2];
    else{
        std::cerr << "MlfMatrixSlicer::Slice > xrange is invalid." << std::endl;
        return false;
    }
    if (xrange[0] > xrange[1]){
        Double tmpx = xrange[0];
        xrange[0] = xrange[1];
        xrange[1] = tmpx;
    }

    // Check Y Range
    if (yrange.size()==3)
        ybin = yrange[2];
    else{
        std::cerr << "MlfMatrixSlicer::Slice > yrange is invalid." << std::endl;
        return false;
    }
    if (yrange[0] > yrange[1]){
        Double tmpy = yrange[0];
        yrange[0] = yrange[1];
        yrange[1] = tmpy;
    }

    // Calc constants of plane equation
    _const_a = (ux[1])*(uy[2]) - (uy[1])*(ux[2]);
    _const_b = (ux[2])*(uy[0]) - (uy[2])*(ux[0]);
    _const_c = (ux[0])*(uy[1]) - (uy[0])*(ux[1]);
    _const_d = (_const_a * org[0] + _const_b * org[1] + _const_c * org[2]) * -1.0;
    //std::cout << "const: " <<  _const_a << " : " <<  _const_b << " : " << _const_b << " : " << _const_b << std::endl;

    _thick2 = thickness / 2.0;

    // calc size of std::vector X
    Double xlength = sqrt((ux[0])*(ux[0]) + (ux[1])*(ux[1]) + (ux[2])*(ux[2]));

    // calc size of std::vector Y
    Double ylength = sqrt((uy[0])*(uy[0]) + (uy[1])*(uy[1]) + (uy[2])*(uy[2]));

    //std::cout << "Length: " <<  xlength << " : " <<  ylength << std::endl;

    // Make unit std::vector of X
    //std::vector<Double> ux = std::vector<Double>(3);
    ux[0] = (ux[0]) / xlength;
    ux[1] = (ux[1]) / xlength;
    ux[2] = (ux[2]) / xlength;
    //  Make unit std::vector of Y
    //std::vector<Double> uy = std::vector<Double>(3);
    uy[0] = (uy[0]) / ylength;
    uy[1] = (uy[1]) / ylength;
    uy[2] = (uy[2]) / ylength;

    // calculate Dot product of both unit vectors
    Double Dot = ux[0]*uy[0] + ux[1]*uy[1] +  ux[2]*uy[2];

    //std::cout << "Dot: " <<  Dot << std::endl;

    // Too small angle
    if ((1.0 - fabs(Dot)) < DBL_MIN * 1000.0){
        std::cerr << "MlfMatrixSlicer::GetPlane > 3 points are on a same line." << std::endl;
        return false;
    }
    // Make x bin
    std::vector<Double> xdat;

    _MakeBin(xdat, xrange, xbin);
    Double xstart = xdat[0] - xbin/2.0;

    // Make y bin
    std::vector<Double> ydat;

    _MakeBin(ydat, yrange, ybin);
    Double ystart = ydat[0] - ybin/2.0;

    // Create an ElementContainerArray Header
    HeaderBase hh_plane;
    // Set xbin data to array header
    hh_plane.Add(_plane_Xkey, xdat);
    // Create an ElementContainerArray to contain results
    if (ret_eca!=NULL) ret_eca->InputHeader( hh_plane );

    UInt4 numX = (UInt4)(xdat.size());
    UInt4 numY = (UInt4)(ydat.size());

    // Prepare integration counter and buffer
    std::vector<std::vector<int> > counter = std::vector<std::vector<int> >(numX, std::vector<int>(numY, 0));
    std::vector<std::vector<Double> > intSum = std::vector<std::vector<Double> >(numX, std::vector<Double>(numY, 0.0));
    std::vector<std::vector<Double> > errSum = std::vector<std::vector<Double> >(numX, std::vector<Double>(numY, 0.0));

    if ((_datType == 0)||(_datType==2)){
        _SlicePlaneFromMatrix(counter, intSum, errSum, org, ux, uy, xbin, ybin, xstart, ystart);
    }
    else{
        _SlicePlaneFromArray(_eca,counter, intSum, errSum, org, ux, uy, xbin, ybin, xstart, ystart);
    }

    // Make Binning
    std::vector<Double> ybinning(numY+1);
    ybinning[0] = ystart;
    for (UInt4 i=1; i<(numY+1); i++) ybinning[i]=ybinning[i-1]+ybin;

    // Devide by counter
    Double xval = xstart;
    for (UInt4 i = 0; i < numX; i++){
        // Create an ElementContainer
        HeaderBase hh_ec;
        std::vector<Double> xrange(2,0.0);
        xrange[0]=xval;
        xrange[1]=xval+xbin;
        xval += xbin;
        hh_ec.Add("XRANGE", xrange );
        hh_ec.Add(_plane_Xkey, xdat[i]);
        ElementContainer* ec = new ElementContainer(hh_ec);
        std::vector<Double> ecInt = std::vector<Double>(numY, 0.0);
        std::vector<Double> ecErr = std::vector<Double>(numY, 0.0);

        for (UInt4 j = 0; j < numY; j++){
            // Exist counter?
            if (counter[i][j] > 0){
                if (_isAverage){
                    ecInt[j] = intSum[i][j]/counter[i][j];
                    ecErr[j] = sqrt(errSum[i][j])/counter[i][j];
                }else{
                    ecInt[j] = intSum[i][j];
                    ecErr[j] = sqrt(errSum[i][j]);
                }
            }else{
                ecErr[j] = -1.0;
            }

        }
        ec->Add(_plane_Ykey, ydat);
        ec->Add(_plane_Ykey+"_bin", ybinning);
        ec->Add("Intensity", ecInt);
        ec->Add("Error", ecErr);
        ec->SetKeys(_plane_Ykey+"_bin","Intensity","Error");
        // Add ec to array(Plane)
        ret_eca->Add(*ec);
    }
    _sliced = true;
    _plane = new ElementContainerArray( (*ret_eca) );
    return true;
}
bool MlfMatrixSlicer::
Slice(ElementContainerArray* eca, PyObject* org, PyObject* ux, PyObject* uy, PyObject* xrange, PyObject* yrange, Double thickness){
    std::vector<Double> org_v = __gCppToPython.ListToDoubleVector( org );
    std::vector<Double> ux_v = __gCppToPython.ListToDoubleVector( ux );
    std::vector<Double> uy_v = __gCppToPython.ListToDoubleVector( uy );
    std::vector<Double> xrange_v = __gCppToPython.ListToDoubleVector( xrange );
    std::vector<Double> yrange_v = __gCppToPython.ListToDoubleVector( yrange );
    bool ret = true;
    if (org_v.size()!=3){
        std::cerr << "MlfMatrixSlicer::Slice > org is invalid." << std::endl;
        ret = false;
    }else if (ux_v.size()!=3){
        std::cerr << "MlfMatrixSlicer::Slice > ux is invalid." << std::endl;
        ret = false;
    }else if (uy_v.size()!=3){
        std::cerr << "MlfMatrixSlicer::Slice > uy is invalid." << std::endl;
        ret = false;
    }else if (xrange_v.size()!=3){
        std::cerr << "MlfMatrixSlicer::Slice > xrange is invalid." << std::endl;
        ret = false;
    }else if (yrange_v.size()!=3){
        std::cerr << "MlfMatrixSlicer::Slice > yrange is invalid." << std::endl;
        ret = false;
    }
    if (ret)
        return Slice( eca, org_v, ux_v, uy_v, xrange_v, yrange_v, thickness );
    else
        return false;
}
ElementContainerArray* MlfMatrixSlicer::
GetPlane(std::vector<Double> org, std::vector<Double> x, std::vector<Double> y, Double xbin, Double ybin,
        std::vector<Double> xrange, std::vector<Double> yrange, Double thickness)
{
    ElementContainerArray* eca = new ElementContainerArray();
    bool ret = GetPlane( eca, org, x, y, xbin, ybin, xrange, yrange, thickness );
    if (ret){
    }else{
        std::cerr << "MlfMatrixSlicer::GetPlane > Failed." << std::endl;
    }
    return eca;
}
bool MlfMatrixSlicer::
GetPlane(ElementContainerArray* ret_eca, std::vector<Double> org, std::vector<Double> x, std::vector<Double> y, Double xbin, Double ybin, std::vector<Double> xrange, std::vector<Double> yrange, Double thickness){
    std::vector<Double> ux(3,0.0);
    std::vector<Double> uy(3,0.0);
    for (UInt4 i=0; i<3; i++){
        ux[i] = x[i]-org[i];
        uy[i] = y[i]-org[i];
    }
    xrange.push_back(xbin);
    yrange.push_back(ybin);
    return Slice(ret_eca, org, ux, uy, xrange, yrange, thickness);
}
bool MlfMatrixSlicer::
GetPlane(ElementContainerArray* eca, PyObject *org, PyObject *x, PyObject *y, Double xbin, Double ybin, PyObject *xrange, PyObject *yrange, Double thickness){
    std::vector<Double> org_v = __gCppToPython.ListToDoubleVector( org );
    std::vector<Double> x_v = __gCppToPython.ListToDoubleVector( x );
    std::vector<Double> y_v = __gCppToPython.ListToDoubleVector( y );
    std::vector<Double> xrange_v = __gCppToPython.ListToDoubleVector( xrange );
    std::vector<Double> yrange_v = __gCppToPython.ListToDoubleVector( yrange );
    bool ret = true;
    if (org_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetPlane > org cannot be converted." << std::endl;
        ret = false;
    }
    if (x_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetPlane > x cannot be converted." << std::endl;
        ret = false;
    }
    if (y_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetPlane > y cannot be converted." << std::endl;
        ret = false;
    }
    if (xrange_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetPlane > xrange cannot be converted." << std::endl;
        ret = false;
    }
    if (yrange_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetPlane > yrange cannot be converted." << std::endl;
        ret = false;
    }

    if (ret) return GetPlane( eca, org_v, x_v, y_v, xbin, ybin, xrange_v, yrange_v, thickness );
    else return false;

}

//// Make bins std::vector////
void MlfMatrixSlicer::_MakeBin(std::vector<Double>& bins, std::vector<Double> binRange, Double bin)
{
    Double start = binRange[0];
    Double end = binRange[1];
    if (binRange[0] > binRange[1]){
        start = binRange[1];
        end = binRange[0];
    }

    int num = (int)(fabs(start)/ bin);

    // Start point must be divisible by bin.
    Double startN =  num * bin;

    // In case of minus
    if (start < 0.0){
        startN = 0.0 - startN;
        if (startN > start){
            startN -= bin;
        }
    }
    while (startN < end){
        bins.push_back(startN);
        startN += bin;
    }
    // Add the last point
    bins.push_back(startN);
}

//// Slice Plane from matrix////
void MlfMatrixSlicer::
_SlicePlaneFromMatrix(std::vector<std::vector<int> > &counter, std::vector<std::vector<Double> > &intSum, std::vector<std::vector<Double> > &errSum,
                        std::vector<Double> org, std::vector<Double> x, std::vector<Double> y, Double xbin, Double ybin,
                        Double xstart, Double ystart)
{

    for (UInt4 i=0; i < _ecm->PutSize(); i++){

        ElementContainerArray* eca = _ecm->PutPointer(i);
        HeaderBase* h_a = eca->PutHeaderPointer();
        // Masked array?
        if ((h_a->CheckKey("MASKED")==1)&&(h_a->PutInt4("MASKED")==1)){
            continue;
        }

        _SlicePlaneFromArray(eca,counter, intSum, errSum, org, x, y, xbin, ybin, xstart, ystart,i);
    }

}
//// Slice Plane from array////
void MlfMatrixSlicer::
_SlicePlaneFromArray(ElementContainerArray* eca, std::vector<std::vector<int> > &counter, std::vector<std::vector<Double> > &intSum, std::vector<std::vector<Double> > &errSum,
                        std::vector<Double> org, std::vector<Double> ux, std::vector<Double>uy, Double xbin, Double ybin, Double xstart, Double ystart, UInt4 eca_ind)
{
    Double sqd = _const_a*_const_a + _const_b*_const_b + _const_c*_const_c;  // sqd never be 0.0.
    Double sqrd = sqrt(sqd);

    UInt4 xlimit = (UInt4)(intSum.size())-1;
    UInt4 ylimit = (UInt4)(intSum[0].size())-1;

    #ifdef MULTH
    omp_set_num_threads( MULTH );
    #endif

    #ifdef _OPENMP
    #pragma omp parallel for

    #if (_OPENMP >= 200805)  // OpenMP 3.0 and later
    for (UInt4 i=0; i < eca->PutSize(); i++){
    #else
    for (Int4 i=0; i < (Int4)(eca->PutSize()); i++){
    #endif

    #else
    // Pararell for
    for (UInt4 i=0; i < eca->PutSize(); i++){
    #endif  // #ifdef _OPENMP
        //std::cout << "Index: " << i << std::endl;

        ElementContainer* ec = eca->PutPointer(i);
        HeaderBase* h_ec = ec->PutHeaderPointer();
        // Masked ec?
        if ((h_ec->CheckKey("MASKED")==1)&&(h_ec->PutInt4("MASKED")==1)){
            continue;
        }
        std::vector<Double> err = ec->PutE();
        std::vector<Double> intY = ec->PutY();
        std::vector<Double>* vx;
        std::vector<Double>* vy;
        std::vector<Double>* vz;
        std::vector<Double>* vv;

        std::vector<bool> isDelete(3,true);
        UInt4 v_index = 0; // X-axis
        if (_data_Keys_Info[v_index].first==0){
            if ( (*ec)(_xi)->size() == intY.size()+1 ){
                vv = new std::vector<Double>( intY.size() );
                for (UInt4 j=0; j<intY.size(); j++) vv->at(j) = ( (*ec)(_xi)->at(j)+(*ec)(_xi)->at(j+1))/2.0;
            }else{
                vv = (*ec)(_xi);
                isDelete[v_index]=false;
            }
        }else if (_data_Keys_Info[v_index].first==1){ // x-val is in ECA Header
            if (_data_Keys_Info[v_index].second<0)
                vv = new std::vector<Double>( intY.size(), (Double)eca_ind );
            else if (_data_Keys_Info[v_index].second<3)
                vv = new std::vector<Double>( intY.size(), eca->PutHeaderPointer()->PutDouble( _data_Keys[v_index] ) );
            else if (_data_Keys_Info[v_index].second==4){
                std::vector<Int4> t1 = eca->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (Double)((t1[0]+t1[1]))/2.0 );
            }
            else{
                std::vector<Double> t1 = eca->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (t1[0]+t1[1])/2.0 );
            }
        }else{  // x-val is in EC Header
            if (_data_Keys_Info[v_index].second<0)
                vv = new std::vector<Double>( intY.size(), (Double)i );
            else if (_data_Keys_Info[v_index].second<3)
                vv = new std::vector<Double>( intY.size(), ec->PutHeaderPointer()->PutDouble( _data_Keys[v_index] ) );
            else if (_data_Keys_Info[v_index].second==4){
                std::vector<Int4> t1 = ec->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (Double)((t1[0]+t1[1]))/2.0 );
            }
            else{
                std::vector<Double> t1 = ec->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (t1[0]+t1[1])/2.0 );
            }
        }
        vx = vv;

        v_index = 1; // Y-axis
        if (_data_Keys_Info[v_index].first==0){
            if ( (*ec)(_yi)->size() == intY.size()+1 ){
                vv = new std::vector<Double>( intY.size() );
                for (UInt4 j=0; j<intY.size(); j++) vv->at(j) = ( (*ec)(_yi)->at(j)+(*ec)(_yi)->at(j+1) )/2.0;
            }else{
                vv = (*ec)(_yi);
                isDelete[v_index]=false;
            }
        }else if (_data_Keys_Info[v_index].first==1){  // y-val is in ECA Header
            if (_data_Keys_Info[v_index].second<0)
                vv = new std::vector<Double>( intY.size(), (Double)eca_ind );
            else if (_data_Keys_Info[v_index].second<3)
                vv = new std::vector<Double>( intY.size(), eca->PutHeaderPointer()->PutDouble( _data_Keys[v_index] ) );
            else if (_data_Keys_Info[v_index].second==4){
                std::vector<Int4> t1 = eca->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (Double)((t1[0]+t1[1]))/2.0 );
            }
            else{
                std::vector<Double> t1 = eca->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (t1[0]+t1[1])/2.0 );
            }
        }else{  // y-val is in EC Header
            if (_data_Keys_Info[v_index].second<0)
                vv = new std::vector<Double>( intY.size(), (Double)i );
            else if (_data_Keys_Info[v_index].second<3)
                vv = new std::vector<Double>( intY.size(), ec->PutHeaderPointer()->PutDouble( _data_Keys[v_index] ) );
            else if (_data_Keys_Info[v_index].second==4){
                std::vector<Int4> t1 = ec->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (Double)((t1[0]+t1[1]))/2.0 );
            }
            else{
                std::vector<Double> t1 = ec->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (t1[0]+t1[1])/2.0 );
            }
        }
        vy = vv;

        v_index = 2; // Z-axis
        if (_data_Keys_Info[v_index].first==0){
            if ( (*ec)(_zi)->size() == intY.size()+1 ){
                vv = new std::vector<Double>( intY.size() );
                for (UInt4 j=0; j<intY.size(); j++) vv->at(j) = ( (*ec)(_zi)->at(j)+(*ec)(_zi)->at(j+1) )/2.0;
            }else{
                vv = (*ec)(_zi);
                isDelete[v_index]=false;
            }
        }else if (_data_Keys_Info[v_index].first==1){  // z-val is in ECA Header
            if (_data_Keys_Info[v_index].second<0)
                vv = new std::vector<Double>( intY.size(), (Double)eca_ind );
            else if (_data_Keys_Info[v_index].second<3)
                vv = new std::vector<Double>( intY.size(), eca->PutHeaderPointer()->PutDouble( _data_Keys[v_index] ) );
            else if (_data_Keys_Info[v_index].second==4){
                std::vector<Int4> t1 = eca->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (Double)((t1[0]+t1[1]))/2.0 );
            }
            else{
                std::vector<Double> t1 = eca->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (t1[0]+t1[1])/2.0 );
            }
        }else{  // z-val is in EC Header
            if (_data_Keys_Info[v_index].second<0)
                vv = new std::vector<Double>( intY.size(), (Double)i );
            else if (_data_Keys_Info[v_index].second<3)
                vv = new std::vector<Double>( intY.size(), ec->PutHeaderPointer()->PutDouble( _data_Keys[v_index] ) );
            else if (_data_Keys_Info[v_index].second==4){
                std::vector<Int4> t1 = ec->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (Double)((t1[0]+t1[1]))/2.0 );
            }
            else{
                std::vector<Double> t1 = ec->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                vv = new std::vector<Double>( intY.size(), (t1[0]+t1[1])/2.0 );
            }
        }
        vz = vv;

        //std::cout << "------ size all=" << vx->size() << "," << vy->size() << "," << vz->size() << std::endl;
        if (vx->size() != vy->size() || vx->size() != vz->size())
        {
            continue;
        }
        for (UInt4 j=0; j < err.size(); j++){
            // Masked point ?
            if ((err[j] < 0.0)||(intY[j]==MLF_MASKVALUE)){
                continue;
            }

            // Calc distance from the plane
            Double point = _const_a * (*vx)[j] +  _const_b * (*vy)[j] + _const_c * (*vz)[j] + _const_d;
            Double dist = fabs(point)/ sqrd;

            // Check if the point is withen a thickness of the plane
            if (dist <= _thick2)
            {
                // Calc projection point on the plane
                Double t = point / sqd;
                Double hx = (*vx)[j] - _const_a * t;
                Double hy = (*vy)[j] - _const_b * t;
                Double hz = (*vz)[j] - _const_c * t;

                // Shift to org
                hx -= org[0];
                hy -= org[1];
                hz -= org[2];

                // Calc projected std::vector length on X-axis
                Double xx = hx * ux[0] + hy * ux[1] + hz * ux[2];
                // calc index of xbin
                UInt4 xindex = (UInt4)(xx/xbin);
                //std::cout <<j << ":" <<  xindex << "/" <<  xlimit << std::endl;
                if (xindex < 0 || xindex > xlimit){
                    // Out of xrange
                    continue;
                }

                // Calc projected std::vector length
                Double yy = hx * uy[0] + hy * uy[1] + hz * uy[2];
                // calc index of ybin
                UInt4 yindex = (UInt4)(yy/ybin);
                //std::cout <<j << ":" <<  yindex << "/" <<  ylimit << std::endl;
                if (yindex < 0 || yindex > ylimit){
                    // out of y range
                    continue;
                }
                //std::cout <<j << ":" <<  xindex << " : " <<  yindex << " : " << intY[j] << std::endl;
                #ifdef _OPENMP
                #pragma omp critical
                {
                #endif
                    // Critical section for paralell
                    intSum[xindex][yindex] += intY[j];
                    errSum[xindex][yindex] += err[j]*err[j];
                    counter[xindex][yindex] += 1;
                #ifdef _OPENMP
                }
                #endif
            }
        }
        if (isDelete[0]) delete vx;
        if (isDelete[1]) delete vy;
        if (isDelete[2]) delete vz;

    }

}

//// Slice Plane as DetectMap////
bool MlfMatrixSlicer::
GetPlaneAsDetectMap( ElementContainerArray* _eca, std::string keyX, std::string keyY, std::string keyZ, std::vector<Double> zrange, bool withBankGap ){
    _valid = false;

    ElementContainer* ec = _ecm->PutPointer(0)->PutPointer(0);
    HeaderBase* hh_eca = _ecm->PutPointer(0)->PutHeaderPointer();
    HeaderBase* hh_ec = ec->PutHeaderPointer();

    if (ec->CheckKey(keyZ)!=1){
        std::cerr << "MlfMatrixSlicer::GetPlaneAsDetectMap > No keyZ in data (" << keyZ << ")" << std::endl;
        return false;
    }

    if (zrange.size()<2){
        std::vector<Double> _v = ec->Put(keyZ);
        zrange.resize(2);
        zrange[0] = _v.front();
        zrange[1] = _v.back();
    }

    bool ret = SetAxes( keyX, keyY, keyZ );
    if ((ret)&&(_data_Keys_Info[0].first==1)&&(_data_Keys_Info[1].first==2)){
    }else{
        return false;
    }

    // Make xaxis and yaxis std::vector
    std::vector<Double> xaxis;
    std::vector<Double> yaxis;
    for (UInt4 i=0; i<(_ecm->PutSize()); i++){
        for (UInt4 j=0; j<(_ecm->PutPointer(i)->PutSize()); j++){
            Double xd = -1.;
            Double yd = -1.;

            if (_data_Keys[0]==""){
                xd = (Double)i;
            }else if (_data_Keys_Info[0].second<3){
                xd = _ecm->PutPointer(i)->PutHeaderPointer()->PutDouble( _data_Keys[0] );
            }else if (_data_Keys_Info[0].second==4){
                std::vector<Int4> vx = _ecm->PutPointer(i)->PutHeaderPointer()->PutInt4Vector( _data_Keys[0] );
                xd = (Double)(vx[0]+vx[1])/2.0;
            }else if (_data_Keys_Info[0].second==5){
                std::vector<Double> vx = _ecm->PutPointer(i)->PutHeaderPointer()->PutDoubleVector( _data_Keys[0] );
                xd = (Double)(vx[0]+vx[1])/2.0;
            }

            if (_data_Keys[1]==""){
                yd = (Double)j;
            }else if (_data_Keys_Info[1].second<3){
                yd = _ecm->PutPointer(i)->PutPointer(j)->PutHeaderPointer()->PutDouble( _data_Keys[1] );
            }else if (_data_Keys_Info[1].second==4){
                std::vector<Int4> vy = _ecm->PutPointer(i)->PutHeaderPointer()->PutInt4Vector( _data_Keys[1] );
                yd = (Double)(vy[0]+vy[1])/2.0;
            }else if (_data_Keys_Info[1].second==5){
                std::vector<Double> vy = _ecm->PutPointer(i)->PutHeaderPointer()->PutDoubleVector( _data_Keys[1] );
                yd = (Double)(vy[0]+vy[1])/2.0;
            }

            xaxis.push_back(xd);
            yaxis.push_back(yd);
        }
    }

    sort( xaxis.begin(), xaxis.end() );
    sort( yaxis.begin(), yaxis.end() );
    decltype(xaxis)::iterator result_xaxis = unique(xaxis.begin(), xaxis.end());
    xaxis.erase(result_xaxis, xaxis.end());
    decltype(yaxis)::iterator result_yaxis = unique(yaxis.begin(), yaxis.end());
    yaxis.erase(result_yaxis, yaxis.end());

    // Make Slice parameters
    Double z_val = (zrange[0]+zrange[1])/2.0;
    Double thickness = fabs( zrange[1]-zrange[0] );
    std::vector<Double> pv_org(3,0.0);
    std::vector<Double> pv_x(3,0.0);
    std::vector<Double> pv_y(3,0.0);

    pv_org[0] = xaxis.front();
    pv_org[1] = yaxis.front();
    pv_org[2] = z_val;
    pv_x[0] = xaxis.back();
    pv_x[1] = yaxis.front();
    pv_x[2] = z_val;
    pv_y[0] = xaxis.front();
    pv_y[1] = yaxis.back();
    pv_y[2] = z_val;

    Double xbin = xaxis[1]-xaxis[0];
    Double ybin = yaxis[1]-yaxis[0];

    std::vector<Double> xrange(2,0.0);
    std::vector<Double> yrange(2,0.0);
    xrange[0] = xaxis.front();
    xrange[1] = xaxis.back();
    yrange[0] = yaxis.front();
    yrange[1] = yaxis.back();

    if (_data_Keys[0]=="")
        _plane_Xkey = "index";
    else
        _plane_Xkey = _data_Keys[0];

    if (_data_Keys[1]=="")
        _plane_Ykey = "index";
    else
        _plane_Ykey = _data_Keys[1];

    ElementContainerArray* eca = new ElementContainerArray();
    ret = GetPlane( eca, pv_org, pv_x, pv_y, xbin, ybin, xrange, yrange, thickness );

    if ( (ret)&&(withBankGap)&&(_plane_Xkey=="index")&&(_plane_Ykey=="index")
         &&(_ecm->PutHeaderPointer()->CheckKey("BANKIDLIST")==1) ){
        _eca->InputHeader( eca->PutHeader() );

        std::vector<Int4> bankIdList = _ecm->PutHeaderPointer()->PutInt4Vector("BANKIDLIST");
        std::vector<Int4> bankSizeList = _ecm->PutHeaderPointer()->PutInt4Vector("BANKSIZELIST");
        if ( bankIdList.size()!=bankSizeList.size() ){
            std::cerr << "MlfMatrixSlicer::GetPlaneAsDetectMap >> invalid Bank Id List in given data." << std::endl;
            delete eca;
            return false;
        }
        _ConvMat.clear();
        std::pair<Int4,Int4> init_conts;
        init_conts.first = -1;
        init_conts.second= -1;

        ElementContainer empty_ec( (*eca).Put(0) );
        empty_ec.AddToHeader("MASKED",1);

        std::vector< std::pair<Int4,Int4> > empty_conts;
        for (UInt4 iy=0; iy<(*_ecm)(0)->PutSize(); iy++){
            std::pair<Int4,Int4> tmp_conts;
            tmp_conts.first = -1;
            tmp_conts.second = -1;
            empty_conts.push_back(tmp_conts);
        }

        UInt4 indexInEca = 0;
        std::vector<Int4> bankSeparators;
        for (UInt4 i=0; i<bankIdList.size(); i++){
            Int4 bankId = bankIdList[i];
            for (UInt4 j=0; j<(UInt4)(bankSizeList[i]); j++){
                std::vector< std::pair<Int4,Int4> > conts;
                Int4 indexInBank = -1;
                std::vector<Double> xrange(2,0);
                xrange[0] = (Double)indexInEca - 0.5;
                xrange[1] = (Double)indexInEca + 0.5;
                for (UInt4 k=0; k<_ecm->PutSize(); k++){
                    if ( ((*_ecm)(k)->PutHeaderPointer()->PutInt4("BANKID") == bankId)&&
                         ((*_ecm)(k)->PutHeaderPointer()->PutInt4("INDEXINBANK") == j) ) indexInBank = k;
                }
                UInt4 y_size = (*_ecm)(0)->PutSize();
                if (indexInBank!=-1){ // Data in Bank
                    for (UInt4 iy=0; iy<y_size; iy++){
                        std::pair<Int4,Int4> tmp_conts;
                        //tmp_conts.first = indexInEca;
                        tmp_conts.first = indexInBank;
                        tmp_conts.second = (Int4)iy;
                        conts.push_back(tmp_conts);
                    }
                    ElementContainer tmp = eca->Put(indexInBank);
                    if (tmp.PutHeaderPointer()->CheckKey("XRANGE")==1) tmp.PutHeaderPointer()->OverWrite("XRANGE",xrange);
                    else tmp.AddToHeader("XRANGE",xrange);
                    _eca->Add( tmp );
                }else{
                    for (UInt4 iy=0; iy<y_size; iy++){
                        std::pair<Int4,Int4> tmp_conts;
                        tmp_conts.first = -1;
                        tmp_conts.second = -1;
                        conts.push_back(tmp_conts);
                    }
                    if (empty_ec.PutHeaderPointer()->CheckKey("XRANGE")==1) empty_ec.PutHeaderPointer()->OverWrite("XRANGE",xrange);
                    else empty_ec.AddToHeader("XRANGE",xrange);
                    _eca->Add( empty_ec );
                }
                indexInEca++;
                _ConvMat.push_back( conts );
            }

            if (i!=(bankIdList.size()-1)){
                std::vector<Double> xrange(2,0);
                xrange[0] = (Double)indexInEca - 0.5;
                xrange[1] = (Double)indexInEca + 0.5;
                if (empty_ec.PutHeaderPointer()->CheckKey("XRANGE")==1) empty_ec.PutHeaderPointer()->OverWrite("XRANGE",xrange);
                else empty_ec.AddToHeader("XRANGE",xrange);
                _eca->Add(empty_ec); // Between Banks
                bankSeparators.push_back(indexInEca-1);
                bankSeparators.push_back(indexInEca);
                _ConvMat.push_back( empty_conts );
            }
        }
        _eca->PutHeaderPointer()->Add("BANKSEPARATOR",bankSeparators);

    }else if (ret){
        // Make Conversion std::map from x,y on sliced plane to i,j of _ecm
        // make retVal _eca
        _eca->InputHeader( eca->PutHeader() );
        for (UInt4 i=0; i<(eca->PutSize()); i++) _eca->Add( eca->Put(i) );

        // initialize
        _ConvMat.clear();
        for (UInt4 x=0; x<(_eca->PutSize() ); x++){
            std::vector< std::pair<Int4,Int4> > tmp;
            std::vector<Double> zv = _eca->PutPointer(0)->PutY();
            for (UInt4 y=0; y<(zv.size()); y++){
                std::pair<Int4,Int4> tmp2;
                tmp2.first = -1;
                tmp2.second= -1;
                tmp.push_back(tmp2);
            }
            _ConvMat.push_back(tmp);
        }
        // Set Conversion std::map
        for (UInt4 i=0; i<(_ecm->PutSize()); i++){
            for (UInt4 j=0; j<(_ecm->PutPointer(i)->PutSize()); j++){
                Double xd = -1.;
                Double yd = -1.;

                if (_data_Keys[0]==""){
                    xd = (Double)i;
                }else if (_data_Keys_Info[0].second<3){
                    xd = _ecm->PutPointer(i)->PutHeaderPointer()->PutDouble( _data_Keys[0] );
                }else if (_data_Keys_Info[0].second==4){
                    std::vector<Int4> vx = _ecm->PutPointer(i)->PutHeaderPointer()->PutInt4Vector( _data_Keys[0] );
                    xd = (Double)(vx[0]+vx[1])/2.0;
                }else if (_data_Keys_Info[0].second==5){
                    std::vector<Double> vx = _ecm->PutPointer(i)->PutHeaderPointer()->PutDoubleVector( _data_Keys[0] );
                    xd = (Double)(vx[0]+vx[1])/2.0;
                }

                if (_data_Keys[1]==""){
                    yd = (Double)j;
                }else if (_data_Keys_Info[1].second<3){
                    yd = _ecm->PutPointer(i)->PutPointer(j)->PutHeaderPointer()->PutDouble( _data_Keys[1] );
                }else if (_data_Keys_Info[1].second==4){
                    std::vector<Int4> vy = _ecm->PutPointer(i)->PutHeaderPointer()->PutInt4Vector( _data_Keys[1] );
                    yd = (Double)(vy[0]+vy[1])/2.0;
                }else if (_data_Keys_Info[1].second==5){
                    std::vector<Double> vy = _ecm->PutPointer(i)->PutHeaderPointer()->PutDoubleVector( _data_Keys[1] );
                    yd = (Double)(vy[0]+vy[1])/2.0;
                }

                Int4 indX = -1;
                Int4 indY = -1;
                for (UInt4 k=0; k<xaxis.size(); k++){
                    if (fabs(xaxis[k]-xd)<1e-30){
                        indX = (Int4)k;
                        break;
                    }
                }
                for (UInt4 k=0; k<yaxis.size(); k++){
                    if (fabs(yaxis[k]-yd)<1e-30){
                        indY = (Int4)k;
                        break;
                    }
                }
                if ((indX<0.)||(indY<0.)){
                    continue;
                }else if ((UInt4)(indX)>=(_ConvMat.size())){
                    std::string msg = "MlfMatrixSlicer::GetPlaneAsDetectMap > indX Over _ConvMat.size() (";
                    msg += _st->UInt4ToString(indX)+"/"+_st->UInt4ToString((UInt4)(_ConvMat.size()));
                    std::cerr << msg << std::endl;
                }else if ((UInt4)(indY)>=(_ConvMat[0].size())){
                    std::string msg = "MlfMatrixSlicer::GetPlaneAsDetectMap > indY Over _ConvMat[0].size() (";
                    msg += _st->UInt4ToString(indY)+"/"+_st->UInt4ToString((UInt4)(_ConvMat[0].size()));
                    std::cerr << msg << std::endl;
                }else{
                    _ConvMat[ (UInt4)indX ][ (UInt4)indY ].first = (Int4)i;
                    _ConvMat[ (UInt4)indX ][ (UInt4)indY ].second = (Int4)j;
                }
            }
        }
    }

    delete eca;
    return ret;
}
bool MlfMatrixSlicer::
GetPlaneAsDetectMap( ElementContainerArray* _eca, std::string keyX, std::string keyY, std::string keyZ, PyObject *zrange, bool withBankGap ){
    std::vector<Double> z_v = __gCppToPython.ListToDoubleVector( zrange );
    if (z_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetPlaneAsDetectMap > zrange cannot be converted." << std::endl;
        return false;
    }else{
        return GetPlaneAsDetectMap( _eca, keyX, keyY, keyZ, z_v, withBankGap );
    }
}
////////// Pick Up index of EC from sliced ECA position ////////////
std::vector<Int4> MlfMatrixSlicer::
PutDetectMapIndex( UInt4 xind, UInt4 yind ){
    std::vector<Int4> ret;
    if ( (xind>=_ConvMat.size()) || (yind>=_ConvMat[0].size()) ){
        std::cerr << "MlfMatrixSlicer::PutDetectorMapIndex > Failed Convert from [x][y] to (i,j) : index over" << std::endl;
        return ret;
    }
    ret.push_back((UInt4)(_ConvMat[xind][yind].first));
    ret.push_back((UInt4)(_ConvMat[xind][yind].second));
    return ret;
}

//// Set Plane ////
void MlfMatrixSlicer::
SetPlane(ElementContainerArray* plane)
{
    if (_sliced )
    {
        delete _plane;
    }
    _sliced = false;
    _valid = true;
    _plane = plane;
}

//////////////// Get Line //////////////////
ElementContainer MlfMatrixSlicer::
GetLine( std::vector<Double> start, std::vector<Double> end, Double bin, Double width )
{

    ElementContainer ec0;

    if (!_valid)
    {
        std::cerr << "MlfMatrixSlicer::GetLine > There is no valid plane." << std::endl;
        return ec0;
    }
    // Get Xdata from plane's header
    HeaderBase* h_plane = _plane->PutHeaderPointer();
    std::vector<Double> xdat = h_plane->PutDoubleVector(0);

    // Make header
    HeaderBase hh_ec;
    hh_ec.Add("STARTPOINT", start);
    hh_ec.Add("ENDPOINT", end);
    ec0.InputHeader(hh_ec);

    // calc size of std::vector
    Double xx = end[0] - start[0];
    Double yy = end[1] - start[1];
    Double vlength = sqrt(xx * xx+ yy * yy);
    //std::cout << "vlength : " <<  vlength << std::endl;
    // Make unit std::vector
    Double ux = xx / vlength;
    Double uy = yy / vlength;

    Double width2 = width / 2.0;

    // Make  bin
    std::vector<Double> x_vec;
    Double vbin = 0.0;
    while ( vbin < vlength)
    {
        x_vec.push_back(vbin);
        vbin += bin;
    }
    // add the last point
    x_vec.push_back(vbin);

    UInt4 numBin = (UInt4)(x_vec.size());
    //std::cout << "numBin : " <<  numBin << std::endl;
    // prepare
    std::vector<int> counter = std::vector<int>(numBin, 0);
    std::vector<Double> y_vec = std::vector<Double>(numBin, 0.0);
    std::vector<Double> e_vec = std::vector<Double>(numBin, 0.0);



    for (UInt4 i=0; i < _plane->PutSize(); i++){
        ElementContainer* ec = _plane->PutPointer(i);
        HeaderBase* h_ec = ec->PutHeaderPointer();
        // Masked ec?
        if ((h_ec->CheckKey("MASKED")==1)&&(h_ec->PutInt4("MASKED")==1)){
            continue;
        }
        std::vector<Double> err = ec->PutE();
        std::vector<Double> intY = ec->PutY();
        // xdata of ec means ydata for plane
        std::vector<Double> ydat = ec->PutX();

        UInt4 index;

        for (UInt4 j=0; j < err.size(); j++){
            // Masked point ?
            if ((err[j] < 0.0)||(intY[j]==MLF_MASKVALUE)){
                continue;
            }
            // shift the origine
            Double xp = xdat[i] - start[0];
            Double yp = ydat[j] - start[1];
            // Calc distance from the line
            Double distance = fabs(xx * yp - yy * xp) / vlength ;


            // Check if in width?
            if (distance > width2){
                continue;
            }
            //std::cout << "distance : " << i << ":" << j << ":" << distance << std::endl;
            // Calc Dot
            Double dot = xp  * ux + yp * uy;
            if (dot >= 0.0){
                index = (UInt4)((dot + bin/2.0) / bin);
                if ( index >= numBin){
                    continue;
                }
            }
            else if (fabs(dot) < bin/2.0){
                index = 0;
            }
            else{
                continue;
            }
            //std::cout << i << " : " << j <<":"<< intY[j] << std::endl;
            y_vec[index] += intY[j];
            counter[index] += 1;
            e_vec[index] += err[j]*err[j];
        }
    }
    for (UInt4 i = 0; i < numBin; i++){
        // Exist counter?
        if ((counter[i] > 0)&&(y_vec[i]<MLF_MASKVALUE)){
            if (_isAverage){
                y_vec[i] /= counter[i];
                e_vec[i] = sqrt(e_vec[i])/counter[i];
            }else{
                e_vec[i] = sqrt(e_vec[i]);
            }
        }
    }

    ec0.Add(_line_Xkey,x_vec);
    ec0.Add(_line_Ykey,y_vec);
    ec0.Add("Error",e_vec);
    ec0.SetKeys(_line_Xkey,_line_Ykey,"Error");

    return ec0;
}
ElementContainer MlfMatrixSlicer::
GetLine( PyObject *start, PyObject *end, Double bin, Double width ){
    std::vector<Double> s_v = __gCppToPython.ListToDoubleVector( start );
    std::vector<Double> e_v = __gCppToPython.ListToDoubleVector( end );
    ElementContainer ret;
    if (s_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetLine > start cannot be converted." << std::endl;
        return ret;
    }
    if (e_v.empty()){
        std::cerr << "MlfMatrixSlicer::GetPlane > end cannot be converted." << std::endl;
        return ret;
    }

    return GetLine( s_v, e_v, bin, width );
}
std::vector<Double> MlfMatrixSlicer::
PutAllAxesRanges(std::string keyX, std::string keyY, std::string keyZ){
    // Check if valid data
    if (!_valid){
        if (_datType < 0){
            std::cerr << "MlfMatrixSlicer::PutAllAxesRanges > There is no data to slice." << std::endl;
        }
        else{
            std::cerr << "MlfMatrixSlicer::PutAllAxesRanges > Keys have not set yet." << std::endl;
        }
        // Return empty array
        std::vector<Double> empty_vec;
        return empty_vec;
    }

    if ((keyX!="")||(keyY!="")||(keyZ!="")){
        if (keyX=="") keyX = _data_Keys[0];
        if (keyY=="") keyY = _data_Keys[1];
        if (keyZ=="") keyZ = _data_Keys[2];
        if (SetAxes(keyX,keyY,keyZ)){
        }else{
            std::cerr << "MlfMatrixSlicer::PutAllAxesRanges > Keys are invalid." << std::endl;
            // Return empty array
            std::vector<Double> empty_vec;
            return empty_vec;
        }
    }
    ElementContainerMatrix* t_ecm=NULL;
    ElementContainerArray*  eca=NULL;
    if ((_datType==0)||(_datType==2)){ // ElementContainerMatrix
        t_ecm = _ecm;
    }else{
        t_ecm = new ElementContainerMatrix();
        t_ecm->Add( *_eca );
    }

    ElementContainer* t_ec = t_ecm->PutPointer(0)->PutPointer(0);
    Double xMin = *(std::min_element( (*t_ec)(_xi)->begin(),(*t_ec)(_xi)->end() ) );
    Double xMax = *(std::max_element( (*t_ec)(_xi)->begin(),(*t_ec)(_xi)->end() ) );
    Double yMin = *(std::min_element( (*t_ec)(_yi)->begin(),(*t_ec)(_yi)->end() ) );
    Double yMax = *(std::max_element( (*t_ec)(_yi)->begin(),(*t_ec)(_yi)->end() ) );
    Double zMin = *(std::min_element( (*t_ec)(_zi)->begin(),(*t_ec)(_zi)->end() ) );
    Double zMax = *(std::max_element( (*t_ec)(_zi)->begin(),(*t_ec)(_zi)->end() ) );

    for (UInt4 i_ecm=0; i_ecm<(t_ecm->PutSize()); i_ecm++){
        eca = t_ecm->PutPointer(i_ecm);
#ifdef MULTH
        omp_set_num_threads( MULTH );
#endif

#ifdef _OPENMP
#pragma omp parallel for

        // Pararell for
#if (_OPENMP >= 200805)  // OpenMP 3.0 and later
        for (UInt4 i=0; i < eca->PutSize(); i++){
#else
        for (Int4 i=0; i < (Int4)(eca->PutSize()); i++){
#endif
#else
        for (UInt4 i=0; i < eca->PutSize(); i++){
#endif  // #ifdef _OPENMP

            ElementContainer* ec = eca->PutPointer(i);
            HeaderBase* h_ec = ec->PutHeaderPointer();
            // Masked ec?
            if ((h_ec->CheckKey("MASKED")==1)&&(h_ec->PutInt4("MASKED")==1)){
                continue;
            }
            UInt4 v_index = 0; // X-axis
            if (_data_Keys_Info[v_index].first==0){
                Double t_xMin = *(std::min_element( (*ec)(_xi)->begin(),(*ec)(_xi)->end() ) );
                Double t_xMax = *(std::max_element( (*ec)(_xi)->begin(),(*ec)(_xi)->end() ) );
                if (t_xMin<xMin) xMin=t_xMin;
                if (t_xMax>xMax) xMax=t_xMax;
            }else if (_data_Keys_Info[v_index].first==1){ // x-val is in ECA Header
                if (_data_Keys_Info[v_index].second<0){
                    if (double(i_ecm)<xMin) xMin=double(i_ecm);
                    if (double(i_ecm)>xMax) xMax=double(i_ecm);
                }else if (_data_Keys_Info[v_index].second<3){
                    Double k=eca->PutHeaderPointer()->PutDouble( _data_Keys[v_index] );
                    if (k<xMin) xMin=k;
                    if (k>xMax) xMax=k;
                }else if (_data_Keys_Info[v_index].second==4){
                    std::vector<Int4> t1 = eca->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<xMin) xMin=k;
                    if (k>xMax) xMax=k;
                }else{
                    std::vector<Double> t1 = eca->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<xMin) xMin=k;
                    if (k>xMax) xMax=k;
                }
            }else{  // x-val is in EC Header
                if (_data_Keys_Info[v_index].second<0){
                    if (double(i)<xMin) xMin=double(i);
                    if (double(i)>xMax) xMax=double(i);
                }else if (_data_Keys_Info[v_index].second<3){
                    Double k = ec->PutHeaderPointer()->PutDouble( _data_Keys[v_index] );
                    if (k<xMin) xMin=k;
                    if (k>xMax) xMax=k;
                }else if (_data_Keys_Info[v_index].second==4){
                    std::vector<Int4> t1 = ec->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<xMin) xMin=k;
                    if (k>xMax) xMax=k;
                }
                else{
                    std::vector<Double> t1 = ec->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<xMin) xMin=k;
                    if (k>xMax) xMax=k;
                }
            }

            v_index = 1; // Y-axis
            if (_data_Keys_Info[v_index].first==0){
                Double t_yMin = *(std::min_element( (*ec)(_yi)->begin(),(*ec)(_yi)->end() ) );
                Double t_yMax = *(std::max_element( (*ec)(_yi)->begin(),(*ec)(_yi)->end() ) );
                if (t_yMin<yMin) yMin=t_yMin;
                if (t_yMax>yMax) yMax=t_yMax;
            }else if (_data_Keys_Info[v_index].first==1){ // y-val is in ECA Header
                if (_data_Keys_Info[v_index].second<0){
                    if (double(i_ecm)<yMin) yMin=double(i_ecm);
                    if (double(i_ecm)>yMax) yMax=double(i_ecm);
                }else if (_data_Keys_Info[v_index].second<3){
                    Double k=eca->PutHeaderPointer()->PutDouble( _data_Keys[v_index] );
                    if (k<yMin) yMin=k;
                    if (k>yMax) yMax=k;
                }else if (_data_Keys_Info[v_index].second==4){
                    std::vector<Int4> t1 = eca->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<yMin) yMin=k;
                    if (k>yMax) yMax=k;
                }else{
                    std::vector<Double> t1 = eca->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<yMin) yMin=k;
                    if (k>yMax) yMax=k;
                }
            }else{  // y-val is in EC Header
                if (_data_Keys_Info[v_index].second<0){
                    if (double(i)<yMin) yMin=double(i);
                    if (double(i)>yMax) yMax=double(i);
                }else if (_data_Keys_Info[v_index].second<3){
                    Double k = ec->PutHeaderPointer()->PutDouble( _data_Keys[v_index] );
                    if (k<yMin) yMin=k;
                    if (k>yMax) yMax=k;
                }else if (_data_Keys_Info[v_index].second==4){
                    std::vector<Int4> t1 = ec->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<yMin) yMin=k;
                    if (k>yMax) yMax=k;
                }
                else{
                    std::vector<Double> t1 = ec->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<yMin) yMin=k;
                    if (k>yMax) yMax=k;
                }
            }

            v_index = 2; // Z-axis
            if (_data_Keys_Info[v_index].first==0){
                Double t_zMin = *(std::min_element( (*ec)(_zi)->begin(),(*ec)(_zi)->end() ) );
                Double t_zMax = *(std::max_element( (*ec)(_zi)->begin(),(*ec)(_zi)->end() ) );
                if (t_zMin<zMin) zMin=t_zMin;
                if (t_zMax>zMax) zMax=t_zMax;
            }else if (_data_Keys_Info[v_index].first==1){ // z-val is in ECA Header
                if (_data_Keys_Info[v_index].second<0){
                    if (double(i_ecm)<zMin) zMin=double(i_ecm);
                    if (double(i_ecm)>zMax) zMax=double(i_ecm);
                }else if (_data_Keys_Info[v_index].second<3){
                    Double k=eca->PutHeaderPointer()->PutDouble( _data_Keys[v_index] );
                    if (k<zMin) zMin=k;
                    if (k>zMax) zMax=k;
                }else if (_data_Keys_Info[v_index].second==4){
                    std::vector<Int4> t1 = eca->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<zMin) zMin=k;
                    if (k>zMax) zMax=k;
                }else{
                    std::vector<Double> t1 = eca->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<zMin) zMin=k;
                    if (k>zMax) zMax=k;
                }
            }else{  // z-val is in EC Header
                if (_data_Keys_Info[v_index].second<0){
                    if (double(i)<zMin) zMin=double(i);
                    if (double(i)>zMax) zMax=double(i);
                }else if (_data_Keys_Info[v_index].second<3){
                    Double k = ec->PutHeaderPointer()->PutDouble( _data_Keys[v_index] );
                    if (k<zMin) zMin=k;
                    if (k>zMax) zMax=k;
                }else if (_data_Keys_Info[v_index].second==4){
                    std::vector<Int4> t1 = ec->PutHeaderPointer()->PutInt4Vector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<zMin) zMin=k;
                    if (k>zMax) zMax=k;
                }
                else{
                    std::vector<Double> t1 = ec->PutHeaderPointer()->PutDoubleVector( _data_Keys[v_index] );
                    Double k = (t1[0]+t1[1])/2.0;
                    if (k<zMin) zMin=k;
                    if (k>zMax) zMax=k;
                }
            }
        } // ec in eca loop
    } // eca in ecm loop

    //std::cout << "MlfMatrixSlicer::PutAllAxesRange >> xMin, xMax = " << xMin << "," << xMax << std::endl;
    //std::cout << "MlfMatrixSlicer::PutAllAxesRange >> yMin, yMax = " << yMin << "," << yMax << std::endl;
    //std::cout << "MlfMatrixSlicer::PutAllAxesRange >> zMin, zMax = " << zMin << "," << zMax << std::endl;

    std::vector<Double> ret(6,0.0);
    ret[0] = xMin;
    ret[1] = xMax;
    ret[2] = yMin;
    ret[3] = yMax;
    ret[4] = zMin;
    ret[5] = zMax;
    return ret;
}
