#!/usr/bin/python3
# -*- coding: utf-8 -*-

import sys
import utsusemi.SAS.ana.Reduction.TransBaseCommands as TBC
import Manyo as mm

# reserved words for return value in this.
DAT = None
ecs = None
EC = None
ret = None
PyObj = None
Iq = None
Tr = None

SASBankDic={"SD":0,"SU":1,"SM16":2,"SM14":3,"SC":4,"MD":5,"MU":6,"ML":7,"MR":8,"HD":9,"HU":10,"HL":11,"BW":13}

#########################################
def TransmittanceCorrection(dat=DAT,params=PyObj,calcError=False):
    """
    Transmittance correction using the function which indicated by params.
    @param dat      (ElementContainerMatrix)
    @param params   (PyObject) Function parameters and their errors
    @param calcError    (bool) True: Calculate an error w/ errors of parameters.
                            False: Calculate it w/o errors of parameters.
    """
    if params[-1] == "exp":
        TBC.TransmittanceCorrectionExp(dat,params,calcError)
    elif params[-1] == "poly3":
        TBC.TransmittanceCorrectionPoly3(dat,params,calcError)
    else:
        raise UserWarning("Unknown fitting function")

#########################################
def TransmittanceCorrectionExp(dat=DAT,params=PyObj,calcError=False):
    """
    Transmittance correction by an exponential function, which is p0*exp(-p1*lambda)+p2.
    @param dat      (ElementContainerMatrix)
    @param params   (PyObject) Function parameters and their errors
    @param calcError    (bool) True: Calculate an error w/ errors of parameters.
                            False: Calculate it w/o errors of parameters.
    """
    if params[-1] == "exp":
        TBC.TransmittanceCorrectionExp(dat,params,calcError)
    elif params[-1] == "poly3":
        raise UserWarning("The choosed funtion dose not match.")
    else:
        raise UserWarning("Unknown fitting function")

#########################################
def TransmittanceCorrectionPoly3(dat=DAT,params=PyObj,calcError=False):
    """
    Transmittance correction by a degree 3 polynomial function, which is p0+p1*lamb+p2*lamb^2+p3*lamb^3.
    @param dat      (ElementContainerMatrix)
    @param params   (PyObject) Function parameters and their errors
    @param calcError    (bool) True: Calculate an error w/ errors of parameters.
                            False: Calculate it w/o errors of parameters.
    """
    if params[-1] == "exp":
        raise UserWarning("The choosed funtion dose not match.")
    elif params[-1] == "poly3":
        TBC.TransmittanceCorrectionPoly3(dat,params,calcError)
    else:
        raise UserWarning("Unknown fitting function")

#########################################
def FitExponentialWoBase(dat=Tr,iniVals="1.0 0.01" ):
    """
    An exponential fitting. The fitting function is p0*exp(-p1*lambda).
    @params dat         (ElementContainer)
    @params iniVals     (str) initial parameters separated with space
    @retval PyObj
    """
    return TBC.FitExponentialWoBase(dat,iniVals,False)

#########################################
def FitExponential(dat=Tr,iniVals="1.0 0.01 0.0001" ):
    """
    An exponential fitting. The fitting function is p0*exp(-p1*lambda)+p2.
    @params dat         (ElementContainer)
    @params iniVals     (str) initial parameters separated with space
    @retval PyObj
    """
    return TBC.FitExponential(dat,iniVals,False)

#########################################
def FitPolynomial3(dat=Tr,iniVals="0.3 -0.05 0.004 -0.0001" ):
    """
    A degree 3 polynomial fitting. The fitting function is p0+p1*lamb+p2*lamb^2+p3*lamb^3.
    @params dat         (ElementContainer)
    @params iniVals     (str) initial parameters separated with space
    @retval PyObj
    """
    return TBC.FitPolynomial3(dat,iniVals, False)

#########################################                                
def GetTransNormByMonCounts(SmplRunNo=104000, DrctRunNo=104000, AxType="lambda", XRange="0.7 7.6 0.1", frameBoundary=0.0, CalibEffi=True ):
    """
    Monitor histogram of sample and direct followed by a transmittance calculation.
    @param SmplRunNo     (int)    Sample measurement run number
    @param DrctRunNo     (int)    Direct beam measurement run number
    @param AxType        (string)  Type must be "tof', "lambda" or "lamda2"
    @param XRange        (string) "startX endX widthX" separated with space (not comma)
    @param frameBoundary (float)
    @param CalibEffi     (bool)    Monitor-detector efficiency calibration
    @retval Tr
    """
    return TBC.GetTransNormByMonCounts(SmplRunNo, DrctRunNo, AxType, XRange, frameBoundary, CalibEffi)

#########################################                                
def GetTransNormByMonCountsTS(SmplRunNo=104000, DrctRunNo=104000, AxType="lambda", XRange="0.7 7.6 0.1", frameBoundary=0.0, CalibEffi=True, SmplTSlice="-1 -1", DrctTSlice="-1 -1"):
    """
    Monitor histogram of sample and direct followed by a transmittance calculation.
    @param SmplRunNo     (int)    Sample measurement run number
    @param DrctRunNo     (int)    Direct beam measurement run number
    @param AxType        (string)  Type must be "tof', "lambda" or "lamda2"
    @param XRange        (string) "startX endX widthX" separated with space (not comma)
    @param frameBoundary (float)
    @param CalibEffi     (bool)    Monitor-detector efficiency calibration
    @param SmplTSlice    (string) Time Slice parameters for Sample run
    @param DrctTSlice    (string) Time Slice parameters for Directrun
    @retval Tr
    """
    return TBC.GetTransNormByMonCountsTS(SmplRunNo, DrctRunNo, AxType, XRange, frameBoundary, CalibEffi, SmplTSlice, DrctTSlice )

#########################################                                
def GetCalcTransmittance(dat=Tr):
    """
    Get trnasmittance data using a fitting result
    @param dat      (ElementContainer) Observed transmittance which was used by fitting facades
    @retval Tr_calc
    """
    return TBC.GetCalcTransmittance(dat)

#########################################                                
def CreateCoefficientSet(coeff="1.0 0.01 0.0001", err="0.1 0.001 0.00001", funcType="exp"):
    """
    Create a coefficient set
    @param coeff    (str) coefficient separated with space
    @param err      (str) error separated with space
    @param funcType (str) function type. "exp" or "poly3"
    @retval PyObj
    """
    return TBC.CreateCoefficientSet(coeff, err, funcType)

########################################
def SetCoeffToTransmittance(dat=Tr, params=PyObj):
    """
    Set a coefficient set to transmittance data container.
    @param dat      (ElementContainer) Observed transmittance
    @param params   (PyObj) Coefficient set
    @retval None
    """
    TBC.SetCoeffToTransmittance(dat, params)

########################################
def GetTransRPMTNormByMon(SmplRunNo=104000, DrctRunNo=104000, PixelRange="134 131 158 155", AxType="lambda", XRange="0.7 7.6 0.1", frameBoundary=0.0, CalibEffi=True ):
    """
    Monitor histogram of sample and direct followed by a transmittance calculation.
    @param SmplRunNo     (int)    Sample measurement run number
    @param DrctRunNo     (int)    Direct beam measurement run number
    @param PixelRange    (string) "x0 y0 x1 y1"
    @param AxType        (string)  Type must be "tof', "lambda" or "lamda2"
    @param XRange        (string) "startX endX widthX" separated with space (not comma)
    @param frameBoundary (float)
    @param CalibEffi     (bool)    Monitor-detector efficiency calibration
    @retval Tr
    """
    return TBC.GetTransRPMTNormByMon(SmplRunNo, DrctRunNo, PixelRange, AxType, XRange, frameBoundary, CalibEffi )

# Dictionary for entry the name of functions
_functions = {
            "TransmittanceCorrection":TransmittanceCorrection,
            "TransmittanceCorrectionExp":TransmittanceCorrectionExp,
            "TransmittanceCorrectionPoly3":TransmittanceCorrectionPoly3,
            "FitExponentialWoBase":FitExponentialWoBase,
            "FitExponential":FitExponential,
            "FitPolynomial3":FitPolynomial3,
            "GetTransNormByMonCounts":GetTransNormByMonCounts,
            "GetTransNormByMonCountsTS":GetTransNormByMonCountsTS,
            "GetTransRPMTNormByMon":GetTransRPMTNormByMon,
            "GetCalcTransmittance":GetCalcTransmittance,
            "CreateCoefficientSet":CreateCoefficientSet,
            "SetCoeffToTransmittance":SetCoeffToTransmittance
            }
