#!/usr/bin/python3
# -*- coding: utf-8 -*-
from __future__ import print_function
import time
import os
import Manyo
import Manyo.Utsusemi as mu
import math

import utsusemi.ana.Reduction.BaseCommandsInEla as BaseCom
import utsusemi.SIK.ana.Reduction.BaseCommandsSIK as BaseComSIK

# special reserved word for commands history
HISTORY = None

# reserved words for return value in this.
DAT = None
ecs = None
EC = None
ret = None

########################################


def CopyData(first_run=10, last_run=-1, target="-"):
    """
    Copy EventData file from DAQ0 or DAQ1 into /data/XXX or target
    @param first_run (int) First Run No of a measurement to be copied
    @param last_run (int) Last Run No of a measurement to be copied
    @param target (string) target folder
    @retval None
    """
    BaseCom.CopyData(first_run, last_run, 0, target)

#########################################


def GetNeunetHist(runNo="197", HistParam="tof,0,40000,100", DetParam="psd", DetRange="All", TimeRange="All", FrameInfo="None", MaskInfo="Default", CaseInfo="None", BGInfo="TimeDep:None", TofShift="None", AnaMode="0,-,-", Options="-"):
    """
    Convert Event data of NEUNET to Histogram
    GetNeunetHist(runNo="197,198",
                  HistParam="tof,0,40000,100",
                  DetParam="psd",
                  DetRange="All",
                  TimeRange="All",
                  FrameInfo="None",
                  MaskInfo="Default"
                  CaseInfo="None",
                  BGInfo="TimeDep:None",
                  TofShift="None",
                  AnaMode="0,-,-",
                  Options="-"
                  )

    -----------------------
    HistParam format
        TOF
           delta-TOF const :  tof,<min_tof>,<max_tof>,<delta_tof>
           d-TOF/TOF const : rtof,<min_tof>,<max_tof>,<racio>

        TOF with time forcusing
           delta-TOF const : tf-tof,<min_tof>,<max_tof>,<delta_tof>
           d-TOF/TOF const : tf-rtof,<min_tof>,<max_tof>,<racio>

        Lambda
           delta-lambda const        :  lambda,<min_lambda>,<max_lambda>,<delta_lambda>
           delta-lambda/lambda const : rlambda,<min_lambda>,<max_lambda>,<racio>

        Energy            : energy,<min_energy>,<max_energy>,<delta_energy>

        Momentum Transfer : q,<min_Q>,<max_Q>,<delta_Q>[,<ki_x>,<ki_y>,<ki_z>]

        Energy Transfer   : hw,<Ei>,<min_hw>,<max_hw>,<delta_hw>[,<t0_shift>]

        d-value
            delta-d-value                 : d,<min_d-val>,<max_d-val>,<delta_d-val>
            delta-d-value / d-value const : rd,<min_d-val>,<max_d-val>,<ratio>

    -----------------------
    DetParam format
        "<detType>[[<LLD_min>:<LLD_max>]][,..]" : detType in {"PSD","MON","PSD-PH","MON-PH", "UPSD", "TRIGNET"}
                                                "-PH" means that Pulse Height is calculated at the same time
    -----------------------
    DetRange format
        "All[:<number_of_pixels>]"                     : use all detectors
        "<startDetId>-<endDetId>[:<number_of_pixels>]" : use detectors from <startDetId> to <endDetId>
                                                         "X-Y:Z,A-B:C,D-E:F..." can be used for more settings
    -----------------------
    TofShift format
        "<ptnId>[,<p1>,<p2>,...]"
             ptnId = 0 : no use in this facade( only use if scattered data of tof origin shift vs lambda is given )
             ptnId = 1 : use shift = p1/(exp(-p2*(lambda-p3)) +p4) + p5*lambda
                         default p1=5.0, p2=8.464, p3=2.08765, p4=1.23477, p5=5.32657
    -----------------------
    Options format
        "<KEY>:<VALUE>[,<KEY>:<VALUE>,..]"
             KEY                VALUE example       descriptions
             "ISCHECKPULSEID"   "True"    [bool]    Set flag to check T0 between neutron data and trignet data.
             "ISADDTOFBIN"      "True"    [bool]    Set flag to add TOF binning vector to each ElementContainer
             "L1"               "18030.0" [mm]      Change L1
             "SX"               "5.0"     [mm]      Set x axis of sample position
             "SY"               "-1.0"    [mm]      Set y axis of sample position
             "SZ"               "2.0"     [mm]      Set z axis of sample position

    @param runNo     (string) Run Numbers
    @param HistParam (string) "type,param,param,param,..." type must be in "tof" "rtof" "lambda" "rlambda" "hw" "energy" "q" "d" "rd"
    @param DetParam  (string) "DetType[LLD:HLD]" DetType must be in "psd","n2mon","psd-ph" or "n2mon-ph"
    @param DetRange  (string) "All[:number_of_pixels]" or "startPsdId-endPsdId[:number_of_pixels]"
    @param TimeRange (string) "All" or "startTime,endTime" in [second] or "YYYY/MM/DD,hh:mm:ss,YYYY/MM/DD,hh:mm:ss"
    @param FrameInfo (string) "None", "<FrameNumber>, <boudaryTime>" or "<type>,<boundaryValue>" where type="tof", "Ei", "Energy", "Lambda"
    @param MaskInfo  (string) "<maskfile> [,<TOFmask>-<TOFmask>,..]" : usually use "default" to use suitable one. If no use maskfile, set "NoFile".
    @param CaseInfo  (string) CaseInfo file must be in ${UTSUSEMI_BASE_DIR}/${UTSUSEMI_INST_CODE}/ana/xml or ${UTSUSEMI_HOME_DIR}/ana/xml
    @param BGInfo    (string) "TimeDep:None" TimeDep:tof1-tof2
    @param TofShift  (string) "None" or "<ptnId> [,<p1>,<p2>,...]"
    @param AnaMode   (string or int) change analysis mode, which means to swich WiringInfo and DetectorInfo to be used for histogram.
    @param Options   (string)
    @retval DAT (ElementContainerMatrix)
    """
    mu.UtsusemiMessage('Cmm.GetNeunetHist("{}","{}","{}","{}","{}","{}","{}","{}","{}","{}","{}","{}")'.format(
        runNo, HistParam, DetParam, DetRange, TimeRange, FrameInfo, MaskInfo, CaseInfo, BGInfo, TofShift, AnaMode, Options), True)
    return BaseCom.GetNeunetHist(runNo=runNo,
                                 HistParam=HistParam,
                                 DetParam=DetParam,
                                 DetRange=DetRange,
                                 TimeRange=TimeRange,
                                 FrameInfo=FrameInfo,
                                 MaskInfo=MaskInfo,
                                 CaseInfo=CaseInfo,
                                 BGInfo=BGInfo,
                                 TofShift=TofShift,
                                 AnaMode=AnaMode,
                                 Options=Options)


#########################################
def GetMultiEi2(runNo="0", expEi="0.0", PSD="96,159", Pixel="30,70", Range=5.0, deltaE=0.1, PlotFlag=1, ProfType=2):
    """
    Get Incident Energy from raw event data after converting it to energy histogram (R.Kajimoto)
    @param runNo (string) run number
    @param expEi (string) initial Ei or multi Ei with comma as separater
    @param PSD   (string) range of PSDs to be used for fitting
    @param Pixel (string) range of Pixels to be used for fitting
    @param Range (double) range [%] relative to Ei used for fitting
    @param deltaE (double) Energy bin [%] relative to Ei
    @param PlotFlag (int) if 1, plot graph
    @param ProfType (int) profile type. 1: Gaussian, 2: Toraya's split pseudo Voigt function
    @retval None
    """
    mu.UtsusemiMessage('Cmm.GetMultiEi2("{}","{}","{}","{}",{},{},{},{})'.format(
        runNo, expEi, PSD, Pixel, Range, deltaE, PlotFlag, ProfType), True)
    BaseCom.GetMultiEi2(runNo, expEi, PSD, Pixel, Range,
                        deltaE, PlotFlag, ProfType)

#########################################


def GetMultiEi(runNo=0, expEi="0.0", PSD="75", PIXEL="52", NumOfPixel=100, Resolution=0.5, PlotFlag=0):
    """
    Get Incident Energy from raw event data
    @param runNo (int) run number
    @param expEi (string) Initial Ei or multi Ei with comma as separater
    @param PSD   (string) range of PSDs to be used for fitting
    @param PIXEL (string) range of Pixels to be used for fitting
    @param NumOfPixel (int) the number of pixels in one PSD
    @param Resolution (double) resolution [%] for tof binning
    @param PlotFlag (int) if 1, plot graph
    @retval None
    """
    mu.UtsusemiMessage('Cmm.GetEiFromEvent({},"{}","{}",{},{},{})'.format(
        runNo, expEi, PSD, PIXEL, NumOfPixel, Resolution, PlotFlag), True)
    BaseCom.GetEiFromEvent(runNo, expEi, PSD, PIXEL,
                           NumOfPixel, Resolution, PlotFlag)

#########################################


def GetEi(dat=DAT, expEi=50.0, MonPsd=75, MonPixel=52, L1=18.03, T0shift=0.0, FittingType=0, Debug=0):
    """
    Get Incident Energy from Monitor Counter data
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  expEi        (Double)  expected incident energy
    @param  MonPsd       (int) PSD ID to be used as monitor
    @param  MonPixel     (int) Pixel No to be used as monitor
    @param  L1           (double) Distance between moderator to sample [m]
    @param  T0shift      (double) T0 offset [micro-sec]
    @param  FittingType  (UInt4)  Gaussian:0
    @param  Debug        (UInt4)  OFF:0, ON:1
    """
    mu.UtsusemiMessage('Cmm.GetEi(<ElementContainerMatrix>,{},{},{},{},{},{})'.format(
        MonPsd, MonPixel, L1, T0shift, FittingType, Debug), True)
    BaseCom.GetEi(dat, expEi, MonPsd, MonPixel,
                  L1, T0shift, FittingType, Debug)

##########################################


def NormByBeamCurrent(dat=DAT, factor=1000.0, offset_sec=0.0, isCT8n=True, isTagCorrect=True):
    """
    Normailze by Beam current from web and calculate summation of the number of shots and protons. Experimental info file, like SIK000XXX.xml file, must be placed in your ~/exp/xml directory.
    @param dat [Def:DAT] (ElementContainer,-Array,-Matrix)
    @param factor     (double) normalize factor data*factor/beamCurrent
    @param offset_sec (double) offset time
    @param isCT8n (bool) use CT8Neutron (corrected by muon target effect) or not
    @param isTagCorrect (bool) use corrected proton current by shot tags
    @retval None
    """
    mu.UtsusemiMessage('Cmm.NormByBeamCurrent(<ElementContainerMatrix>,{},{},{},{})'.format(
        factor, offset_sec, isCT8n, isTagCorrect), True)
    BaseCom.NormByBeamCurrent(dat, factor, offset_sec, isCT8n, isTagCorrect)

#########################################


def NormByKickers(dat=DAT, factor=1.0):
    """
    Normalize data intensity by the number of kickers

    @params dat [Def:DAT] (ElementContainerMatrix)
    @params factor (double) kickers = <original kickers>*<factor>
    @retval None
    """
    mu.UtsusemiMessage(
        'Cmm.NormByKickers(<ElementContainerMatrix>,{})'.format(factor), True)
    BaseCom.NormByKickers(dat, factor)

#########################################


def DoMask(dat=DAT, filename="mask.txt"):
    """
    Add MaskInfo to the data
    @param dat [Def:DAT] (ElementContainerMatrix)
    @param filename (string) filename of mask information
    """
    mu.UtsusemiMessage(
        'Cmm.DoMask(<ElementContainerMatrix>,"{}")'.format(filename), True)
    BaseCom.DoMask(dat, filename)

#########################################


def SolidAngleCorrection(dat=DAT):
    """
    Correction for solid angle of pixel
    Each intensity is devided by solid angle * default value
    @param dat [Def:DAT] (ElementContainerMatrix)
    @retval None
    """
    mu.UtsusemiMessage(
        'Cmm.SolidAngleCorrection(<ElementContainerMatrix>)', True)
    BaseCom.SolidAngleCorrection(dat)

#########################################


def MakeWhiteVanCorr(dat=DAT, Emin=20.0, Emax=40.0):
    """
    Make data of White Vanadium correction

    @param dat [Def:DAT] (ElementContainerMatrix) target data
    @retval None
    """
    BaseCom.MakeWhiteVanCorr(dat, Emin, Emax)

#########################################


def WhiteVanCorr(dat=DAT, runno=0):
    """
    White Vanadium correction

    @param dat [Def:DAT] (ElementContainerMatrix) target data
    @param runno (int) RunNo for white vanadium
    @retval None
    """
    BaseCom.WhiteVanCorr(dat, runno)

#########################################


def KiKfCorrection(dat=DAT):
    """
    Ki/Kf correction for the inelastic scattering
    @param dat [Def:DAT] (ElementContainerMatrix)
    """
    mu.UtsusemiMessage('Cmm.KiKfCorrection(<ElementContainerMatrix>)', True)
    BaseCom.KiKfCorrection(dat)

#########################################


def dHWCorrection(dat=DAT):
    """
    1/(delta-hw) correction for the inelastic scattering
    @param dat [Def:DAT] (ElementContainerMatrix)
    """
    mu.UtsusemiMessage('Cmm.dHWCorrection(<ElementContainerMatrix>)', True)
    BaseCom.dHWCorrection(dat)

##########################################


def BoseFactorCorrection(dat=DAT, T=300.0):
    """
    Functon to do correction of Bose Factor

    @param dat [Def:DAT] (ElementContainerMatrix)
    @param T   (double)  [K]
    """
    mu.UtsusemiMessage(
        'Cmm.BoseFactorCorrection(<ElementContainerMatrix>,{})'.format(T), True)
    BaseCom.BoseFactorCorrection(dat, T)

##########################################


def VisContOutputTextByPhi(dat=DAT, param_file="param.xml", phi=0.0, output_file="./"):
    """
    Output text data using VisualCont function

    @param dat [Def:DAT] (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param phi (float) angle phi
    @param output_file (string) Path to output file
    @return None
    """
    mu.UtsusemiMessage('Cmm.VisContOutputTextByPhi(<ElementContainerMatrix>,"{}",{},"{}")'.format(
        param_file, phi, output_file), True)
    BaseCom.VisContOutputTextByPhi(dat, param_file, phi, output_file)

#########################################


def VisContOutputFileByPhi(dat=DAT, param_file="param.xml", phi=0.0, output_file="./", hasMuchMem=True):
    """
    Output data to a file using VisualContCommon function
    [inamura 110425]
    @param dat [Def:DAT] (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param phi (float) angle phi
    @param output_file (string) Path to output file
    @param hasMuchMem (Bool) if Ture, method to output occupies much memory.
    @return None
    """
    mu.UtsusemiMessage('Cmm.VisContOutputFileByPhi(<ElementContainerMatrix>,"{}",{},"{}",{})'.format(
        param_file, phi, output_file, hasMuchMem), True)
    BaseCom.VisContOutputFileByPhi(
        dat, param_file, phi, output_file, hasMuchMem)

#########################################


def VisContMOutputFileByPhi(dat=DAT, param_file="param.xml", phi=0.0, output_file="./"):
    """
    Output data to a file using VisualContM function
    [inamura 110425]
    @param dat [Def:DAT] (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param phi (float) angle phi
    @param output_file (string) Path to output file
    @return None
    """
    mu.UtsusemiMessage('Cmm.VisContMOutputFileByPhi(<ElementContainerMatrix>,"{}",{},"{}")'.format(
        param_file, phi, output_file), True)
    BaseCom.VisContMOutputFileByPhi(dat, param_file, phi, output_file)

##########################################


def BoseFactCorrEC(dat=EC, T=300.0):
    """
    Functon to do Bose Factor correction on ElementContainer

    @param dat [Def:EC] (ElementContaienr)
    @param T   (double) [K]
    @retval None
    """
    mu.UtsusemiMessage(
        'Cmm.BoseFactCorrEC(<ElementContainer>,{})'.format(T), True)
    BaseCom.BoseFactCorrEC(dat, T)

##########################################


def ToPowder(dat=DAT, startQ=0.0, endQ=10.0, deltaQ=0.1):
    """
    Convert data from crystal to powder

    @param dat [Def:DAT] (ElementContaienrMatrix)
    @param startQ (double) Q value of range
    @param endQ   (double) Q value of range
    @param deltaQ (double) binning of Q
    @retval ECM   (ElementContainerMatrix)
    """
    mu.UtsusemiMessage('Cmm.ToPowder(<ElementContainerMatrix>,{},{},{})'.format(
        startQ, endQ, deltaQ), True)
    return BaseCom.ToPowder(dat, startQ, endQ, deltaQ)

#########################################


def FlexCalc(dat=DAT, Intensity="I", Error="Err"):
    """
    Flexible calculation of S(Q,hw) on ElementContainerMatrix.
    @param dat [Def:DAT] (ElementContainerMatrix)
    @param Intensity (string) the expression of Intensity calculation
    @param Error     (string) the expression of Error calculation
    @retval dat (ElementContainerMatrix)
    """
    mu.UtsusemiMessage('Cmm.FlexCalc(<ElementContainerMatrix>,"{}","{}")'.format(
        Intensity, Error), True)
    return BaseCom.FlexCalc(dat, Intensity, Error)

#########################################


def GetPH(dat=DAT, binSize=1):
    """
    Get Pulse Height Profile of PSD
    Each PSD pulse heignt profile is stored in header of ElementContaienrArray.
    @param  dat [Def:DAT] (ElementContainerMatrix) Loaded ElementContainerMatrix.
    @param  binSize (UInt4)
    @retval ECS    ElementContainer or list of ElementContainers of pulse height profile
    """
    mu.UtsusemiMessage(
        'Cmm.GetPH(<ElementContainerMatrix>,{})'.format(binSize), True)
    return BaseCom.GetPH(dat, binSize)

##########################################


def NormByBeam(dat=DAT, start_day="2009/5/30", start_time="10:00:00", end_day="2009/5/30", end_time="11:00:00", factor=1.0, offset_sec=0.0, isCT8n=True, isTagCorrect=True):
    """
    Normailze by Beam current from web and calculate summation of the number of shots and protons
    @param dat [Def:DAT] (ElementContainer,-Array,-Matrix)
    @param start_day  (string) the day of measurement start [yyyy/mm/dd]
    @param start_time (string) the time of measurement start [hh:mm:ss]
    @param end_day    (string) the day of measurement end [yyyy/mm/dd]
    @param end_time   (string) the time of measurement end [hh:mm:ss]
    @param factor     (float) normalize factor data*factor/beamCurrent
    @param offset_sec (float) offset second for Measurement Period to get protons
    @param isCT8n (bool) use CT8Neutron (corrected by muon target effect) or not
    @param isTagCorrect (bool) use corrected proton current by shot tags
    @retval None
    """
    mu.UtsusemiMessage('Cmm.NormByBeam(<ElementContainerMatrix>,"{}","{}","{}","{}",{},{},{},{})'.format(
        start_day, start_time, end_day, end_time, factor, offset_sec, isCT8n, isTagCorrect), True)
    try:
        BaseCom.NormByBeam(dat, start_day, start_time, end_day,
                           end_time, factor, offset_sec, isCT8n, isTagCorrect)
    except:
        msg = "Normalize by Proton Current is disable."
        raise UserWarning(msg)

#########################################


def SumOfTOF(Target=DAT, PSD=1, tof_min=0, tof_max=40000):
    """
    Put summation of intensity along TOF
    @param  Target [Def:DAT] ElementContainerMatrix
    @param  PSD        PSD for summation
    @param tof_min     Minimum point of TOF for summation
    @param tof_max     Maximum point of TOF for summation
    @retval EC         ElementContainer
    """
    mu.UtsusemiMessage('Cmm.SumOfTOF(<ElementContainerMatrix>,{},{},{})'.format(
        PSD, tof_min, tof_max), True)
    return BaseCom.SumOfTOF(Target, PSD, tof_min, tof_max)


#########################################
def SumOfPixel(Target=DAT, PSD_min=1, PSD_max=1, Pixel_min=1, Pixel_max=1):
    """
    Put summation of intensity along Pixel
    @param Target [Def:DAT] ElementContainerMatrix
    @param PSD_min    lower limit of PSD range
    @param PSD_max    upper limit of PSD range
    @param Pixel_min  lower limit of Pixel range
    @param Pixel_max  upper limit of Pixel range
    @retval EC (ElementContainer)
    """
    mu.UtsusemiMessage('Cmm.SumOfPixel(<ElementContainerMatrix>,{},{},{},{})'.format(
        PSD_min, PSD_max, Pixel_min, Pixel_max), True)
    return BaseCom.SumOfPixel(Target, PSD_min, PSD_max, Pixel_min, Pixel_max)


#########################################
def PickUpRegion(Target=DAT, PSD_min=1, PSD_max=1, Pixel_min=1, Pixel_max=1):
    """
    Put summation of intensity along Pixel
    @param Target [Def:DAT] ElementContainerMatrix
    @param PSD_min    lower limit of PSD range
    @param PSD_max    upper limit of PSD range
    @param Pixel_min  lower limit of Pixel range
    @param Pixel_max  upper limit of Pixel range
    @retval ECM       ElementContainerMatrix
    """
    mu.UtsusemiMessage('Cmm.PickUpRegion(<ElementContainerMatrix>,{},{},{},{})'.format(
        PSD_min, PSD_max, Pixel_min, Pixel_max), True)
    return BaseCom.PickUpRegion(Target, PSD_min, PSD_max, Pixel_min, Pixel_max)

#########################################


def SaveDataToSPE(dat=DAT, path="./", filename="Sample.spe", phxfile="default", parfile="default"):
    """
    Save ElementContainerMatrix to SPE and phx file.
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  path     (String) path to data file
    @param  filename (String) SPE data file name
    @param  phxfile  (String) PHX file name ("None": not output, "default" means using same name of SPE file)
    @param  parfile  (String) PAR file name ("None": not output, "default" means using same name of SPE file)
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SaveDataToSPE(<ElementContainerMatrix>,"{}","{}","{}","{}")'.format(
        path, filename, phxfile, parfile), True)
    BaseCom.SaveDataToSPE(dat, path, filename, phxfile, parfile)

#########################################


def LoadDataFromSPE(path="./", spefile="Sample.spe", Ei=100.0, num_of_pixels=100, runNo="XXX000001", phxfile="default", parfile="default"):
    """
    Make ElementContainerMatrix from SPE and phx file.
    @param  path     (String) path to folder including spefile
    @param  spefile  (String) SPE file name (extention is .spe or .SPE)
    @param  Ei       (Float) Incident Energy
    @param  num_of_pixels (Int) the number of pixels in a detector
    @param  runNo    (String) Run Number with instrument codes
    @param  phxfile  (String) PHX file (extention is .phx)
    @param  parfile  (String) PAR file (extention is .par)
    @retval DAT      ElementContainerMatrix
    """
    mu.UtsusemiMessage('Cmm.LoadDataFromSPE("{}","{}",{},{},"{}","{}","{}")'.format(
        path, spefile, Ei, num_of_pixels, runNo, phxfile, parfile), True)
    return BaseCom.LoadDataFromSPE(path, spefile, Ei, num_of_pixels, runNo, phxfile, parfile)

#########################################


def SaveDataToNxspe(dat=DAT, path="./", filename="Sample", params="PSI:0.0,TEMP:300.0,USER:mlfuser"):
    """
    Save ElementContainerMatrix to NXSPE file.
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name without extension of .nxspe
    @param  params   (string) Other information : psi [deg], temperature[K] and user name
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SaveDataToNxspe(<ElementContainerMatrix>,"{}","{}","{}")'.format(
        path, filename, params), True)
    BaseCom.SaveDataToNxspe(dat, path, filename, params)

#########################################


def LoadDataFromNxspe(path="./", filename="Sample.nxspe", num_of_pixels=100):
    """
    Make ElementContainerMatrix from nxspe.
    @param  path     (String) path to folder including spefile
    @param  spefile  (String) SPE file name (extention is .spe or .SPE)
    @param  num_of_pixels (Int) the number of pixels in a detector
    @retval DAT      (ElementContainerMatrix)
    """
    mu.UtsusemiMessage('Cmm.LoadDataFromNxspe("{}","{}",{})'.format(
        path, filename, num_of_pixels), True)
    return BaseCom.LoadDataFromNxspe(path, filename, num_of_pixels)

#########################################


def SavePowderDataToText(dat=DAT, deltaQ=0.0, path="./", filename="Sample.txt"):
    """
    Save ElementContainerMatrix of powder to text file.
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  deltaQ   (double) bin width of Q
    @param  path     (String) path to data file
    @param  filename (String) text file name
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SavePowderDataToText(<ElementContainerMatrix>,{},"{}","{}")'.format(
        deltaQ, path, filename), True)
    BaseCom.SavePowderDataToText(dat, deltaQ, path, filename)

#########################################


def LoadDataFromDump(path="./", filename="Sample.dmp"):
    """
    Load ElementContainerMatrix from dump file made by ECMSerialize class
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval DAT        ElementContainer/Array/Matrix
    """
    mu.UtsusemiMessage(
        'Cmm.LoadDataFromDump("{}","{}")'.format(path, filename), True)
    return BaseCom.LoadDataFromDump(path, filename)


#########################################
def SaveDataToDump(dat=DAT, path="./", filename="Sample.dmp"):
    """
    Serialize ElementContainerMatrix to dump file using ECMSerialize class.
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SaveDataToDump(<ElementContainerMatrix>,"{}","{}")'.format(
        path, filename), True)
    BaseCom.SaveDataToDump(dat, path, filename)

############################################


def SaveDataToManyoBinary(dat=DAT, path="./", filename="Sample.mdb"):
    """
    Save ElementContainerMatrix to manyo binary data file
    This is the binary format for storing ElementContainers using HDF5
    @param  dat [Def:DAT] (ElementContainer, -Array, -Matrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SaveDataToManyoBinary(<ElementContainerMatrix>,"{}","{}")'.format(
        path, filename), True)
    BaseCom.SaveDataToManyoBinary(dat, path, filename)


############################################
def LoadDataFromManyoBinary(path="./", filename="Sample.mdb"):
    """
    Load ElementContainerMatrix from manyo binary data file
    which is the binary format for storing ElementContainers usign HDF5
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval data (ElementContaienr, -Array, -Matrix)
    """
    mu.UtsusemiMessage(
        'Cmm.LoadDataFromManyoBinary("{}","{}")'.format(path, filename), True)
    return BaseCom.LoadDataFromManyoBinary(path, filename)


#########################################
def GetDataFromNeXus(path="./", filename="Sample.nx"):
    """
    Load ElementContainer/Array/Matrix from NeXus file.
    @param  path     (String)   path to data file
    @param  filename (String)  data file name
    @retval DAT      ElementContainer/Array/Matrix
    """
    mu.UtsusemiMessage(
        'Cmm.GetDataFromNeXus("{}","{}")'.format(path, filename), True)
    return BaseCom.GetDataFromNeXus(path, filename)


#########################################
def SaveDataToNeXus(dat=DAT, name="Data", path="./", filename="Sample.nx"):
    """
    Save ElementContainer/Array/Matrix to NeXus file.
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  name      (String)   Title of saved data
    @param  path      (String)   path to data file
    @param  filename  (String)  data file name
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SaveDataToNeXus(<ElementContainerMatrix>,"{}","{}","{}")'.format(
        name, path, filename), True)
    BaseCom.SaveDataToNeXus(dat, name, path, filename)

#########################################


def SaveDataToDAVEgrp(dat=DAT, path="./", filename="Sample.grp", startQ=1.0, endQ=1.0, frcQ=1, binE=0.0):
    """
    Save grouped text (DAVE format) files of Q-sliced powder data
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @param  startQ   (double) minimum limit of Q-range
    @param  endQ     (double) maximum limit of Q-range
    @param  frcQ     (int) number of division
    @param  binE      (double) rebinning width of energy
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SaveDataToDAVEgrp(<ElementContainerMatrix>,"{}","{}",{},{},{},{})'.format(
        path, filename, startQ, endQ, frcQ, binE), True)
    BaseCom.SaveDataToDAVEgrp(dat, path, filename, startQ, endQ, frcQ, binE)

##########################################


def getBeamCurrent(start_day="2009/5/30", start_time="10:00:00", end_day="2009/5/30", end_time="11:00:00", offset_sec=0, isCT8n=True):
    """
    get Beam current from web and calculate summation of the number of shots and protons
    @param start_day  the day of measurement start
    @param start_time the time of measurement start
    @param end_day    the day of measurement end
    @param end_time   the time of measurement end
    @param offset_sec the offset value of the time [seconds]
    @param isCT8n (bool) use CT8Neutron ( corrected by muon target effect ) or not
    @retval ret       the number of protons [TelaProtons]
    """
    mu.UtsusemiMessage('Cmm.getBeamCurrent("{}","{}","{}","{}",{},{})'.format(
        start_day, start_time, end_day, end_time, offset_sec, isCT8n), True)
    return BaseCom.getBeamCurrent(start_day, start_time, end_day, end_time, offset_sec, isCT8n)

#########################################


def AppendRawDataFiles(OutRunNo=990000, outDataPath="", RunNo1=0, RunNo2=0, RunNo3=0, RunNo4=0, RunNo5=0, RunNo6=0):
    """
    append several EventData iles into one

    @param OutRunNo (int) output run number
    @param outDataPath (string) output data path (default is empty)
    @param RunNo1   (int) run number 1
    @param RunNo2   (int) run number 2
    @param RunNo3   (int) run number 3
    @param RunNo4   (int) run number 4
    @param RunNo5   (int) run number 5
    @param RunNo6   (int) run number 6
    @retval None
    """
    BaseCom.AppendRawDataFiles(
        OutRunNo, outDataPath, RunNo1, RunNo2, RunNo3, RunNo4, RunNo5, RunNo6)

#########################################


def VisContMOutputFigure(dat=DAT, param_file="param.xml", output_file="./figure.png", zmin=-1.0, zmax=-1.0):
    """
    Output figure file of sliced data using VisualContM function
    [inamura 120417]
    @param dat [Def:DAT] (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param output_file (string) Path to output file name of figure.
    @param zmin (float) min value of plotting Z-range
    @param zmax (float) max value of plotting Z-range
    @return None
    """
    mu.UtsusemiMessage('Cmm.VisContMOutputFigure(<ElementContainerMatrix>,"{}","{}",{},{})'.format(
        param_file, output_file, zmin, zmax), True)
    BaseCom.VisContMOutputFigure(dat, param_file, output_file, zmin, zmax)

#########################################


def DetectorEffi(dat=DAT, dataFile="SIK-SUS304-3He-PSD.dat"):
    """
    DetectorEfficiency correction
    @param dat [Def:DAT] (ElementContainerMatrix)
    @param dataFile (string)
    @return None
    """
    mu.UtsusemiMessage(
        'Cmm.DetectorEffi(<ElementContainerMatrix>,"{}")'.format(dataFile), True)
    BaseCom.DetectorEffi(dat, dataFile)

############################################


def SaveDataToSrlz(dat=DAT, path="./", filename="Sample.srlz"):
    """
    Serialize ElementContainerMatrix to dump file using Boost Serialize class.
    Data File from this function has the operation system dependency.
    @param  dat      (ElementContainer, -Array, -Matrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval None
    """
    mu.UtsusemiMessage('Cmm.SaveDataToSrlz(<ElementContainerMatrix>,"{}","{}")'.format(
        path, filename), True)
    BaseCom.SaveDataToSrlz(dat, path, filename)

############################################


def LoadDataFromSrlz(path="./", filename="Sample.srlz", DataType="ECM"):
    """
    Serialize ElementContainerMatrix to dump file using Boost Serialize class.
    Data File from this function has the operation system dependency.
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @param  DtaType  (String) "EC" or "ECA" or "ECM"
    @retval DAT        ElementContainer/Array/Matrix
    """
    mu.UtsusemiMessage('Cmm.LoadDataFromSrlz("{}","{}","{}")'.format(
        path, filename, DataType), True)
    return BaseCom.LoadDataFromSrlz(path, filename, DataType)

#########################################


def TofToEnergyTransfer(dat=DAT, Ei=-1.0, L1=18.03, t0shift=0.0, deltaE=0.0, EminRate=0.4, EmaxRate=0.9):
    """
    Function to convert Time-of-Flight to Energy

    @param dat [Def:DAT] (ElementContainerMatrix)
    @param Ei       (double) Incident Energy [meV]: if -1, use Ei value in header
    @param L1       (double) Distance between moderator to sample [m]
    @param t0shift  (double) T0 shift [micro-sec]
    @param deltaE   (double) Delta Energy Transfer [meV]
    @param EminRate (double) range for minus E (<1.0)
    @param EmaxRate (double) range for plus E (<1.0)
    """
    mu.UtsusemiMessage('Cmm.TofToEnergyTransfer(<ElementContainerMatrix>,{},{},{},{},{},{})'.format(
        Ei, L1, t0shift, deltaE, EminRate, EmaxRate), True)
    BaseCom.TofToEnergyTransfer(
        dat, Ei, L1, t0shift, deltaE, EminRate, EmaxRate)

##########################################


def TofToEnergyTransferStrict(dat=DAT, Ei=-1.0, L1=18.03, t0shift=0.0, Emin=-10.0, Emax=10.0, deltaE=0.0):
    """
    Function to convert Time-of-Flight to Energy Transfer

    @param dat [Def:DAT] (ElementContainerMatrix)
    @param Ei       (double) Incident Energy [meV]
    @param L1       (double) Distance between moderator to sample [m]
    @param t0shift  (double) T0 shift [micro-sec]
    @param Emin (double) minimum of E range
    @param Emax (double) maximum of E range
    @param deltaE   (double) Delta Energy Transfer [meV]
    """
    mu.UtsusemiMessage('Cmm.TofToEnergyTransferStrict(<ElementContainerMatrix>,{},{},{},{},{},{})'.format(
        Ei, L1, t0shift, Emin, Emax, deltaE), True)
    BaseCom.TofToEnergyTransferStrict(dat, Ei, L1, t0shift, Emin, Emax, deltaE)

##########################################


def RadialCollimatorCorrect(dat=DAT, vdat=DAT, integRange="All", pixelRange="All"):
    """
    Data Correction for radial collimator on SIK

    @param dat [Def:DAT] (ElementContainerMatrix) Target data
    @param vdat [Def:DAT] (ElementContainerMatrix) Vanadium data
    @param integRange (string) Integrate range "All" or "<start>:<end>"
    @param pixelRange (string) Pixel range to be used "All" or "<start>:<end>"
    @retval None
    """
    mu.UtsusemiMessage('Cmm.RadialCollimatorCorrect(<EleemntContainerMatrix>,<EleemntContainerMatrix>,{}.{}'.format(
        integRange, pixelRange), True)
    BaseCom.RadialCollimatorCorrectSIK(dat, vdat, integRange, pixelRange)

#########################################


def GetOrigPixelFromQXtalParam(XtalParam="XtalParam.xml", Q="1.0,1.0,1.0", Ei=45.56, hw=0.0, runNo=999999, pixNum=100):
    """
    Gets origina pixel position from Q position on caclucated S(Q,E) using XtalParam.xml
    @param XtalParam (string) path to XtalParam.xml
    @param Q         (string) "<Qa>, <Qb>, <Qc>"
    @param Ei        (float)  Incident Energy
    @param hw        (float)  Energy Transfer
    @param runNo     (int)    Run Number used to select suitable DetecterInfo.xml
    @param pixNum    (int)    The number of pixels on one PSD
    @retval posi     (tupple) detId and pixelNo of Detector
    """
    return BaseCom.GetOrigPixelFromQXtalParam(XtalParam, Q, Ei, hw, runNo, pixNum)

#########################################


def GetOrigPixelFromQString(LatticeConst="4.8,8.4,2.9", LatticeAngles="90.0,90.0,90.0", Uvect="0.0,1.0,0.0", Vvect="0.0,0.0,1.0", RotAngle="None", ViewAxis1="1,0,0", ViewAxis2="0,1,0", ViewAxis3="0,0,1", Q="1.0,1.0,1.0", Ei=45.56, hw=0.0, runNo=999999, pixNum=100):
    """
    Gets origina pixel position from Q position on caclucated S(Q,E) using lattice constants, U and V vectores etc.
    @param LatticeConst (string) "<a>,<b>,<c>"
    @param LatticeAngle (string) "<alpha>,<beta>,<gamma>"
    @param Uvect        (string) "<Ua>, <Ub>, <Uc>"
    @param Vvect        (string) "<Va>, <Vb>, <Vc>"
    @param RotAngle     (string) "None" or "<AX>:<deg>[,<AX>:<deg>,..]" <AX>="X","Y","Z"
    @param ViewAxis1    (string) "<Ax1_a*>,<Ax1_b*>,<Ax1_c*>"
    @param ViewAxis2    (string) "<Ax2_a*>,<Ax2_b*>,<Ax2_c*>"
    @param ViewAxis3    (string) "<Ax3_a*>,<Ax3_b*>,<Ax3_c*>"
    @param Q            (string) "<Q1>,<Q2>,<Q3>"
    @param Ei           (float)  Incident Energy [meV]
    @param hw           (float)  Energy Transfer [meV]
    @param runNo        (int)    Run Number used to select suitable DetecterInfo.xml
    @param pixNum       (int)    The number of pixels on one PSD
    @retval posi        (tupple) detId and pixelNo of Detector
    """
    return BaseCom.GetOrigPixelFromQString(LatticeConst, LatticeAngles, Uvect, Vvect, RotAngle, ViewAxis1, ViewAxis2, ViewAxis3, Q, Ei, hw, runNo, pixNum)

#########################################


def CalcYRotAngleFromQXtalParam(XtalParam="XtalParam.xml", Q="1.0,1.0,1.0", Ei=45.56, hw=0.0, runNo=999999, pixNum=100):
    """
    Calc angle around Y-axis to measure given Q-hw position on caclucated S(Q,E) using XtalParam file
    @param XtalParam (string) path to XtalParam.xml
    @param Q         (string) "<Qa>, <Qb>, <Qc>"
    @param Ei        (float)  Incident Energy
    @param hw        (float)  Energy Transfer
    @param runNo     (int)    Run Number used to select suitable DetecterInfo.xml
    @param pixNum    (int)    The number of pixels on one PSD
    @retval None
    """
    BaseCom.CalcYRotAngleFromQXtalParam(XtalParam, Q, Ei, hw, runNo, pixNum)

#########################################


def CalcYRotAngleFromQString(LatticeConst="4.8,8.4,2.9", LatticeAngles="90.0,90.0,90.0", Uvect="0.0,1.0,0.0", Vvect="0.0,0.0,1.0", RotAngle="None", ViewAxis1="1,0,0", ViewAxis2="0,1,0", ViewAxis3="0,0,1", Q="1.0,1.0,1.0", Ei=45.56, hw=0.0, runNo=999999, pixNum=100):
    """
    Calc angle around Y-axis to measure given Q-hw position on caclucated S(Q,E) using lattice constants, U and V vectores etc.
    @param LatticeConst (string) "<a>,<b>,<c>"
    @param LatticeAngle (string) "<alpha>,<beta>,<gamma>"
    @param Uvect        (string) "<Ua>, <Ub>, <Uc>"
    @param Vvect        (string) "<Va>, <Vb>, <Vc>"
    @param RotAngle     (string) "None" or "<AX>:<deg>[,<AX>:<deg>,..]" <AX>="X","Y","Z"
    @param ViewAxis1    (string) "<Ax1_a*>,<Ax1_b*>,<Ax1_c*>"
    @param ViewAxis2    (string) "<Ax2_a*>,<Ax2_b*>,<Ax2_c*>"
    @param ViewAxis3    (string) "<Ax3_a*>,<Ax3_b*>,<Ax3_c*>"
    @param Q            (string) "<Q1>,<Q2>,<Q3>"
    @param Ei           (float)  Incident Energy [meV]
    @param hw           (float)  Energy Transfer [meV]
    @param runNo        (int)    Run Number used to select suitable DetecterInfo.xml
    @param pixNum       (int)    The number of pixels on one PSD
    @retval None
    """
    BaseCom.CalcYRotAngleFromQString(LatticeConst, LatticeAngles, Uvect, Vvect,
                                     RotAngle, ViewAxis1, ViewAxis2, ViewAxis3, Q, Ei, hw, runNo, pixNum)

#########################################


def DymPDF(DAT=DAT, hwSlice="0.2,1.0,12.5", rRange="0.1,40.0", magFFa="0.4220,17.684", magFFb="0.5948,6.005", magFFc="0.0043,-0.609", magFFd="-0.0219", CF=74.84, isXaxisR=True):
    """Calculate Dynamical PDF with magnetic form factor

    @param DAT [Def:DAT] (ElementContainerMatrix)
    @param hwSlice  (string) hw Slicing infor "<bin width>,<min hw>,<max hw>"
    @param rRange   (string) output r range "<r bin width>,<max r>"
    @param magFFa   (string) magnetic form factor "<A1>,<a2>"
    @param magFFb   (string) magnetic form factor "<B1>,<b2>"
    @param magFFc   (string) magnetic form factor "<C1>,<c2>"
    @param magFFd   (string) magnetic form factor "<D>"
    @param CF       (float)  Correction factor
    @param isXaxisR (bool) X-axis of output 2d data, True:R, False:hw
    @retval ECA (ElementContainerArray)
    """
    return BaseCom.DymPDF(DAT, hwSlice, rRange, magFFa, magFFb, magFFc, magFFd, CF, isXaxisR)

#########################################


def WideDymPDF(DAT=DAT, hwSlice="0.2,1.0,12.5", rRange="0.1,40.0", magFFa="0.4220,17.684", magFFb="0.5948,6.005", magFFc="0.0043,-0.609", magFFd="-0.0219", CF=74.84, qParam="6.0,5.0", isXaxisR=True):
    """Wide-Q dynamic magnetic pair density funcation analysis (wideDymPDF)

    @param DAT [Def:DAT] (ElementContainerMatrix)
    @param hwSlice  (string) hw Slicing infor "<bin width>,<min hw>,<max hw>"
    @param rRange   (string) output r range "<r bin width>,<max r>"
    @param magFFa   (string) magnetic form factor "<A1>,<a2>"
    @param magFFb   (string) magnetic form factor "<B1>,<b2>"
    @param magFFc   (string) magnetic form factor "<C1>,<c2>"
    @param magFFd   (string) magnetic form factor "<D>"
    @param CF       (float)  Correction factor
    @param qParam   (string) "Qmin,Qmax2", Qmin for phonon background fitting, Qmax2 for dynamic magnetic PDF analysis
    @param isXaxisR (bool) X-axis of output 2d data, True:R, False:hw
    @retval ECA (ElementContainerArray)
    """
    return BaseCom.WideDymPDF(DAT, hwSlice, rRange, magFFa, magFFb, magFFc, magFFd, CF, qParam, isXaxisR)

#########################################


def DyPDF(DAT=DAT, hwSlice="0.2,1.0,12.5", rRange="0.1,40.0", nav=5, CF=1.0, isXaxisR=True):
    """Calculate Dynamical PDF

    @param DAT [Def:DAT] (ElementContainerMatrix)
    @param hwSlice  (string) hw Slicing infor "<bin width>,<min hw>,<max hw>"
    @param rRange   (string) output r range "<r bin width>,<max r>"
    @param nav      (int) the number of bins for averaging
    @param CF       (float)  Correction factor
    @param isXaxisR (bool) X-axis of output 2d data, True:R, False:hw
    @retval ECA (ElementContainerArray)
    """
    return BaseCom.DyPDF(DAT, hwSlice, rRange, nav, CF, isXaxisR)

#########################################


def DymPDFFromSpinWText(filename="result_SQw.txt", hwSlice="0.2,1.0,12.5", rRange="0.1,40.0", magFFa="0.4220,17.684", magFFb="0.5948,6.005", magFFc="0.0043,-0.609", magFFd="-0.0219", CF=74.84, isXaxisR=True):
    """Calculate Dynamical PDF with magnetic form factor using output file from SpinW

    @param filename (string) path and file name of the output file from SpinW
    @param hwSlice  (string) hw Slicing infor "<bin width>,<min hw>,<max hw>"
    @param rRange   (string) output r range "<r bin width>,<max r>"
    @param magFFa   (string) magnetic form factor "<A1>,<a2>"
    @param magFFb   (string) magnetic form factor "<B1>,<b2>"
    @param magFFc   (string) magnetic form factor "<C1>,<c2>"
    @param magFFd   (string) magnetic form factor "<D>"
    @param CF       (float)  Correction factor
    @param isXaxisR (bool) X-axis of output 2d data, True:R, False:hw
    @retval ECA (ElementContainerArray)
    """
    return BaseCom.DymPDFFromSpinWText(filename, hwSlice, rRange, magFFa, magFFb, magFFc, magFFd, CF, isXaxisR)

#########################################
# Dictionary for entry the name of functions
_functions = {
    "GetNeunetHist": GetNeunetHist,
    "GetMultiEi2": GetMultiEi2,
    "GetMultiEi": GetMultiEi,
    "GetEi": GetEi,
    "NormByBeamCurrent": NormByBeamCurrent,
    "NormByKickers": NormByKickers,
    "DoMask": DoMask,
    "MakeWhiteVanCorr": MakeWhiteVanCorr,
    "WhiteVanCorr": WhiteVanCorr,
    "KiKfCorrection": KiKfCorrection,
    "dHWCorrection": dHWCorrection,
    "VisContOutputFileByPhi": VisContOutputFileByPhi,
    "VisContMOutputFileByPhi": VisContMOutputFileByPhi,
    "BoseFactorCorrection": BoseFactorCorrection,
    "BoseFactCorrEC": BoseFactCorrEC,
    "ToPowder": ToPowder,
    "SolidAngleCorrection": SolidAngleCorrection,
    "GetOrigPixelFromQXtalParam": GetOrigPixelFromQXtalParam,
    "GetOrigPixelFromQString": GetOrigPixelFromQString,
    "CalcYRotAngleFromQXtalParam": CalcYRotAngleFromQXtalParam,
    "CalcYRotAngleFromQString": CalcYRotAngleFromQString,
    "DymPDF": DymPDF,
    "DymPDFFromSpinWText": DymPDFFromSpinWText,
    "DyPDF": DyPDF,
    "WideDymPDF": WideDymPDF,
    "FlexCalc": FlexCalc,
    "SumOfTOF": SumOfTOF,
    "SumOfPixel": SumOfPixel,
    "PickUpRegion": PickUpRegion,
    "GetPH": GetPH,
    "NormByBeam": NormByBeam,
    "getBeamCurrent": getBeamCurrent,
    "SaveDataToSPE": SaveDataToSPE,
    "LoadDataFromSPE": LoadDataFromSPE,
    "SaveDataToNxspe": SaveDataToNxspe,
    "LoadDataFromNxspe": LoadDataFromNxspe,
    "SavePowderDataToText": SavePowderDataToText,
    "SaveDataToDump": SaveDataToDump,
    "LoadDataFromDump": LoadDataFromDump,
    "SaveDataToManyoBinary": SaveDataToManyoBinary,
    "LoadDataFromManyoBinary": LoadDataFromManyoBinary,
    "SaveDataToNeXus": SaveDataToNeXus,
    "GetDataFromNeXus": GetDataFromNeXus,
    "SaveDataToDAVEgrp": SaveDataToDAVEgrp,
    "AppendRawDataFiles": AppendRawDataFiles,
    "VisContMOutputFigure": VisContMOutputFigure,
    "DetectorEffi": DetectorEffi,
    "SaveDataToSrlz": SaveDataToSrlz,
    "LoadDataFromSrlz": LoadDataFromSrlz,
    "TofToEnergyTransfer": TofToEnergyTransfer,
    "TofToEnergyTransferStrict": TofToEnergyTransferStrict,
    "RadialCollimatorCorrect": RadialCollimatorCorrect
}
_functionsOrder = [
    "GetNeunetHist",
    "GetMultiEi2",
    "GetMultiEi",
    "GetEi",
    "NormByBeamCurrent",
    "NormByKickers",
    "SolidAngleCorrection",
    "KiKfCorrection",
    "dHWCorrection",
    "DoMask",
    "DetectorEffi",
    "ToPowder",
    "RadialCollimatorCorrect",
    "VisContOutputFileByPhi",
    "VisContMOutputFileByPhi",
    "BoseFactorCorrection",
    "BoseFactCorrEC",
    "DymPDF",
    "DymPDFFromSpinWText",
    "DyPDF",
    "WideDymPDF",
    "FlexCalc",
    "GetOrigPixelFromQXtalParam",
    "GetOrigPixelFromQString",
    "CalcYRotAngleFromQXtalParam",
    "CalcYRotAngleFromQString",
    "SumOfTOF",
    "SumOfPixel",
    "PickUpRegion",
    "GetPH",
    "NormByBeam",
    "getBeamCurrent",
    "SaveDataToNxspe",
    "LoadDataFromNxspe",
    "SaveDataToSPE",
    "LoadDataFromSPE",
    "SavePowderDataToText",
    "SaveDataToManyoBinary",
    "LoadDataFromManyoBinary",
    "SaveDataToDAVEgrp",
    "SaveDataToDump",
    "LoadDataFromDump",
    "SaveDataToSrlz",
    "LoadDataFromSrlz",
    "SaveDataToNeXus",
    "GetDataFromNeXus",
    "AppendRawDataFiles",
    "VisContMOutputFigure"
    "MakeWhiteVanCorr",
    "WhiteVanCorr",
    "TofToEnergyTransfer",
    "TofToEnergyTransferStrict"
]
