#include "UtsusemiEcsClipper.hh"

//const Double MASKVALUE = 100000000.0;
const Double MASKVALUE = UTSUSEMIMASKVALUE64;

//////////////// Constructor /////////////////////
UtsusemiEcsClipper::
UtsusemiEcsClipper()
{
    _initialize();
}

////////////// Destructor ///////////////////////
UtsusemiEcsClipper::
~UtsusemiEcsClipper()
{
    if (_eca_res != NULL){
        delete _eca_res;
    }
    delete _st;
}
///////////// Constructor with Matrix data //////////////////////
UtsusemiEcsClipper::
UtsusemiEcsClipper( ElementContainerMatrix *ecm, string keyOfX, string keyOfY )
{
    _initialize();
    
    // Size Check
    if(ecm->PutTableSize() > 0){
        // Pick up The first array 
        ElementContainerArray *eca = ecm->PutPointer(0);
        SetInputP( eca );
        SetAxes( keyOfX, keyOfY, false );
        // Take Run No. from Matrix eader.
        HeaderBase* hh = ecm->PutHeaderPointer();
        _runNo = UTSUSEMI_KEY_UNKNOWN;
        if (hh->CheckKey(UTSUSEMI_KEY_HEAD_RUNNUMBER)==1){
            _runNo = hh->PutString(UTSUSEMI_KEY_HEAD_RUNNUMBER);
        }
        // Take Ei from Matrix header, if not exist, this data may be TOF or invalid
        if (hh->CheckKey(UTSUSEMI_KEY_HEAD_EI)==1){
            Double ei = hh->PutDouble(UTSUSEMI_KEY_HEAD_EI);
            UtsusemiMessage( _MessageTag+"Constructor > Ei ="+_st->DoubleToString( ei ) );
            _initData(ei);
        }
        else{
            // Try an array header
            _initData(0.0);
        }
    }
}

///////////// Constructor with Array data //////////////////////
UtsusemiEcsClipper::
UtsusemiEcsClipper( ElementContainerArray *eca, string keyOfX, string keyOfY )
{
    _initialize();

    // Take Run No. from Matrix eader.
    HeaderBase* hh = eca->PutHeaderPointer();
    _runNo = UTSUSEMI_KEY_UNKNOWN;
    if (hh->CheckKey(UTSUSEMI_KEY_HEAD_RUNNUMBER)==1){
        _runNo = hh->PutString(UTSUSEMI_KEY_HEAD_RUNNUMBER);
    }

    //SetInputP( eca );
    SetInput( *eca );
    SetAxes( keyOfX, keyOfY, false );
    _initData(0.0);
}

////////////// Initialize ///////////////////////
void UtsusemiEcsClipper::
_initialize()
{
    _debugMode = false;
    _regions.clear();
    _eca_res = NULL;
    _invalid = true;
    _MessageTag = "UtsusemiEcsClipper::";
    _st = new StringTools();
    
    _givenAxesKeys.clear();
    pair<string,UInt4> tmp;
    tmp.first= "";
    tmp.second= 0;
    _givenAxesKeys.push_back(tmp);
    _givenAxesKeys.push_back(tmp);

}
///////////// Initialize data //////////////////////
void UtsusemiEcsClipper::
_initData( Double ei )
{
    ElementContainerArray* eca = PutInput();
    
    // Initialize Map data
    _invalid = false;
    _anisoBin2d.clear();
    _ints2d.clear();
    _errs2d.clear();
    _isoBin.clear();
    _Angles.clear();
    _keys.clear();
    _units.clear();
    _numMask = 0;
    
    // Check if data size is OK.
    if(eca->PutTableSize() == 0){
        UtsusemiError( _MessageTag+"_initData > Array size is 0." );
        _invalid = true;
        return;
    }
    
    // Judge Data type from the first ElementContainer
    ElementContainer* ec = eca->PutPointer(0);
    
    string xkey = ec->PutXKey();
    _keys.push_back(xkey);
    _keys.push_back(ec->PutYKey());
    _keys.push_back(ec->PutEKey());
    _units.push_back(ec->PutUnit(xkey));
    _units.push_back(ec->PutUnit(ec->PutYKey()));
    _units.push_back(ec->PutUnit(ec->PutEKey()));
    
    //[inamura 130620]-->
    HeaderBase* hh_eca=eca->PutHeaderPointer();
    bool hasEi = false;
    bool hasEnergyAxis = false;
    bool isPowder = false;
    if (hh_eca->CheckKey(UTSUSEMI_KEY_HEAD_EI)==1){
        ei = hh_eca->PutDouble(UTSUSEMI_KEY_HEAD_EI);
        hasEi=true;
    }
    if (xkey.find(UTSUSEMI_KEY_ENERGY)!=-1) hasEnergyAxis=true;
    if (hh_eca->CheckKey(UTSUSEMI_KEY_HEAD_SAMPLETYPE)==1){
        if (hh_eca->PutString(UTSUSEMI_KEY_HEAD_SAMPLETYPE)==UTSUSEMI_KEY_HEAD_SAMPLETYPE_POWDER) isPowder=true;
    }
    if ((hasEnergyAxis)&&(isPowder)&&(ei>0.0)){
        UtsusemiMessage( _MessageTag+"\nThis is Powder data with Energy axis." );
        _MakePowderMap(ei);
        _ei = ei;
    }else{
        // Prepare 2D Data
        _MakeRectMap();
        _ei = 0.0;
    }
    /*
    if (xkey == UTSUSEMI_KEY_ENERGY){
        cout << "\nThis is Energy data."<< endl;
        // Not yet take ei?
        if (ei == 0.0){
            // Take Ei from Array header, if not exist, this data is invalid
            HeaderBase* hh = eca->PutHeaderPointer();
            if (hh->CheckKey(UTSUSEMI_KEY_HEAD_EI)!=1){
                cout << "Error!!, Missing Ei in the header."<< endl;
                //[inamura 130619]--> if no Ei, treat as TOF
                //_invalid = true;
                //return;
                cout << "\nThis is treated as TOF data."<< endl;
                // Prepare 2D Data of TOF
                _MakeRectMap();
                _ei = 0.0;
                //<--[inamura 130619]
            }
            ei = hh->PutDouble(UTSUSEMI_KEY_HEAD_EI);
            cout << "UtsusemiEcsClipper::_initData >> Ei =" << ei << endl; //[inamura 130619]
            if (ei < 0.0){ //[inamura 130619]
                cout << "Error!!, EI value is invalid."<< endl;
                _invalid = true;
                return;                    
            }
        }
        // Calc Q and Prepare 2D Data
        _MakePowderMap(ei);
        _ei = ei;
    }
    else{
        cout << "\nThis is TOF data."<< endl;
        // Prepare 2D Data of TOF
        _MakeRectMap();
        _ei = 0.0;
    }
    */
    //<--[inamura 130620]
}

void UtsusemiEcsClipper::
SetAxes(string keyX, string keyY, bool doInitData ){
    ElementContainerArray* eca = PutInput();
    if (keyX!=""){
        _givenAxesKeys[0].first = "";
        _givenAxesKeys[0].second = 0;
        HeaderBase* hh_eca = eca->PutHeaderPointer();
        // keyX is in ECA header ( type = 5,6,7,8 )
        if (hh_eca->CheckKey(keyX)==1){
            UInt4 k = hh_eca->PutKeyLocation( keyX );
            if (k==4){        // keyX is vector<Int4>
                vector<Int4> tmp = hh_eca->PutInt4Vector(keyX);
                UInt4 eca_size = eca->PutTableSize();
                if (tmp.size()==eca_size){
                    _givenAxesKeys[0].first = keyX;
                    _givenAxesKeys[0].second = 5;
                }else if (tmp.size()==(eca_size+1)){
                    _givenAxesKeys[0].first = keyX;
                    _givenAxesKeys[0].second = 7;
                }
            }else if (k==5){  // keyX is vector<Double>
                vector<Double> tmp = hh_eca->PutDoubleVector(keyX);
                UInt4 eca_size = eca->PutTableSize();
                if (tmp.size()==eca_size){
                    _givenAxesKeys[0].first = keyX;
                    _givenAxesKeys[0].second = 6;
                }else if (tmp.size()==(eca_size+1)){
                    _givenAxesKeys[0].first = keyX;
                    _givenAxesKeys[0].second = 8;
                }
            }
        }else{ // keyX is NOT in ECA header ( type = 1,2,3,4 )
            HeaderBase* hh_ec = eca->PutPointer(0)->PutHeaderPointer();
            if (hh_ec->CheckKey(keyX)==1){
                UInt4 k = hh_ec->PutKeyLocation( keyX );
                if ((k==1)||(k==2)){     // KeyX is Int4 or Double
                    _givenAxesKeys[0].first = keyX;
                    _givenAxesKeys[0].second = k; // 1 or 2
                }else if (k==4){         // KeyX is vector<Int4>
                    vector<Int4> tmp = hh_ec->PutInt4Vector(keyX);
                    if (tmp.size()>=2){
                        _givenAxesKeys[0].first = keyX;
                        _givenAxesKeys[0].second = 3;
                    }
                }else if (k==5){         // KeyX is vector<Double>
                    vector<Double> tmp = hh_ec->PutDoubleVector(keyX);
                    if (tmp.size()>=2){
                        _givenAxesKeys[0].first = keyX;
                        _givenAxesKeys[0].second = 4;
                    }
                }
            }
        }
        if (_givenAxesKeys[0].first=="")
            UtsusemiWarning( _MessageTag+"Given KeyX ("+keyX+") is ignored." );
    }
    if (keyY!=""){
        _givenAxesKeys[1].first = "";
        _givenAxesKeys[1].second = 0;
        HeaderBase* hh_eca = eca->PutHeaderPointer();
        vector<Double> ec0_yvec = eca->PutPointer(0)->PutY();
        UInt4 ec0_yvec_size = ec0_yvec.size();
        // If keyY is in ECA header ( type = 5,6,7,8 )
        if (hh_eca->CheckKey(keyY)==1){
            UInt4 k = hh_eca->PutKeyLocation( keyY );
            if (k==4){         // keyY is vector<Int4>
                vector<Int4> tmp = hh_eca->PutInt4Vector(keyY);
                if (tmp.size()==ec0_yvec_size){
                    _givenAxesKeys[1].first = keyY;
                    _givenAxesKeys[1].second = 5;
                }else if (tmp.size()==(ec0_yvec_size+1)){
                    _givenAxesKeys[1].first = keyY;
                    _givenAxesKeys[1].second = 7;
                }
            }else if (k==5){   // keyY is vector<Double>
                vector<Double> tmp = hh_eca->PutDoubleVector(keyY);
                if (tmp.size()==ec0_yvec_size){
                    _givenAxesKeys[1].first = keyY;
                    _givenAxesKeys[1].second = 6;
                }else if (tmp.size()==(ec0_yvec_size+1)){
                    _givenAxesKeys[1].first = keyY;
                    _givenAxesKeys[1].second = 8;
                }
            }
        }else if (eca->PutPointer(0)->CheckKey(keyY)==1){ // keyY is in EC as vector
            if ((eca->PutPointer(0)->PutYKey())==keyY)
                UtsusemiError( _MessageTag+"keyY("+keyY+") is used as Intensity " );
            else{
                _givenAxesKeys[1].first = keyY;
                _givenAxesKeys[1].second = 0;
            }
        }
        if (_givenAxesKeys[1].first=="")
            UtsusemiWarning( _MessageTag+"Given KeyY ("+keyY+") is ignored." );
    }
    if (doInitData) _initData(_ei);
}
//// Make Q and Energy 2D Map ////
void UtsusemiEcsClipper::
_MakePowderMap(Double ei)
{
    ElementContainerArray* eca = PutInput();
    UInt4 dnum = eca->PutTableSize();

    // Prepare vector for angle prservation, -1.0 is masked value.
    vector<Double> pangles(dnum, -1.0);
    //cout << "Data size : " << dnum << endl;
    // Get angle data from each Header
    for (UInt4 i=0;i<dnum;i++){
        ElementContainer* ec = eca->PutPointer(i);
        HeaderBase* hh = ec->PutHeaderPointer();
        // Check if not masked 
        if (hh->CheckKey(UTSUSEMI_KEY_HEAD_MASKED)!=1 || hh->PutInt4(UTSUSEMI_KEY_HEAD_MASKED) != 1){
            if (hh->CheckKey(UTSUSEMI_KEY_HEAD_PIXELPOLARANGLES)!=1){
                UtsusemiError( _MessageTag+"_MakePowderMap >  Missing PixelPolarAngle in EC header:"+ _st->UInt4ToString(i) );
                _invalid = true;
                return;
            }
            // Store angle(degree)
            pangles[i] = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_PIXELPOLARANGLES)[0];
        }
        else{
            UtsusemiMessage( _MessageTag+"_MakePowderMap > "+_st->UInt4ToString(i)+": Masked" );
        }
    }
    // Search adjacent angles that both are not masked for start and end. 
    UInt4 begin = -1;
    UInt4 end = -1;
    for (UInt4 i=0;i<dnum-1;i++){
        if(pangles[i] >=0.0 && pangles[i+1] >=0.0){
            if(begin == (UInt4)-1){
                begin = i;
            }
            end = i;
        }
    }
    // Inappropriate data?
    if(begin == (UInt4)-1){
        UtsusemiError( _MessageTag+"_MakePowderMap > Inappropriate mask." );
        _invalid = true;
        return;
    }
    // Make radian vector without masked ones
    vector<Double> pols;
    for (UInt4 i=0;i<dnum;i++){
        if (pangles[i] >= 0.0){
            // convert degree to radian
            Double radAngle = pangles[i]*M_PI/180.0;;
            pols.push_back(radAngle);
        }
    }

    // Add a dummy to the angle vector's last
    Double dummy = pangles[end+1]-pangles[end]+pangles[end+1];
    pols.push_back(dummy*M_PI/180.0);

    vector<Double> anglebin;
    vector<UInt4> angleIndex;
    anglebin.clear();
    angleIndex.clear();

    // Half length of the not masked first bin
    Double hfGap = (pangles[begin+1]-pangles[begin])*M_PI/180.0/2.0;

    // Start point (Not masked one)
    Double theta = pols[0] - hfGap;
    anglebin.push_back(theta);

    // Threshold for judging bank gap
    // Consider a bank gap when the gap is lerger than 1.5 times than prior gap.
    Double gapTh = hfGap*3.0;

    // Calc scattering angles of both side
    for(UInt4 i=0;i<pols.size()-1;i++){
        // Calc a difference from the prior angle
        Double gap = pols[i+1]-pols[i];
        
        // In case of bank gap or masked data
        if (gap >= gapTh){
            // Calc a left bin from prior data's half angle
            theta = pols[i] + hfGap;
            anglebin.push_back(theta);
            angleIndex.push_back(i);
            // Also write a right bin
            theta = pols[i+1] - hfGap;
            anglebin.push_back(theta);
            angleIndex.push_back(-1);
            //Renew threthold for bank gap
            gapTh = hfGap*3.0;  
        }
        // Regular case 
        else{
            hfGap = gap/2.0;
            theta = pols[i] + hfGap;
            anglebin.push_back(theta);
            angleIndex.push_back(i);
        }
    }

    // Preserve original angle data in header
    for ( UInt4 i = 0; i < angleIndex.size(); i++){
        if (angleIndex[i] == (UInt4)-1){
            _Angles.push_back(-1.0);
        }
        else{
            // Preserve each angle(degree) in header for making result EC7S header
            _Angles.push_back(pangles[angleIndex[i]]);
        }
    }

    // Get the first ebin that is common for every ElementContainer
    ElementContainer* ec0 = eca->PutPointer(0);
    _isoBin = ec0->PutX();
    // Number of int 
    UInt4 ndat = _isoBin.size()-1;
    
    UInt4 numA = anglebin.size();
    // Reserve 2D Area for parallel processing
    _anisoBin2d.resize(numA);
    _ints2d.resize(numA-1);
    _errs2d.resize(numA-1);
    for (UInt4 i=0;i<numA-1;i++){
        _anisoBin2d[i].resize(ndat+1);
        _ints2d[i].resize(ndat);
        _errs2d[i].resize(ndat);
    }
    _anisoBin2d[numA-1].resize(ndat+1);

    UtsusemiUnitConverter ucc;
    Double ki2 = ucc.EtoK2( ei );
    Double ki = sqrt(ki2);

    // For all Q bin 
    //HistogramBinToPoint HTP;
    for (UInt4 i=0;i<numA-1;i++){
        // Bank data or masked data?
        if (angleIndex[i] == (UInt4)-1){
            // Put mask data
            for (UInt4 j=0;j<ndat;j++){
                _ints2d[i][j] = MASKVALUE;
                _errs2d[i][j] = 0.0;
            }
        }else{
            // Store Normal data
            ElementContainer* ec = eca->PutPointer(angleIndex[i]);
            // Convert Intensity to per unit
            //HTP.SetTarget(ec);
            //HTP.Convert();
            //vector<Double> yint = HTP.PutY();
            //vector<Double> err = HTP.PutE();
            vector<Double> yint = ec->PutY();
            vector<Double> err = ec->PutE();
            // For each data point
            for (UInt4 j=0;j<ndat;j++){
                if (err[j]<0.0) yint[j] = MASKVALUE;//[inamura 170131]
                _ints2d[i][j] = yint[j];
                _errs2d[i][j] = err[j];
            }
        }
    }
    
    // Calc Q for each bin
    for (UInt4 j=0;j < ndat+1;j++){
        Double kf = sqrt(ucc.EtoK2( ei - _isoBin[j] ));
        for (UInt4 i=0;i < anglebin.size();i++){
            Double q = sqrt( ki2 + kf*kf - 2.0*ki*kf*cos(anglebin[i]));
            _anisoBin2d[i][j] = q;
        
        }
    }
    return;
}

//// Make TOF 2D Map ////
void UtsusemiEcsClipper::
_MakeRectMap()
{
    ElementContainerArray* eca = PutInput();
    UInt4 dnum = eca->PutTableSize();

    //UtsusemiMessage( _MessageTag+"_MakeRectMap > Data size : "+ _st->UInt4ToString(dnum) );

    // Make 2D map data

    // Make Xbin
    vector<Double> xbin;
    pair<string,UInt4> keys = _givenAxesKeys[0];
    vector<UInt4> ECindex; // index of EC in ECA
    
    if (keys.first!=""){
        vector<Double> tmp;
        if (keys.second==1){
            tmp.resize(dnum);
            for (UInt4 i=0; i<dnum; i++)
                tmp[i] = double(eca->PutPointer(i)->PutHeaderPointer()->PutInt4(keys.first));
        }
        else if (keys.second==2){
            tmp.resize(dnum);
            for (UInt4 i=0; i<dnum; i++)
                tmp[i] = eca->PutPointer(i)->PutHeaderPointer()->PutDouble(keys.first);
        }else if (keys.second==3){
            tmp.resize(dnum);
            for (UInt4 i=0; i<dnum; i++){
                vector<Int4> tmp2 = eca->PutPointer(i)->PutHeaderPointer()->PutInt4Vector(keys.first);
                tmp[i] = (double)(tmp2[0]+tmp2[1])/2.0;
            }
        }else if (keys.second==4){
            tmp.resize(dnum);
            for (UInt4 i=0; i<dnum; i++){
                vector<Double> tmp2 = eca->PutPointer(i)->PutHeaderPointer()->PutDoubleVector(keys.first);
                tmp[i] = (tmp2[0]+tmp2[1])/2.0;
            }
        }else if (keys.second==5){
            tmp.resize(dnum);
            vector<Int4> tmp2 = eca->PutHeaderPointer()->PutInt4Vector(keys.first);
            for (UInt4 i=0; i<dnum; i++)
                tmp[i] = (double)(tmp2[i]);
        }else if (keys.second==6){
            tmp = eca->PutHeaderPointer()->PutDoubleVector(keys.first);
        }else if (keys.second==7){
            vector<Int4> tmp2 =  eca->PutHeaderPointer()->PutInt4Vector(keys.first);
            for (UInt4 i=0; i<tmp2.size(); i++)
                xbin.push_back( (Double)(tmp2[i]) );
        }else if (keys.second==8){
            xbin = eca->PutHeaderPointer()->PutDoubleVector(keys.first);
        }

        // sort x-axis values and make index for EC in ECA
        if (!tmp.empty()){
            ECindex.resize( tmp.size(), 0 );
            vector<Double> tmp_org( tmp );
            sort(tmp.begin(), tmp.end());
            for (UInt4 i=0; i<tmp.size(); i++)
                for (UInt4 j=0; j<tmp_org.size(); j++)
                    if (tmp[i]==tmp_org[j]) ECindex[i]=j;
        }
        
        if (tmp.empty()){
        }else{
            xbin.push_back(tmp[0]-(tmp[1]-tmp[0])/2.0);
            for (UInt4 i=0; i<(dnum-1); i++)
                xbin.push_back( tmp[i] + (tmp[i+1]-tmp[i])/2.0 );
            xbin.push_back(tmp.back()+(tmp.back()-tmp[tmp.size()-2])/2.0 );
        }
    }

    // Make Ybin
    vector<Double> eca_header_ybin;

    // If Ybin is in ECA Header
    if ((_givenAxesKeys[1].first!="")&&(_givenAxesKeys[1].second!=0)){
        vector<Double> tmp;
        if ((_givenAxesKeys[1].second==5)||(_givenAxesKeys[1].second==7)){     // Ybin is vector<Int4>
            vector<Int4> tmp_int = eca->PutHeaderPointer()->PutInt4Vector(_givenAxesKeys[1].first);
            tmp.resize( tmp_int.size() );
            for (UInt4 i=0; i<tmp_int.size(); i++)
                tmp[i] = (Double)tmp_int[i];

        }
        else if ((_givenAxesKeys[1].second==6)||(_givenAxesKeys[1].second==8)){ // Ybin is vector<Double>
            tmp = eca->PutHeaderPointer()->PutDoubleVector(_givenAxesKeys[1].first);
        }
        // Check size of Ybin, size of Ybin must be ( size of Intensity vect +1 )
        if ((_givenAxesKeys[1].second==5)||(_givenAxesKeys[1].second==6)){ // tmp.size()=ec0.PutY().size()
            eca_header_ybin.resize( tmp.size()+1 );
            eca_header_ybin[0] = tmp[0] - (tmp[1]-tmp[0])/2.0;
            for (UInt4 i=0; i<(tmp.size()-1); i++)
                eca_header_ybin[i+1] = (tmp[i+1]+tmp[i])/2.0;
            eca_header_ybin[ eca_header_ybin.size()-1 ] = tmp.back() + (tmp.back()-tmp[ tmp.size()-2 ])/2.0;
        }else{
            eca_header_ybin.resize( tmp.size() );
            copy( tmp.begin(), tmp.end(), eca_header_ybin.begin() );
        }
    }

    vector<Double> xrange;
    for (UInt4 i=0;i<dnum;i++){
        //cout <<"Index : "<< i <<endl;
        UInt4 ind = i;
        if (!ECindex.empty()) ind = ECindex[i];
        ElementContainer* ec = eca->PutPointer(ind);
        HeaderBase* hh = ec->PutHeaderPointer();

        if (!(eca_header_ybin.empty())){
            _anisoBin2d.push_back( eca_header_ybin );
        }else{
            vector<Double> ybin;
            if (_givenAxesKeys[1].first==""){
                vector<Double> ec_bin = ec->PutX();
                ybin.resize( ec_bin.size() );
                copy( ec_bin.begin(), ec_bin.end(), ybin.begin() );
            }else{
                vector<Double> ec_vec=ec->Put( _givenAxesKeys[1].first );
                ybin.resize( ec_vec.size() );
                copy( ec_vec.begin(), ec_vec.end(), ybin.begin() );
            }

            UInt4 ybin_size = ybin.size();
            UInt4 int_size  = ec->PutY().size();
            if (ybin_size==(int_size+1)){
                _anisoBin2d.push_back(ybin);
            }else if (ybin_size==int_size){
                vector<Double> tmp( int_size+1 );
                tmp[0] = ybin[0] - (ybin[1]-ybin[0])/2.0;
                for (UInt4 j=0;j<(ybin_size-1);j++)
                    tmp[j+1] = (ybin[j+1]+ybin[j])/2.0;
                tmp[ tmp.size()-1 ] = ybin[ ybin_size-1 ] + (ybin[ybin_size-1]-ybin[ybin_size-2])/2.0;
                _anisoBin2d.push_back(tmp);
            }else{
                UtsusemiWarning( _MessageTag+" Y axis info is invalid. Use index." );
                vector<Double> tmp(ec->PutY().size()+1);
                for (UInt4 j=0; j<tmp.size(); j++)
                    tmp[j] = (Double)j;
                _anisoBin2d.push_back(tmp);
            }
        }
        // Check if not masked 
        if (hh->CheckKey(UTSUSEMI_KEY_HEAD_MASKED)!=1 || hh->PutInt4(UTSUSEMI_KEY_HEAD_MASKED) != 1){
            vector<Double> yint = ec->PutY();
            vector<Double> errs = ec->PutE();

            for (UInt4 j=0; j<yint.size(); j++)                   //[inamura 170131]
                if (errs[j]<0.0) yint[j] = MASKVALUE;
            _ints2d.push_back(yint);
            _errs2d.push_back(errs);
            // Preserve PolarAngle in header for reconstructing Header
            if (hh->CheckKey(UTSUSEMI_KEY_HEAD_PIXELPOLARANGLES)==1){
                Double angle = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_PIXELPOLARANGLES)[0];
                _Angles.push_back( angle);
                //cout <<"PixelPolarAngle: "<< angle <<endl;
            }
            else{
                _Angles.push_back( -1.0);
            }
        }
        else{
            //cout <<"Masked Data: "<< i << endl;
            // Set Mask value to ints 
            _ints2d.push_back(vector<Double>(_anisoBin2d[i].size()-1, MASKVALUE));
            _errs2d.push_back(vector<Double>(_anisoBin2d[i].size()-1, -1.0));
            _Angles.push_back( -1.0);
        }
        if (xbin.empty()){
            if (hh->CheckKey(UTSUSEMI_KEY_HEAD_XRANGE)==1){
                xrange = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_XRANGE);
                //_isoBin.push_back(xrange[0]-0.5);
                _isoBin.push_back(xrange[0]);
            }
            else{
                //Double xx = (Double)i-0.5;
                Double xx = (Double)i;
                //cout <<"X bin: "<< xx << endl;
                _isoBin.push_back(xx);
            }
        }
    }
    if (xbin.empty()){
        if (xrange.size() == 2){
            //_isoBin.push_back(xrange[1]-0.5);
            _isoBin.push_back(xrange[1]);
        }
        else{
            //_isoBin.push_back((Double)dnum-0.5);
            _isoBin.push_back((Double)dnum);
        }
    }else{
        _isoBin.resize( xbin.size() );
        copy( xbin.begin(), xbin.end(), _isoBin.begin() );
    }
}

//////////////// Set mask value to the 2D Map //////////////////
void UtsusemiEcsClipper::
SetMask( Double x0, Double y0, Double x1, Double y1 )
{
    // TOF data ?
    if (_ei == 0.0){
        x0 = x0 - 0.5;
        for (UInt4 i=0; i < _isoBin.size()-1; i++){
            // Exceed y range end?
            if (x1 < _isoBin[i]){
                break;
            }
            if (x0 <= _isoBin[i+1]){
                for (UInt4 j=0; j < _anisoBin2d[0].size()-1; j++){
                    // Exceed y range end?
                    if (y1 < _anisoBin2d[i][j]){
                        break;
                    }
                    // In masked region?
                    if (y0 <=_anisoBin2d[i][j+1]){
                        // Set mask value
                        _ints2d[i][j] = MASKVALUE;
                        _errs2d[i][j] = 0.0;
                    }
                }
            }
        }

    }
    else{
        // search y index
        for (UInt4 i=0; i < _isoBin.size()-1; i++){
            // Exceed y range end?
            if (y1 < _isoBin[i]){
                break;
            }
            if (y0 <= _isoBin[i+1]){
                for (UInt4 j=0; j < _anisoBin2d.size()-1; j++){
                    // Exceed y range end?
                    if (x1 < _anisoBin2d[j][i]){
                        break;
                    }
                    // In masked region?
                    if (x0 <=_anisoBin2d[j+1][i]){
                        // Set mask value
                        _ints2d[j][i] = MASKVALUE;
                        _errs2d[j][i] = 0.0;
                    }
                }
            }
        }
    }
    // Increment mask resions number
    _numMask++;

    UtsusemiMessage( _MessageTag+"SetMask > _numMask= "+_st->UInt4ToString(_numMask) );
}

///////// Clear Mask regions /////////////////
void UtsusemiEcsClipper::
ClearMask( )
{
    // Remake 2D map data to erase mask regions
    _initData(_ei); 
}

//////////////////////////////////////////////////////////
void UtsusemiEcsClipper::
SetClipRegion( Double x0, Double y0, Double x1, Double y1 )
{
    if( x1 < x0 || y1 < y0){
        UtsusemiError( _MessageTag+"SetClipRegion > Range error!!" );
    }

    if (_ei > 0.0){
        if (y1 < _isoBin[0] || y0 > _isoBin[_isoBin.size()-1]){
            UtsusemiError( _MessageTag+"SetClipRegion > Out of Energy range!!" );
            return;
        }
        if (y0 < _isoBin[0]){
            y0 = _isoBin[0];
        }
        if (y1 > _isoBin[_isoBin.size()-1]){
            y1 = _isoBin[_isoBin.size()-1];
        }
    }
    else{
        if (x1 < _isoBin[0] || x0 > _isoBin[_isoBin.size()-1]){
            UtsusemiError( _MessageTag+"SetClipRegion > Out of index range!!" );
            return;
        }
        if (x0 < _isoBin[0]){
            x0 = _isoBin[0];
        }
        if (x1 > _isoBin[_isoBin.size()-1]){
            x1 = _isoBin[_isoBin.size()-1];
        }
    }

    vector<Double> region;
    region.push_back(x0);
    region.push_back(y0);
    region.push_back(x1);
    region.push_back(y1);
    _regions.push_back(region);

    //cout << "Set Clip reagion " << _regions.size() << endl;

}

//////////////////////////////////////////////////////////
void UtsusemiEcsClipper::
ClearClipRegion( )
{
    _regions.clear();
}

///////////////Make TOF bin or E Bin data ///////////////////
ElementContainerArray UtsusemiEcsClipper::
ExecInteg(bool average, int axis, Double width, Double bin)
{
    if (_eca_res != NULL){
        delete _eca_res;
    }
    // Create an ElementContainerArray for contain results
    HeaderBase hh_eca_res;
    _eca_res = new ElementContainerArray(hh_eca_res);

    if (_invalid){
        UtsusemiError( _MessageTag+"ExecInteg > Invalid data!!" );
        return *_eca_res;
    }

    vector< vector<Double> >::iterator it = _regions.begin();
    while (it != _regions.end()) {
        vector<Double> region = *it;
        // Prepare EC for result
        ElementContainer ec_res;
        if (axis==2){
            _MakeRectDiagCutData(region, bin, width, average, ec_res);
        }else{
            // Energy Data?
            if (_ei > 0.0){
                if (axis ==0){
                    _MakeEbinCutData(region, average, ec_res);
                }else{
                    _MakeQbinCutData(region, average, ec_res);
                }
            }
            else{
                if (axis ==0){
                    _MakeRectYbinCutData(region, average, ec_res);
                }else{
                    _MakeRectXbinCutData(region, average, ec_res);
                }
            }
        }
        _eca_res->Add( ec_res);
        it++;
    }
    return *_eca_res;
}

////////// Rebinning(source bins, source ints, source errs, Target bins, Result Int, Result Err)//////////
// Sizes of res_int and res_err must be bins.size() -1, and cleard by 0.0. /// 
void UtsusemiEcsClipper::
_Rebin(UInt4 index,    vector<Double>& res_bins, vector<Double>& res_ints,vector<Double>& res_ers)
{
    // Prepare vector to store weight of each point
    vector<Double> weights(res_bins.size()-1, 0.0);
    // Search the first index of source bin
    UInt4 i=0;
    while (_anisoBin2d[i][index] < res_bins[0]){
        i++;
    }
    // Search the first index of result bin
    UInt4 j=0;
    while (res_bins[j+1] < _anisoBin2d[0][index]){
        j++;
    }

    Double bin, sint, serr, weight;
    // Repeat all source data
    while (i < _ints2d.size()){
        // Break if reached to end of the target data 
        if (j >= res_bins.size()-1){
            break;
        }
        // if not masked value
        if (_ints2d[i][index] < MASKVALUE){
            // Bin width of current target bin
            bin = res_bins[j+1]-res_bins[j];
            sint = _ints2d[i][index];
            serr = _errs2d[i][index];
            //cout << i << " : " << res_bins[j] << " : " << sint << endl;
            // In case of larger source bin
            if (_anisoBin2d[i+1][index] > res_bins[j+1]){
                weight = res_bins[j+1]-_anisoBin2d[i][index];
                res_ints[j] += sint*weight;
                res_ers[j] += weight*weight*serr*serr;
                weights[j] += weight;
                // Until result bin reaches to source bin
                while (true){
                    j += 1;
                    // End if reach the last point
                    if (j >= res_ints.size()){
                        break;
                    }
                    bin = res_bins[j+1]-res_bins[j];
                    // Not yet reach to prior bin?
                    if (res_bins[j+1] < _anisoBin2d[i+1][index]){
                        weight = res_bins[j+1]-res_bins[j];
                        res_ints[j] += sint*weight;
                        res_ers[j] += weight*weight*serr*serr;
                        weights[j] += weight;
                    }
                    else{
                        weight = _anisoBin2d[i+1][index]-res_bins[j];
                        break;
                    }
                }
            }
            // In case of larger result bin
            else if(_anisoBin2d[i][index] < res_bins[j]){
                // proportional weight
                weight = _anisoBin2d[i+1][index] - res_bins[j];
            }
            else{
                // weight is width of source bin
                weight = _anisoBin2d[i+1][index] - _anisoBin2d[i][index];
            }
            // End if reach the last point
            if (j >= res_bins.size()-1){
                break;
            }
            res_ints[j] += sint*weight;
            res_ers[j] += weight*weight*serr*serr;
            weights[j] += weight;
        }
        // Advance source bin 
        i += 1;
        // Advance result bin if source bin exceed 
        if (_anisoBin2d[i][index] > res_bins[j+1]){
           j+= 1;
        }
    }
    // Devide integrated values from weight
    for (UInt4 i = 0; i < res_ints.size(); i++){
        // Exist integrated values? 
        if (weights[i] > 0.0){
            res_ints[i] = res_ints[i]/weights[i];
            res_ers[i] = sqrt(res_ers[i])/weights[i];
        }
        else{
            //cout << "Mask: " << i << endl;
            res_ints[i] = MASKVALUE;
            res_ers[i] = 0.0;
        }
    }
}

////////// Make TOF bin data //////////
void UtsusemiEcsClipper::
_MakeRectYbinCutData(vector<Double> region, bool average, ElementContainer& ec_res)
{
    Double x0o = region[0];
    Double y0 = region[1];
    Double x1o = region[2];
    Double y1 = region[3];
    
       // Calc center index of region
    //UInt4 cIndex = int((x1 - x0)/2.0+x0+0.5);
    
    UInt4 cIndex = 0;
    bool isSearchIndex = true;
    Double cx = (x1o-x0o)/2.0+x0o;
    Double x0 = 0.0;
    bool isSearchX0 = true;
    Double x1 = 0.0;
    bool isSearchX1 = true;
    for (UInt4 i=0; i<(_isoBin.size()-1); i++){
        if ( (isSearchIndex)&&(cx>=_isoBin[i])&&(cx<_isoBin[i+1]) ){
            cIndex = i;
            isSearchIndex = false;
        }
        if ( (isSearchX0)&&(x0o>=_isoBin[i])&&(x0o<_isoBin[i+1]) ){
            x0 = (double)i;
            isSearchX0 = false;
        }
        //if ( (isSearchX1)&&(x1o>=_isoBin[i])&&(x1o<_isoBin[i+1]) ){  ##[inamura 150514]
        if ( (isSearchX1)&&(x1o>_isoBin[i])&&(x1o<=_isoBin[i+1]) ){
            x1 = (double)i;
            isSearchX1 = false;
        }
    }
    if ((isSearchIndex)||(isSearchX0)||(isSearchX1)){
        UtsusemiError( _MessageTag+"_MakeRectYbinCutData > Index is out" );
        return;
    }
    
    vector<Double> org_x_vec = _anisoBin2d[cIndex];
    vector<Double> x_vec;
    x_vec.clear();

    // Make new vector of tof, trim from first tof
    UInt4 start = 0;
    // [KCS 2013/05/22 Bug Fix] -->
    //UInt4 end = 0;
    UInt4 end = org_x_vec.size()-1;
    //<-- [KCS 2013/05/22 Bug Fix]

    for (UInt4 i = 0; i < org_x_vec.size(); i++){
        // Exceed tof range?
        if (org_x_vec[i] > y1){
            end = i - 1;
            break;
        }
        // Check if between tof range
        if (org_x_vec[i] >= y0){
            // First exceed ?
            if (x_vec.size() == 0){
                start = i;
            }
            x_vec.push_back(org_x_vec[i]);
        }
    }

    // Check if exist at least 2 data
    if (x_vec.size() <= 2){
        return;
    }
    int datNum = x_vec.size() - 1;
    // Create Intensity and Error Vector
    vector<Double> y_vec(datNum, 0.0);
    vector<Double> e_vec(datNum, 0.0);
    // Vector for counting integration
    vector<Int4> n_vec(datNum, 0);

    // While in specified range
    for (UInt4 index = (UInt4)x0; index <= (UInt4)x1; index++){
        // Add int and Error to result vectors, except both end points 
        for (UInt4 j=start; j<end;j++){
            Double sint = _ints2d[index][j];
            Double serr = _errs2d[index][j];

            // Check if not in masked region
            if (sint < MASKVALUE){
                // Add to result vector
                y_vec[j-start] += sint;
                e_vec[j-start] += serr * serr ;
                n_vec[j-start] += 1;
            }
        }
    }
    // Calc average?
    for(UInt4 i=0; i < y_vec.size(); i++){
        // Exist data?
        if (n_vec[i] > 0){
            // Average?
            if (average){
                y_vec[i] = y_vec[i]/(Double)n_vec[i];
                e_vec[i] = sqrt(e_vec[i])/(Double)n_vec[i];
            }
            else{
                e_vec[i] = sqrt(e_vec[i]);
            }
        }
        else{
            y_vec[i] = MASKVALUE;
            e_vec[i] = 0.0;
        }
        //cout << i << " : "<< y_vec[i] <<endl;
    }


    // Add xbin, int, error vectors
    string x_key = _keys[0];
    if (_givenAxesKeys[1].first!="") x_key = _givenAxesKeys[1].first;
    ec_res.Add(x_key, x_vec, _units[0]);
    ec_res.Add(_keys[1], y_vec, _units[1]);
    ec_res.Add(_keys[2], e_vec, _units[2]);
    ec_res.SetKeys( x_key, _keys[1], _keys[2] );
    
    // Make header and set label
    _makeHeader(region[0], region[2], average, ec_res, false);
    // Set the center angle to the header
    _AddAngle(ec_res.PutHeaderPointer(), region[0], region[2]);
}

////////// Make Index Bin data //////////
void UtsusemiEcsClipper::
_MakeRectXbinCutData(vector<Double> region, bool average, ElementContainer& ec_res)
{
    //Double x0 = region[0]-0.5;
    Double x0 = region[0];
    Double y0 = region[1];
    //Double x1 = region[2]+0.5;
    Double x1 = region[2];
    Double y1 = region[3];

    //cout << "x0: " << x0 << "  y0: " << y0 << "  x1: " << x1 << "  y1: " << y1 << endl;

    vector <Double> x_vec;

    // Make Index bin
    for (UInt4 i = 0; i < _isoBin.size(); i++){
        // Has the begin of region reached Index bin's first point? 
        if (x1 < _isoBin[i]){
            break;
        }
        // Has the end of region exceeded TOF bin's end point? 
        if (x0 <= _isoBin[i]){
            x_vec.push_back(_isoBin[i]);
        }
    }
    
    // Out og range?
    if (x_vec.size() < 2){
        return;
    }
    UInt4 dnum = x_vec.size()-1;
    //cout << "dnum : " << dnum << endl;
    // Prepare an integration Counter
    vector<UInt4> n_vec(dnum,0);
    vector<Double> y_vec(dnum,0.0);
    vector<Double> e_vec(dnum,0.0);

    UInt4 index = 0;
    for (UInt4 i = 0; i < _isoBin.size()-1; i++){
        if (x0 <= _isoBin[i]){
            vector<Double> tofs = _anisoBin2d[i];
            vector<Double> yy =   _ints2d[i];
            vector<Double> ee =   _errs2d[i];
            for (UInt4 j = 0; j < tofs.size()-1; j++){
                if (tofs[j] >= y0){
                    if (yy[j] < MASKVALUE){
                        y_vec[index] += yy[j];
                        e_vec[index] += ee[j]*ee[j];
                        n_vec[index] += 1;
                    }
                }
                if (tofs[j+1] > y1){
                    // Check if range is withen a range
                    if (tofs[j] < y0){
                        if (yy[j] < MASKVALUE){
                            y_vec[index] += yy[j];
                            e_vec[index] += ee[j]*ee[j];
                            n_vec[index] += 1;
                        }
                    }
                    break;
                }
            }
            index++;
            if (index >= dnum){
                break;
            }
        }
        
    }

    // Calc average?
    for(UInt4 i=0; i < y_vec.size(); i++){
        if (n_vec[i] > 0){
            if (average){
                y_vec[i] = y_vec[i]/(Double)n_vec[i];
                e_vec[i] = sqrt(e_vec[i])/(Double)n_vec[i];
            }
            else{
                e_vec[i] = sqrt(e_vec[i]);
            }
        }
        else{
            y_vec[i] = MASKVALUE;
            e_vec[i] = 0.0;
        }
    }


    // Add xbin, int, error vectors
    //
    string x_key = "Xvalue";
    if (_givenAxesKeys[0].first!="") x_key = _givenAxesKeys[0].first;
    ec_res.Add(x_key, x_vec, _units[1]);
    ec_res.Add(_keys[1], y_vec, _units[0]);
    ec_res.Add(_keys[2], e_vec, _units[2]);
    ec_res.SetKeys( x_key, _keys[1], _keys[2] );
    
    // Make header and set label
    _makeHeader(region[1], region[3], average, ec_res, true);

}
////////// Make RectFreeCut data ////////
void UtsusemiEcsClipper::
_MakeRectDiagCutData(vector<Double> region, Double bin, Double width, bool average, ElementContainer& ec_res)
{
    // region = [start_x, start_y, end_x, end_y]
    // calc size of cut vector
    Double xx = region[2] - region[0];
    Double yy = region[3] - region[1];
    Double vlength = sqrt(xx * xx+ yy * yy);
    //cout << "vlength : " <<  vlength << endl;
    // Make unit vector on cut line
    Double ux = xx / vlength;
    Double uy = yy / vlength;

    Double width2 = width / 2.0;
    
    // Make  bin on cut line
    vector<Double> x_vec;
    Double vbin = 0.0;
    while ( vbin < vlength)
    {
        x_vec.push_back(vbin);
        vbin += bin;
    }
    // add the last point
    x_vec.push_back(vbin);

    UInt4 numBin = x_vec.size();
    //cout << "numBin : " <<  numBin << endl;
    // prepare 
    vector<int> counter = vector<int>(numBin, 0);
    vector<Double> y_vec = vector<Double>(numBin, 0.0);
    vector<Double> e_vec = vector<Double>(numBin, 0.0);

    UInt4 index=0;
    for (UInt4 ix=0; ix<(_isoBin.size()-1); ix++){
        for (UInt4 iy=0; iy<(_anisoBin2d[ix].size()-1); iy++){
            Double xdat = (_isoBin[ix]+_isoBin[ix+1])/2.0;
            Double ydat = (_anisoBin2d[ix][iy]+_anisoBin2d[ix][iy+1])/2.0;
            Double intd = _ints2d[ix][iy];
            Double errd = _errs2d[ix][iy];
            
            // shift the origine
            Double xp = xdat - region[0];
            Double yp = ydat - region[1];
            // Calc distance from the line
            Double distance = fabs(xx * yp - yy * xp) / vlength ;
            // Check if in width?
            if (distance > width2){
                continue;
            }
            // Calc Dot 
            Double dot = xp  * ux + yp * uy;
            if (dot >= 0.0){
                index = (UInt4)((dot + bin/2.0) / bin);
                if ( index >= numBin){
                    continue;
                }
            }
            else if (fabs(dot) < bin/2.0){
                index = 0;
            }
            else{
                continue;
            }
            
            y_vec[index] += intd;
            counter[index] += 1;
            e_vec[index] += errd*errd;
        }
    }
    
    // Calc average?
    for(UInt4 i=0; i < y_vec.size(); i++){
        if (counter[i] > 0){
            if (average){
                y_vec[i] = y_vec[i]/(Double)counter[i];
                e_vec[i] = sqrt(e_vec[i])/(Double)counter[i];
            }
            else{
                e_vec[i] = sqrt(e_vec[i]);
            }
        }
        else{
            y_vec[i] = MASKVALUE;
            e_vec[i] = -1.0;
        }
    }
    
    // Add xbin, int, error vectors
    string x_key = "Xval";
    ec_res.Add(x_key, x_vec, _units[1]);
    ec_res.Add(_keys[1], y_vec, _units[0]);
    ec_res.Add(_keys[2], e_vec, _units[2]);
    ec_res.SetKeys( x_key, _keys[1], _keys[2] );
    // Make header and set label
    HeaderBase* hh_ec = ec_res.PutHeaderPointer();
    char label[100];
    if (average){
        sprintf(label,  "Average of (%.3f,%.3f) to (%.3f,%.3f)", region[0],region[1],region[0],region[1]);
    }
    else{
        sprintf(label, "Integrated (%.3f,%.3f) to (%.3f,%.3f)", region[0],region[1],region[0],region[1]);
    }
    hh_ec->Add(UTSUSEMI_KEY_HEAD_LABEL,string(label));
    
    // Add runno 
    hh_ec->Add(UTSUSEMI_KEY_HEAD_RUNNUMBER, _runNo);
}


////////// Make Q Bin data //////////
void UtsusemiEcsClipper::
_MakeQbinCutData(vector<Double> region, bool average, ElementContainer& ec_res)
{
    // Calc the center of Energy range
    Double enc = (region[3] - region[1])/2.0 + region[1]; 

    // Search start, center, and end points
    UInt4 m1 = _isoBin.size()-1;;
    UInt4 m0 = -1;
    UInt4 mp = -1;
    for (UInt4 i = 0; i < _isoBin.size(); i++){
        // Exceed the first point?
        if (_isoBin[i] > region[1]){
            // First time?
            if (m0 == (UInt4)-1){
                if (i > 0){
                    m0 = i-1;
                }else{
                    m0 = i;
                }
            }
        
        }
        // Exceed the center point?
        if (_isoBin[i] > enc){
            // First time?
            if (mp == (UInt4)-1){
                // Center will be the nearest point to the center value
                if (i > 0 && (_isoBin[i] - enc) > (enc - _isoBin[i-1])){
                    mp = i-1;
                }else{
                    mp = i;
                }
            }
        }
        // Exceed the end point?
        if (_isoBin[i] > region[3]){
            // First time?
            if (i > (UInt4)0){
                m1 = i-1;
            }else{
                m1 = i;
            }
            break;
        }
    }
    if (m0 == m1){
        mp = m0;
    }
    //cout << m0 << ":" << mp << ":" << m1 << endl;

    // Xbin at the center
    vector<Double> x_vec;
    UInt4 q0 = -1;
    UInt4 q1 = _anisoBin2d.size()-2; //[inamura 160307]
    // Get vecters at the center from the precreated 2D data 
    for (UInt4 i=0; i < _anisoBin2d.size(); i++){
        // Exceed Q range?
        if (_anisoBin2d[i][mp] > region[2]){
            q1 = i-1;
            break;
        }
        // Withen X range?
        if (_anisoBin2d[i][mp] >= region[0]){
            x_vec.push_back(_anisoBin2d[i][mp]);
            //cout << _anisoBin2d[i][mp] << endl;
            if (q0 == (UInt4)-1){
                q0 = i;
            }
        }
    }
    // Out of range?
    if (q0 == (UInt4)-1){
        return;
    }
    UInt4 dnum = x_vec.size()-1;
    
    // Prepare an integration Counter
    vector<UInt4> integNo(dnum,0);
    vector<Double> y_vec(dnum,0.0);
    vector<Double> e_vec(dnum,0.0);

    // While in integration range
    while (m1 >= m0){
        vector<Double> yy(dnum,0.0);
        vector<Double> ee(dnum,0.0);
        // Rebinning at center bin
        _Rebin(m0,x_vec, yy,ee);
        // Add up
        for(UInt4 i = 0; i < dnum; i++){
            // Both data points are not masked
            if (y_vec[i] < MASKVALUE && yy[i] < MASKVALUE){
                y_vec[i] += yy[i];
                if (integNo[1] == 1){
                    e_vec[i] = e_vec[i]*e_vec[i];
                }
                e_vec[i] += ee[i] * ee[i];
                // Counts up integration No.
                integNo[i] += 1;
            }
            // Source data is masked, but object is not.
            else if (y_vec[i] >= MASKVALUE){
                y_vec[i] = yy[i];
                e_vec[i] = ee[i];
                integNo[i] = 1;
            }
        }
        m0++;
    }
    
    for(UInt4 i = 0; i < dnum; i++){
        // If not data point is masked
        if (y_vec[i] < MASKVALUE){
            if (integNo[i] > 0){
                if (average){
                    y_vec[i] = y_vec[i] / (Double)integNo[i];
                    e_vec[i] = sqrt(e_vec[i]) / (Double)integNo[i]; 
                }
                else{
                    e_vec[i] = sqrt(e_vec[i]);
                }
            }
            else{
                // No data case
                y_vec[i] = MASKVALUE;
                e_vec[i] = 0.0;
            }
        }
    }

    string x_key = _keys[0];
    if (_givenAxesKeys[0].first!="") x_key = _givenAxesKeys[0].first;
    ec_res.Add(x_key, x_vec, _units[1]);
    ec_res.Add(_keys[1], y_vec, _units[0]);
    ec_res.Add(_keys[2], e_vec, _units[2]);
    ec_res.SetKeys( x_key, _keys[1], _keys[2] );
    // Make header and set label
    _makeHeader(region[1], region[3], average, ec_res, false);
    // Set the center angle to the header
    _AddAngle(ec_res.PutHeaderPointer(), q0, q1);
}

////////// Set Polar Angle to Header //////////
void UtsusemiEcsClipper::
_AddAngle(HeaderBase* hh, Int4 index0, Int4 index1)
{
    Double angle0 = _Angles[index0];
    Double angle1 = _Angles[index1];
    if (angle0 < 0.0 || angle1 < 0.0){
        return;
    }
    Double center_angle = (angle0+angle1)/2.0;
    vector<Double> vecPpa;
    vecPpa.push_back(center_angle);
    vecPpa.push_back(0.0);
    // Set the center angle to the header
    hh->Add(UTSUSEMI_KEY_HEAD_PIXELPOLARANGLES, vecPpa);
}

////////// Make E bin data //////////
void UtsusemiEcsClipper::
_MakeEbinCutData(vector<Double> region, bool average, ElementContainer& ec_res)
{
    Double y0 = region[1];
    Double y1 = region[3];

    // Search start and end points from Y-axis
    UInt4 yEnd = _isoBin.size()-1;
    UInt4 index = -1;

    // Make xbin range from Enegy-Axis (iso-bin)
    for (UInt4 i = 0; i < _isoBin.size(); i++){
        if (_isoBin[i] > y1){
            yEnd = i;
            break;
        }
        // Exceed the first point?
        if (_isoBin[i] >= y0){
            if(index == -1){
                index = i;
            }
        }
    }
    //cout << "Enegy Index range: " << index << " : " << yEnd << endl;

    // Q-Range
    Double left = region[0];
    Double right = region[2];

    // Prepare an integration Counter
    vector<Double> x_vec;    
    vector<Double> y_vec;
    vector<Double> e_vec;

    UInt4 xnum = _anisoBin2d.size();

    Double r0, r1;

    // Each X-bin (Y-Axis : Q)
    while(yEnd > index){
        // No data region
        /* //[inamura 140605]
        if ((_anisoBin2d[0][index] > left && _anisoBin2d[0][index+1] > left) or
            (_anisoBin2d[xnum-1][index] < right && _anisoBin2d[xnum-1][index+1] < right)){
            y_vec.push_back(MASKVALUE);
            e_vec.push_back(0.0);
            x_vec.push_back(_isoBin[index]);
            index++;
            continue;    
        }
        */
        // Zero Width case 
        if (left == right){
            //[inamura 140605]-->
            if ((_anisoBin2d[0][index] > left && _anisoBin2d[0][index+1] > left) or
                (_anisoBin2d[xnum-1][index] < right && _anisoBin2d[xnum-1][index+1] < right)){
                y_vec.push_back(MASKVALUE);
                e_vec.push_back(0.0);
                x_vec.push_back(_isoBin[index]);
                index++;
                continue;    
            }
            //<--[inamura 140605]
            for(UInt4 k=0; k < xnum-1; k++){
                if (_anisoBin2d[k+1][index] < left && _anisoBin2d[k+1][index+1] < left){
                    continue;
                }
                if ((_anisoBin2d[k+1][index] >= left && _anisoBin2d[k+1][index+1] >= left) || k == xnum - 2 ){
                    // Center or edge point
                    if (_ints2d[k][index] >= MASKVALUE){
                        y_vec.push_back(MASKVALUE);
                        e_vec.push_back(0.0);
                    }
                    else{
                        y_vec.push_back(_ints2d[k][index]);
                        e_vec.push_back(_errs2d[k][index]);
                    }
                    x_vec.push_back(_isoBin[index]);
                }
                else{
                    // On the boundary point
                    if (_ints2d[k][index] >= MASKVALUE){
                        if (_ints2d[k+1][index] >= MASKVALUE){
                            y_vec.push_back(MASKVALUE);
                            e_vec.push_back(0.0);
                        }
                        else{
                            y_vec.push_back(_ints2d[k][index]);
                            e_vec.push_back(_errs2d[k][index]);
                        }
                        x_vec.push_back(_isoBin[index]);
                    }
                    else{
                        if (_ints2d[k+1][index] >= MASKVALUE){
                            y_vec.push_back(_ints2d[k][index]);
                            e_vec.push_back(_errs2d[k][index]);
                            x_vec.push_back(_isoBin[index]);
                        }
                        else{
                            if (_anisoBin2d[k+1][index] >= left){
                                r0 = (_anisoBin2d[k+1][index] - left) / (_anisoBin2d[k+1][index] - _anisoBin2d[k+1][index+1]);
                                r1 = (left - _anisoBin2d[k+1][index+1]) / (_anisoBin2d[k+1][index] - _anisoBin2d[k+1][index+1]);    
                            }
                            else{
                                r0 = (_anisoBin2d[k+1][index+1] - left) / (_anisoBin2d[k+1][index+1] - _anisoBin2d[k+1][index]);
                                r1 = (left - _anisoBin2d[k+1][index]) / (_anisoBin2d[k+1][index+1] - _anisoBin2d[k+1][index]);
                            }
                            y_vec.push_back(_ints2d[k][index]*r0 + _ints2d[k+1][index]*r1);
                            e_vec.push_back(sqrt(_errs2d[k][index]*_errs2d[k][index]*r0+_errs2d[k+1][index]*_errs2d[k+1][index]*r1));
                            x_vec.push_back(_isoBin[index]);
                        }
                    }
                    
                }
                break;
            }
        }
        else{
            Double sumcount = 0.0;
            Double sumerr = 0.0;
            Double sumarea = 0.0;
            
            Double l01 = _anisoBin2d[0][index] - left;
            Double l11 = _anisoBin2d[0][index+1] - left;
            Double r01 = _anisoBin2d[0][index] - right;
            Double r11 = _anisoBin2d[0][index+1] - right;
            Double l00, l10, r00, r10;

            Double area;    // Weight for minuend
            Double exarea;  // Weight for subtracter
            Double effect;
            

            // Search data within X-range
            for(UInt4 k=0; k < xnum-1; k++){
                l00 = l01;
                l10 = l11;
                l01 = _anisoBin2d[k+1][index] - left;
                l11 = _anisoBin2d[k+1][index+1] - left;
                r00 = r01;
                r10 = r11;
                r01 = _anisoBin2d[k+1][index] - right;
                r11 = _anisoBin2d[k+1][index+1] - right;

                if (_ints2d[k][index] >= MASKVALUE){ 
                    continue;
                }
                // Calc minuend's qrea (Weight)
                // Branch of 3 x 3 ways by slicing method of parallelogram
                if (l01 < 0){
                    if (l11 < 0){
                        continue;
                    }
                    else if (l10 < 0){
                        area = l11 * l11 / (l11 - l01);
                    }
                    else{
                        area = l11 * l11 / (l11 - l01) - l10 * l10 / (l10 - l00);
                    }
                }
                else if(l00 < 0 ){
                    if (l11 < 0) {
                        area = l01 * l01 / (l01 - l11);
                    }
                    else if (l10 < 0){
                        area = l01 + l11;
                    }
                    else{
                        area = (l01 + l11) - l10 * l10 / (l10 - l00);
                    }
                }
                else{
                    if (l11 < 0){
                        area = l01 * l01 / (l01 - l11) - l00 * l00 / (l00 - l10);
                    }
                    else if (l10 < 0){
                        area = (l01 + l11) - l00 * l00 / (l00 - l10);
                    }
                    else{
                        area = (l01 - l00) + (l11 - l10);
                    }
                }
                // Calc Subtracter's area (Weight)
                // Branch of 3 x 3 ways by slicing method of parallelogram
                if (r01 < 0){
                    if (r11 < 0){
                        exarea = 0;
                    }
                    else if (r10 < 0){
                        exarea = r11 * r11 / (r11 - r01);
                    }
                    else{
                        exarea = r11 * r11 / (r11 - r01) - r10 * r10 / (r10 - r00);
                    }
                }
                else if (r00 < 0){
                    if (r11 < 0){
                        exarea = r01 * r01 / (r01 - r11);
                    }
                    else if (r10 < 0){
                        exarea = r01 + r11;
                    }
                    else{
                        exarea = (r01 + r11) - r10 * r10 / (r10 - r00);
                    }
                }
                else{
                    if (r11 < 0 ){
                        exarea = r01 * r01 / (r01 - r11) - r00 * r00 / (r00 - r10);
                    }
                    else if (r10 < 0){
                        exarea = (r01 + r11) - r00 * r00 / (r00 - r10);
                    }
                    else{
                        break;
                    }
                }
                // Calc Intensity and err's effective weight 
                effect = area - exarea;
                sumcount = sumcount + _ints2d[k][index] * effect;

                sumerr = sumerr + _errs2d[k][index] * _errs2d[k][index] * effect * effect;
                sumarea = sumarea + effect;

            }
            if (sumarea == 0.0){
                y_vec.push_back(MASKVALUE);
                e_vec.push_back(0.0);
            }
            else{
                if (average){
                    y_vec.push_back(sumcount / sumarea);
                    e_vec.push_back(sqrt(sumerr) / sumarea);
                }
                else{
                    y_vec.push_back(sumcount);
                    e_vec.push_back(sqrt(sumerr));
                }
            }
            x_vec.push_back(_isoBin[index]);
        }
        index++;
    }
    // The last point
    x_vec.push_back(_isoBin[index]);
    
    // Cut off masked redion in taile 
    index = y_vec.size()-1;
    while ( index > 0){
        if (y_vec[index] < MASKVALUE){
            break;
        } 
        x_vec.pop_back();
        y_vec.pop_back();
        e_vec.pop_back();
        --index;
    }
    //cout << "Size of x : " << x_vec.size() << endl;
    //cout << "Size of y : " << y_vec.size() << endl;

    // Add xbin, int, error vectors
    ec_res.Add(_keys[0], x_vec, _units[0]);
    ec_res.Add(_keys[1], y_vec, _units[1]);
    ec_res.Add(_keys[2], e_vec, _units[2]);
    ec_res.SetKeys( _keys[0], _keys[1], _keys[2] );

    // Make header and set label
    _makeHeader(region[0], region[2], average, ec_res, false);
}

////////// Make Header and  Set Label //////////
void UtsusemiEcsClipper::
_makeHeader(Double y0, Double y1, bool average, ElementContainer& ec_res, bool reverse)
{
    // Add xbin, int, error vectors
    HeaderBase hh_ec;
    char label[100];
    if (_ei > 0.0 || reverse){
        if (average){
            sprintf(label,  "Average of %.3f to %.3f", y0, y1);
        }
        else{
            sprintf(label, "Integrated %.3f to %.3f", y0, y1);
        }
    }
    else{
        if (average){
            sprintf(label,  "Average of %d to %d", (int)(y0), (int)(y1));
        }
        else{
            sprintf(label, "Integrated %d to %d",(int)(y0), (int)(y1));
        }
    }
    hh_ec.Add(UTSUSEMI_KEY_HEAD_LABEL,string(label));
    
    // Add runno 
    hh_ec.Add(UTSUSEMI_KEY_HEAD_RUNNUMBER, _runNo);
    ec_res.InputHeader( hh_ec);

    return;
}
