#include "UtsusemiSqeCalc2.hh"
//////////////////////////////////////////////////////////
UtsusemiSqeCalc2::
UtsusemiSqeCalc2(){
    _Initialize();
}
//////////////////////////////////////////////////////////
UtsusemiSqeCalc2::
UtsusemiSqeCalc2( bool isDebug ){
    _Initialize();
    isDEBUG = isDebug;
}
//////////////////////////////////////////////////////////
UtsusemiSqeCalc2::
UtsusemiSqeCalc2( std::string _instType ){
    _Initialize();
    SetInstGeometry( _instType );
}
//////////////////////////////////////////////////////////
UtsusemiSqeCalc2::
~UtsusemiSqeCalc2(){
    delete stool;
    delete UCC;
    delete _XP;
    gsl_matrix_free(L);
    gsl_matrix_free(U);
    gsl_matrix_free(UBI);
    gsl_matrix_free(R);
    gsl_matrix_free(V);
    gsl_matrix_free(A);
}
const Int4 UtsusemiSqeCalc2::ROTATE_AXIS_X = 0;
const Int4 UtsusemiSqeCalc2::ROTATE_AXIS_Y = 1;
const Int4 UtsusemiSqeCalc2::ROTATE_AXIS_Z = 2;
//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::
_Initialize(){
    stool = new StringTools();
    isDEBUG = UtsusemiEnvGetDebugMode();
    //MASKVALUE = 100000000.0;
    MASKVALUE = UTSUSEMIMASKVALUE64;
    MessageTag = "UtsusemiSqeCalc2:: ";
    UCC = new UtsusemiUnitConverter();
    isDirectGeometry = true;

    _NumOfMulTh = UtsusemiGetNumOfMulTh();
    _XP = new UtsusemiSqeCalcXtalParams();
    isSetXtalParams = false;

    L = gsl_matrix_alloc(3,3);
    U = gsl_matrix_alloc(3,3);
    UBI = gsl_matrix_alloc(3,3);
    R = gsl_matrix_alloc(3,3);
    V = gsl_matrix_alloc(3,3);
    A = gsl_matrix_alloc(3,3);
    gsl_matrix_set_all( L, 0.0 );
    gsl_matrix_set_all( U, 0.0 );
    gsl_matrix_set_all( UBI, 0.0 );
    gsl_matrix_set_all( R, 0.0 );
    gsl_matrix_set_identity( R );
    gsl_matrix_set_all( V, 0.0 );
    gsl_matrix_set_all( A, 0.0 );
}
//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::
_CalcReciprocalLatticeMatrix( std::vector<Double> latticeConst, gsl_matrix* M ){
    Double alpha = latticeConst[3]/180.0*M_PI;
    Double beta =  latticeConst[4]/180.0*M_PI;
    Double gamma = latticeConst[5]/180.0*M_PI;

    gsl_vector* a = gsl_vector_alloc(3);
    gsl_vector* b = gsl_vector_alloc(3);
    gsl_vector* c = gsl_vector_alloc(3);
    gsl_vector_set( a, 0, latticeConst[0] );
    gsl_vector_set( a, 1, 0.0 );
    gsl_vector_set( a, 2, 0.0 );
    gsl_vector_set( b, 0, latticeConst[1]*cos(gamma) );
    gsl_vector_set( b, 1, latticeConst[1]*sin(gamma) );
    gsl_vector_set( b, 2, 0.0 );
    /*
    gsl_vector_set( c, 0, latticeConst[2]*cos(alpha) );
    gsl_vector_set( c, 1, latticeConst[2]*cos(beta)*sin(gamma) );
    Double c3 = fabs(latticeConst[2])*sqrt( (1.0 - cos(alpha)*cos(alpha) + (cos(beta)*cos(beta))*(sin(beta)*sin(beta))) );
    */
    gsl_vector_set( c, 0, latticeConst[2]*cos(beta) );
    gsl_vector_set( c, 1, latticeConst[2]*cos(alpha)*sin(gamma) );
    Double c3 = fabs(latticeConst[2])*sqrt( (1.0 - cos(beta)*cos(beta) - (cos(alpha)*sin(gamma))*(cos(alpha)*sin(gamma))) );
    gsl_vector_set( c, 2, c3 );
    if (isDEBUG){
        std::cout << "vectora =" << gsl_vector_get(a,0) << "," << gsl_vector_get(a,1) << "," << gsl_vector_get(a,2) << std::endl;
        std::cout << "vectorb =" << gsl_vector_get(b,0) << "," << gsl_vector_get(b,1) << "," << gsl_vector_get(b,2) << std::endl;
        std::cout << "vectorb =" << gsl_vector_get(c,0) << "," << gsl_vector_get(c,1) << "," << gsl_vector_get(c,2) << std::endl;
    }

    gsl_vector* astar = gsl_vector_alloc(3);
    gsl_vector* bstar = gsl_vector_alloc(3);
    gsl_vector* cstar = gsl_vector_alloc(3);

    _CalcCrossProduct( b, c, astar );
    _CalcCrossProduct( c, a, bstar );
    _CalcCrossProduct( a, b, cstar );

    Double v0;
    gsl_blas_ddot( a, astar, &v0 );
    if (isDEBUG) std::cout << "v0=" << v0 << std::endl;

    gsl_blas_dscal( (2.0*M_PI/v0), astar );
    gsl_blas_dscal( (2.0*M_PI/v0), bstar );
    gsl_blas_dscal( (2.0*M_PI/v0), cstar );

    _SetBasisVectToMatrix( astar, bstar, cstar, M );

    gsl_vector_free( a );
    gsl_vector_free( b );
    gsl_vector_free( c );
    gsl_vector_free( astar );
    gsl_vector_free( bstar );
    gsl_vector_free( cstar );

    if (isDEBUG){
        std::cout << "-------- Matrix L (Inverted Lattice vector)-----------" << std::endl;
        std::cout << "vecRa ";
        std::cout << gsl_matrix_get(M,0,0) <<","<<gsl_matrix_get(M,0,1) <<","<<gsl_matrix_get(M,0,2)<<std::endl;
        std::cout << "vecRb ";
        std::cout << gsl_matrix_get(M,1,0) <<","<<gsl_matrix_get(M,1,1) <<","<<gsl_matrix_get(M,1,2) <<std::endl;
        std::cout << "vecRc ";
        std::cout << gsl_matrix_get(M,2,0) <<","<<gsl_matrix_get(M,2,1) <<","<<gsl_matrix_get(M,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
    }

}
//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::
_CalcUVWMatrix( std::vector<Double> Uvec, std::vector<Double> Vvec, gsl_matrix*M, gsl_matrix* N ){

    // copy Reciprocal Lattice Matrix
    gsl_vector* astar = gsl_vector_alloc(3);
    gsl_vector* bstar = gsl_vector_alloc(3);
    gsl_vector* cstar = gsl_vector_alloc(3);

    gsl_vector_set( astar, 0, gsl_matrix_get( M, 0, 0 ) );
    gsl_vector_set( astar, 1, gsl_matrix_get( M, 1, 0 ) );
    gsl_vector_set( astar, 2, gsl_matrix_get( M, 2, 0 ) );
    gsl_vector_set( bstar, 0, gsl_matrix_get( M, 0, 1 ) );
    gsl_vector_set( bstar, 1, gsl_matrix_get( M, 1, 1 ) );
    gsl_vector_set( bstar, 2, gsl_matrix_get( M, 2, 1 ) );
    gsl_vector_set( cstar, 0, gsl_matrix_get( M, 0, 2 ) );
    gsl_vector_set( cstar, 1, gsl_matrix_get( M, 1, 2 ) );
    gsl_vector_set( cstar, 2, gsl_matrix_get( M, 2, 2 ) );

    gsl_vector* uvec = gsl_vector_alloc(3);
    gsl_vector* vvec = gsl_vector_alloc(3);
    gsl_vector* wvec = gsl_vector_alloc(3);
    gsl_vector_set_all( uvec, 0.0 );
    gsl_vector_set_all( vvec, 0.0 );
    gsl_vector_set_all( wvec, 0.0 );

    gsl_vector* atmp = gsl_vector_alloc(3);
    gsl_vector* btmp = gsl_vector_alloc(3);
    gsl_vector* ctmp = gsl_vector_alloc(3);

    // Make U-vec normalized
    gsl_vector_memcpy( atmp, astar ); // astar -> atmp
    gsl_vector_memcpy( btmp, bstar );
    gsl_vector_memcpy( ctmp, cstar );

    gsl_blas_dscal( Uvec[0], atmp ); // Uvec[0]*atmp = atmp
    gsl_blas_dscal( Uvec[1], btmp );
    gsl_blas_dscal( Uvec[2], ctmp );

    gsl_vector_add( uvec, atmp ); // uvec = uvec+atmp
    gsl_vector_add( uvec, btmp );
    gsl_vector_add( uvec, ctmp );

    gsl_blas_dscal((1.0/gsl_blas_dnrm2( uvec )), uvec); // Normalization of U-vec

    // Make U-vec normalized
    gsl_vector_memcpy( atmp, astar ); // astar -> atmp
    gsl_vector_memcpy( btmp, bstar );
    gsl_vector_memcpy( ctmp, cstar );

    gsl_blas_dscal( Vvec[0], atmp ); // Vvec[0]*atmp = atmp
    gsl_blas_dscal( Vvec[1], btmp );
    gsl_blas_dscal( Vvec[2], ctmp );

    gsl_vector_add( vvec, atmp ); // vvec = vvec+atmp
    gsl_vector_add( vvec, btmp );
    gsl_vector_add( vvec, ctmp );

    Double v1;
    gsl_vector *tmp = gsl_vector_alloc(3);
    gsl_blas_ddot( uvec, vvec, &v1 );
    gsl_vector_memcpy( tmp, uvec ); // U-vec -> tmp
    gsl_blas_dscal( v1, tmp );      // vi*tmp -> tmp

    gsl_vector_sub( vvec, tmp );
    gsl_blas_dscal((1.0/gsl_blas_dnrm2( vvec )), vvec ); // Normalization of U-vec

    gsl_vector_free( tmp );

    // Make W-vec normalized
    _CalcCrossProduct( uvec, vvec, wvec );

    if (isDEBUG){
        std::cout << "VecU norm";
        std::cout << gsl_vector_get(uvec,0) <<","<<gsl_vector_get(uvec,1) <<","<<gsl_vector_get(uvec,2) <<std::endl;
        std::cout << "VecV norm";
        std::cout << gsl_vector_get(vvec,0) <<","<<gsl_vector_get(vvec,1) <<","<<gsl_vector_get(vvec,2) <<std::endl;
    }

    // Set to Matrix
    gsl_matrix_set_all( N, 0.0 );
    //_SetBasisVectToMatrix( uvec, vvec, wvec, N );
    _SetBasisVectToMatrix( vvec, wvec, uvec, N ); //#[inamura 200512]

    gsl_vector_free( astar );
    gsl_vector_free( bstar );
    gsl_vector_free( cstar );
    gsl_vector_free( uvec );
    gsl_vector_free( vvec );
    gsl_vector_free( wvec );
    gsl_vector_free( atmp );
    gsl_vector_free( btmp );
    gsl_vector_free( ctmp );

    if (isDEBUG){
        std::cout << "-------- Matrix UVW ----------------------------------" << std::endl;
        std::cout << "U-vec ";
        std::cout << gsl_matrix_get(N,0,2) <<","<<gsl_matrix_get(N,1,2) <<","<<gsl_matrix_get(N,2,2)<<std::endl;
        std::cout << "V-vec ";
        std::cout << gsl_matrix_get(N,0,0) <<","<<gsl_matrix_get(N,1,0) <<","<<gsl_matrix_get(N,2,0) <<std::endl;
        std::cout << "W-vec ";
        std::cout << gsl_matrix_get(N,0,1) <<","<<gsl_matrix_get(N,1,1) <<","<<gsl_matrix_get(N,2,1) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
    }

}
//////////////////////////////////////////////////////////
bool UtsusemiSqeCalc2::
_CalcRotateMatrix( UInt4 rot_axis, Double angle, gsl_matrix* M ){

    gsl_matrix* Ro = gsl_matrix_alloc(3,3);
    gsl_matrix_memcpy( Ro, M ); // R->Ro
    gsl_matrix* T = gsl_matrix_alloc(3,3);
    gsl_matrix_set_all(T,0.0);

    bool isValidAxis = true;
    if (rot_axis==ROTATE_AXIS_X){       // Xaxis
        double ang_rad = -angle*M_PI/180.0;  // clockwise from axis perpendicular of Y and Z axes  > 0

        gsl_matrix_set( T, 0, 0, 1.0 );
        gsl_matrix_set( T, 0, 1, 0.0 );
        gsl_matrix_set( T, 0, 2, 0.0 );
        gsl_matrix_set( T, 1, 0, 0.0 );
        gsl_matrix_set( T, 1, 1, cos(ang_rad) );
        gsl_matrix_set( T, 1, 2,-sin(ang_rad) );
        gsl_matrix_set( T, 2, 0, 0.0 );
        gsl_matrix_set( T, 2, 1, sin(ang_rad) );
        gsl_matrix_set( T, 2, 2, cos(ang_rad) );
        /*
        gsl_matrix_set( T, 0, 0, cos(ang_rad) );
        gsl_matrix_set( T, 0, 1, 0.0 );
        gsl_matrix_set( T, 0, 2, sin(ang_rad) );
        gsl_matrix_set( T, 1, 0, 0.0 );
        gsl_matrix_set( T, 1, 1, 1.0 );
        gsl_matrix_set( T, 1, 2, 0.0 );
        gsl_matrix_set( T, 2, 0, -sin(ang_rad) );
        gsl_matrix_set( T, 2, 1, 0.0 );
        gsl_matrix_set( T, 2, 2, cos(ang_rad) );
        */

    }else if (rot_axis==ROTATE_AXIS_Y){ // Yaxis
        double ang_rad = -angle*M_PI/180.0;  // clockwise from bird view > 0

        gsl_matrix_set( T, 0, 0, cos(ang_rad) );
        gsl_matrix_set( T, 0, 1, 0.0 );
        gsl_matrix_set( T, 0, 2, sin(ang_rad) );
        gsl_matrix_set( T, 1, 0, 0.0 );
        gsl_matrix_set( T, 1, 1, 1.0 );
        gsl_matrix_set( T, 1, 2, 0.0 );
        gsl_matrix_set( T, 2, 0,-sin(ang_rad) );
        gsl_matrix_set( T, 2, 1, 0.0 );
        gsl_matrix_set( T, 2, 2, cos(ang_rad) );
        /*
        gsl_matrix_set( T, 0, 0, cos(ang_rad) );
        gsl_matrix_set( T, 0, 1, -sin(ang_rad) );
        gsl_matrix_set( T, 0, 2, 0.0 );
        gsl_matrix_set( T, 1, 0, sin(ang_rad) );
        gsl_matrix_set( T, 1, 1, cos(ang_rad) );
        gsl_matrix_set( T, 1, 2, 0.0 );
        gsl_matrix_set( T, 2, 0, 0.0 );
        gsl_matrix_set( T, 2, 1, 0.0 );
        gsl_matrix_set( T, 2, 2, 1.0 );
        */

    }else if (rot_axis==ROTATE_AXIS_Z){ // Zaxis
        double ang_rad = -angle*M_PI/180.0;  // clockwise from upstream > 0

        gsl_matrix_set( T, 0, 0, cos(ang_rad) );
        gsl_matrix_set( T, 0, 1,-sin(ang_rad) );
        gsl_matrix_set( T, 0, 2, 0.0 );
        gsl_matrix_set( T, 1, 0, sin(ang_rad) );
        gsl_matrix_set( T, 1, 1, cos(ang_rad) );
        gsl_matrix_set( T, 1, 2, 0.0 );
        gsl_matrix_set( T, 2, 0, 0.0 );
        gsl_matrix_set( T, 2, 1, 0.0 );
        gsl_matrix_set( T, 2, 2, 1.0 );
        /*
        gsl_matrix_set( T, 0, 0, 1.0 );
        gsl_matrix_set( T, 0, 1, 0.0 );
        gsl_matrix_set( T, 0, 2, 0.0 );
        gsl_matrix_set( T, 1, 0, 0.0 );
        gsl_matrix_set( T, 1, 1, cos(ang_rad) );
        gsl_matrix_set( T, 1, 2, -sin(ang_rad) );
        gsl_matrix_set( T, 2, 0, 0.0 );
        gsl_matrix_set( T, 2, 1, sin(ang_rad) );
        gsl_matrix_set( T, 2, 2, cos(ang_rad) );
        */

    }else{
        UtsusemiError(MessageTag+"_CalcRotateMatrix :: axis is invarid.");
        isValidAxis = false;
    }

    if (isValidAxis){
        gsl_blas_dgemm( CblasNoTrans, CblasNoTrans, 1.0, Ro, T, 0.0, M ); // 1.0*[Ro][T]+0.0*R -> R
    }

    gsl_matrix_free( Ro );
    gsl_matrix_free( T );

    return isValidAxis;
}
//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::
_SetBasisVectToMatrix( gsl_vector* v1, gsl_vector* v2, gsl_vector* v3, gsl_matrix* M ){
    gsl_matrix_set_all(M,0.0);
    for (UInt4 row=0; row<3; row++){
        gsl_matrix_set( M, row, 0,  gsl_vector_get( v1, row ) );
        gsl_matrix_set( M, row, 1,  gsl_vector_get( v2, row ) );
        gsl_matrix_set( M, row, 2,  gsl_vector_get( v3, row ) );
    }
}
//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::
_SetBasisVectToMatrix( std::vector<Double> va, gsl_matrix* M ){
    gsl_matrix_set_all(M,0.0);
    for (UInt4 col=0; col<3; col++){
        for (UInt4 row=0; row<3; row++){
            gsl_matrix_set( M, row, col, va[ col*3 + row ] );
        }
    }
}
//////////////////////////////////////////////////////////
/*
void UtsusemiSqeCalc2::
_SetViewMatrix( std::vector<Double> view_axes, gsl_matrix* M ){
    gsl_matrix_set_all(M,0.0);
    for (UInt4 col=0; col<3; col++){
        for (UInt4 row=0; row<3; row++){
            gsl_matrix_set( M, row, col, view_axes[ col*4 + row ] );
        }
    }
}
*/
//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::
_CalcCrossProduct( gsl_vector* v1, gsl_vector* v2, gsl_vector* ret ){

    Double va = gsl_vector_get( v1, 1 )*gsl_vector_get( v2, 2 ) - gsl_vector_get( v1, 2 )*gsl_vector_get( v2, 1 );
    Double vb = gsl_vector_get( v1, 2 )*gsl_vector_get( v2, 0 ) - gsl_vector_get( v1, 0 )*gsl_vector_get( v2, 2 );
    Double vc = gsl_vector_get( v1, 0 )*gsl_vector_get( v2, 1 ) - gsl_vector_get( v1, 1 )*gsl_vector_get( v2, 0 );

    gsl_vector_set( ret, 0, va );
    gsl_vector_set( ret, 1, vb );
    gsl_vector_set( ret, 2, vc );

}
//////////////////////////////////////////////////////////
bool UtsusemiSqeCalc2::
_CalcInverseMatrix( gsl_matrix *Min, gsl_matrix *Mout ){
    gsl_permutation *p = gsl_permutation_alloc(3);
    int sig;
    gsl_linalg_LU_decomp( Min, p, &sig );
    double det=gsl_linalg_LU_det( Min, sig );
    if (isDEBUG){
        std::cout << "sig="<<sig<<std::endl;
        std::cout << "det="<<det<<std::endl;
    }
    if (det==0.0) return false;

    gsl_linalg_LU_invert( Min, p, Mout );
    gsl_permutation_free(p);
    return true;
    /*
    double a11 = gsl_matrix_get( Min, 0, 0 );
    double a12 = gsl_matrix_get( Min, 0, 1 );
    double a13 = gsl_matrix_get( Min, 0, 2 );
    double a21 = gsl_matrix_get( Min, 1, 0 );
    double a22 = gsl_matrix_get( Min, 1, 1 );
    double a23 = gsl_matrix_get( Min, 1, 2 );
    double a31 = gsl_matrix_get( Min, 2, 0 );
    double a32 = gsl_matrix_get( Min, 2, 1 );
    double a33 = gsl_matrix_get( Min, 2, 2 );
    double det = a11*a22*a33 + a12*a23*a31 + a13*a21*a32 - a13*a22*a31 - a12*a21*a33 - a11*a23*a32;
    if (det==0.0) return false;

    gsl_matrix_set( Mout, 0, 0, (a22*a33-a23*a32)/det );
    gsl_matrix_set( Mout, 0, 1, -1.0*(a12*a33-a13*a32)/det );
    gsl_matrix_set( Mout, 0, 2, (a12*a23-a13*a22)/det );
    gsl_matrix_set( Mout, 1, 0, -1.0*(a21*a33-a23*a31)/det );
    gsl_matrix_set( Mout, 1, 1, (a11*a33-a13*a31)/det );
    gsl_matrix_set( Mout, 1, 2, -1.0*(a11*a23-a13*a21)/det );
    gsl_matrix_set( Mout, 2, 0, (a21*a32-a22*a31)/det );
    gsl_matrix_set( Mout, 2, 1, -1.0*(a11*a32-a12*a31)/det );
    gsl_matrix_set( Mout, 2, 2, (a11*a22-a12*a21)/det );

    return true;
    */
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
_ConvMatrixToVect( gsl_matrix *Min ){
    std::vector<Double> ret;
    for (UInt4 i=0; i<(Min->size1); i++)
        for (UInt4 j=0;j<(Min->size2); j++)
            ret.push_back( gsl_matrix_get( Min, i, j ) );
    return ret;
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::InverseMatrix( std::vector<Double> a ){
    std::vector<Double> ret;

    gsl_matrix* Min = gsl_matrix_alloc(3,3);
    gsl_matrix* Mout = gsl_matrix_alloc(3,3);
    gsl_matrix_set_all(Mout, 0.0);
    UInt4 ind = 0;
    for (UInt4 i=0; i<3; i++){
        for (UInt4 j=0; j<3; j++){
            gsl_matrix_set( Min, i, j, a[ind] );
            ind++;
        }
    }

    if (_CalcInverseMatrix( Min, Mout )){
        double a11 = gsl_matrix_get( Mout, 0, 0 );
        ret.push_back(a11);
        double a12 = gsl_matrix_get( Mout, 0, 1 );
        ret.push_back(a12);
        double a13 = gsl_matrix_get( Mout, 0, 2 );
        ret.push_back(a13);
        double a21 = gsl_matrix_get( Mout, 1, 0 );
        ret.push_back(a21);
        double a22 = gsl_matrix_get( Mout, 1, 1 );
        ret.push_back(a22);
        double a23 = gsl_matrix_get( Mout, 1, 2 );
        ret.push_back(a23);
        double a31 = gsl_matrix_get( Mout, 2, 0 );
        ret.push_back(a31);
        double a32 = gsl_matrix_get( Mout, 2, 1 );
        ret.push_back(a32);
        double a33 = gsl_matrix_get( Mout, 2, 2 );
        ret.push_back(a33);
        if (UtsusemiEnvGetDebugMode()){
            std::cout << a11 << "," << a12 << "," << a13 << std::endl;
            std::cout << a21 << "," << a22 << "," << a23 << std::endl;
            std::cout << a31 << "," << a32 << "," << a33 << std::endl;
        }
    }else{
        if (UtsusemiEnvGetDebugMode()) std::cout << "Det = 0" << std::endl;
    }
    gsl_matrix_free( Min );
    gsl_matrix_free( Mout );
    return ret;
}
//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::TEST_InverseMatrix( std::vector<Double> a ){
    gsl_matrix* Min = gsl_matrix_alloc(3,3);
    gsl_matrix* Mout = gsl_matrix_alloc(3,3);
    gsl_matrix_set_all(Mout, 0.0);
    UInt4 ind = 0;
    for (UInt4 i=0; i<3; i++){
        for (UInt4 j=0; j<3; j++){
            gsl_matrix_set( Min, i, j, a[ind] );
            ind++;
        }
    }

    if (_CalcInverseMatrix( Min, Mout )){
        double a11 = gsl_matrix_get( Mout, 0, 0 );
        double a12 = gsl_matrix_get( Mout, 0, 1 );
        double a13 = gsl_matrix_get( Mout, 0, 2 );
        double a21 = gsl_matrix_get( Mout, 1, 0 );
        double a22 = gsl_matrix_get( Mout, 1, 1 );
        double a23 = gsl_matrix_get( Mout, 1, 2 );
        double a31 = gsl_matrix_get( Mout, 2, 0 );
        double a32 = gsl_matrix_get( Mout, 2, 1 );
        double a33 = gsl_matrix_get( Mout, 2, 2 );
        std::cout << a11 << "," << a12 << "," << a13 << std::endl;
        std::cout << a21 << "," << a22 << "," << a23 << std::endl;
        std::cout << a31 << "," << a32 << "," << a33 << std::endl;
    }else{
        std::cout << "Det = 0" << std::endl;
    }
    gsl_matrix_free( Min );
    gsl_matrix_free( Mout );
}
//////////////////////////////////////////////////////////
bool UtsusemiSqeCalc2::
_MakeUBIMatrix( std::vector<Double> latticeConst, std::vector<Double> Uvec, std::vector<Double> Vvec ){
    if (latticeConst.size()!=6){
        UtsusemiError(MessageTag+"_MakeUBIMatrix >> argument latticeConst size is invalid (must be 6)");
        return false;
    }
    if (Uvec.size()!=3){
        UtsusemiError(MessageTag+"_MakeUBIMatrix >> argument Uvec size is invalid (must be 3)");
        return false;
    }
    if (Vvec.size()!=3){
        UtsusemiError(MessageTag+"_MakeUBIMatrix >> argument Vvec size is invalid (must be 3)");
        return false;
    }
    gsl_matrix_set_all( L, 0.0 );
    gsl_matrix_set_all( U, 0.0 );
    _CalcReciprocalLatticeMatrix( latticeConst, L );
    _CalcUVWMatrix( Uvec, Vvec, L, U );
    gsl_matrix* Li = gsl_matrix_alloc(3,3);
    _CalcInverseMatrix( L, Li );

    gsl_blas_dgemm( CblasNoTrans, CblasNoTrans, 1.0, Li, U, 0.0, UBI ); // 1.0*[L^(-1)][U] -> UBI
    if (isDEBUG){
        std::cout << "--------------- given Uvec and Vvec ---------------------------"<<std::endl;
        std::cout << "U-vec " << Uvec[0] <<","<< Uvec[1] <<","<< Uvec[2] <<std::endl;
        std::cout << "V-vec " << Vvec[0] <<","<< Vvec[1] <<","<< Vvec[2] <<std::endl;
        std::cout << std::endl;
        std::cout << "-------- Matrix [L]-1 -----------------------------" << std::endl;
        std::cout << "a1,b1,c1 " << gsl_matrix_get(Li,0,0) <<","<< gsl_matrix_get(Li,0,1) <<","<< gsl_matrix_get(Li,0,2) <<std::endl;
        std::cout << "a2,b2,c2 " << gsl_matrix_get(Li,1,0) <<","<< gsl_matrix_get(Li,1,1) <<","<< gsl_matrix_get(Li,1,2) <<std::endl;
        std::cout << "a3,b3,c3 " << gsl_matrix_get(Li,2,0) <<","<< gsl_matrix_get(Li,2,1) <<","<< gsl_matrix_get(Li,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
        std::cout << "-------- Matrix [U] -----------------------------" << std::endl;
        std::cout << "U-vec " << gsl_matrix_get(U,0,0) <<","<< gsl_matrix_get(U,0,1) <<","<< gsl_matrix_get(U,0,2) <<std::endl;
        std::cout << "V-vec " << gsl_matrix_get(U,1,0) <<","<< gsl_matrix_get(U,1,1) <<","<< gsl_matrix_get(U,1,2) <<std::endl;
        std::cout << "W-vec " << gsl_matrix_get(U,2,0) <<","<< gsl_matrix_get(U,2,1) <<","<< gsl_matrix_get(U,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
        std::cout << "-------- Matrix [UBI] -----------------------------" << std::endl;
        std::cout << "      " << gsl_matrix_get(UBI,0,0) <<","<< gsl_matrix_get(UBI,0,1) <<","<< gsl_matrix_get(UBI,0,2) <<std::endl;
        std::cout << "UBI=  " << gsl_matrix_get(UBI,1,0) <<","<< gsl_matrix_get(UBI,1,1) <<","<< gsl_matrix_get(UBI,1,2) <<std::endl;
        std::cout << "      " << gsl_matrix_get(UBI,2,0) <<","<< gsl_matrix_get(UBI,2,1) <<","<< gsl_matrix_get(UBI,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
    }
    gsl_matrix_free(Li);

    return true;
}
//////////////////////////////////////////////////////////
bool UtsusemiSqeCalc2::
_MakeUBIMatrix( std::vector<Double> UBmat ){
    if (UBmat.size()!=9){
        UtsusemiError(MessageTag+"_MakeUBIMatrix >> argument UBmat size is invalid (must be 9)");
        return false;
    }
    gsl_matrix_set_all( UBI, 0.0 );
    gsl_matrix* UB = gsl_matrix_alloc(3,3);
    for (UInt4 i=0; i<3; i++)
        for (UInt4 j=0; j<3; j++)
            gsl_matrix_set(UB,i,j,UBmat[ i*3 + j ]);
    _CalcInverseMatrix( UB, UBI );
    gsl_matrix_free(UB);
    return true;
}
//////////////////////////////////////////////////////////
bool UtsusemiSqeCalc2::
_MakeProjectionMatrix( std::vector<Double> rotateSteps, std::vector<Double> viewAxes ){
    gsl_matrix_set_all( R, 0.0 );
    gsl_matrix_set_identity( R );
    gsl_matrix_set_all( V, 0.0 );
    gsl_matrix_set_all( A, 0.0 );
    std::vector<Double> va(9,0.0);
    if (viewAxes.size()==9)
        copy( viewAxes.begin(), viewAxes.end(), va.begin() );
    else if (viewAxes.size()==16){
        for (UInt4 i=0;i<3;i++)
            for (UInt4 j=0;j<3;j++)
                va[i*3+j]=viewAxes[i*4+j];
    }else{
        UtsusemiError(MessageTag + "_MakeProjectionMatrix >>> viewAxes size is invalid. 9 or 16 but given " + stool->Int4ToString((Int4)(viewAxes.size())));
        return false;
    }
    _SetBasisVectToMatrix( va, V);

    for (UInt4 i=0;i<(rotateSteps.size()/2);i++){
        _CalcRotateMatrix( (UInt4)(rotateSteps[2*i]), rotateSteps[2*i+1], R);
    }
    //_SetViewMatrix( viewAxes, V );

    gsl_matrix* T1 = gsl_matrix_alloc(3,3);
    gsl_matrix* T2 = gsl_matrix_alloc(3,3);
    gsl_matrix* T3 = gsl_matrix_alloc(3,3);

    gsl_blas_dgemm( CblasNoTrans, CblasNoTrans, 1.0, UBI, R, 0.0, T1 ); // 1.0*[UB-1][R] -> T1
    _CalcInverseMatrix( V, T2 ); //T2 = V^(-1)
    gsl_blas_dgemm( CblasNoTrans, CblasNoTrans, 1.0, T2, T1, 0.0, T3 ); // 1.0*[V^(-1)][T1] -> T3 = [V^(-1)][UB-1][R]
    gsl_matrix_memcpy( A, T3 );  // T3->A

    if (isDEBUG){
        std::cout << "-------- Matrix [R]---------------------------" << std::endl;
        std::cout << gsl_matrix_get(R,0,0) <<","<<gsl_matrix_get(R,0,1) <<","<<gsl_matrix_get(R,0,2)<< std::endl;
        std::cout << gsl_matrix_get(R,1,0) <<","<< gsl_matrix_get(R,1,1) <<","<< gsl_matrix_get(R,1,2) <<std::endl;
        std::cout << gsl_matrix_get(R,2,0) <<","<< gsl_matrix_get(R,2,1) <<","<< gsl_matrix_get(R,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
        std::cout << "-------- Matrix [UB-1][R]---------------------------" << std::endl;
        std::cout << gsl_matrix_get(T1,0,0) <<","<< gsl_matrix_get(T1,0,1) <<","<< gsl_matrix_get(T1,0,2) <<std::endl;
        std::cout << gsl_matrix_get(T1,1,0) <<","<< gsl_matrix_get(T1,1,1) <<","<< gsl_matrix_get(T1,1,2) <<std::endl;
        std::cout << gsl_matrix_get(T1,2,0) <<","<< gsl_matrix_get(T1,2,1) <<","<< gsl_matrix_get(T1,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
        std::cout << "-------- Matrix [V] ----------------------------------" << std::endl;
        std::cout << gsl_matrix_get(V,0,0) <<","<< gsl_matrix_get(V,0,1) <<","<< gsl_matrix_get(V,0,2) <<std::endl;
        std::cout << gsl_matrix_get(V,1,0) <<","<< gsl_matrix_get(V,1,1) <<","<< gsl_matrix_get(V,1,2) <<std::endl;
        std::cout << gsl_matrix_get(V,2,0) <<","<< gsl_matrix_get(V,2,1) <<","<< gsl_matrix_get(V,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
        std::cout << "-------- Matrix [V-1][UB-1][R] -----------------------" << std::endl;
        std::cout << gsl_matrix_get(T3,0,0) <<","<< gsl_matrix_get(T3,0,1) <<","<< gsl_matrix_get(T3,0,2) <<std::endl;
        std::cout << gsl_matrix_get(T3,1,0) <<","<< gsl_matrix_get(T3,1,1) <<","<< gsl_matrix_get(T3,1,2) <<std::endl;
        std::cout << gsl_matrix_get(T3,2,0) <<","<< gsl_matrix_get(T3,2,1) <<","<< gsl_matrix_get(T3,2,2) <<std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
    }

    gsl_matrix_free(T1);
    gsl_matrix_free(T2);
    gsl_matrix_free(T3);

    return true;
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
MakeProjectionMatrix( std::vector<Double> latticeConst, std::vector<Double> Uvec, std::vector<Double> Vvec, std::vector<Double> rotateSteps, std::vector<Double> viewAxes ){
    std::vector<Double> empty;
    if (_MakeUBIMatrix(latticeConst, Uvec, Vvec)){
        if (_MakeProjectionMatrix(  rotateSteps, viewAxes )){
        }else{
            return empty;
        }
    }else{
        return empty;
    }

    std::vector<Double> ret(16,0.0);
    for (UInt4 i=0; i<3; i++){
        for (UInt4 j=0; j<3; j++){
            ret[i*4 + j] = gsl_matrix_get(A,i,j);
        }
    }
    ret[15] = 1.0;
    if (isDEBUG){
        std::cout << "-------- Matrix [A]=[V][L][UVW] ----------------------" << std::endl;
        std::cout << ret[0] << "," << ret[1] << "," << ret[2] << std::endl;
        std::cout << ret[4] << "," << ret[5] << "," << ret[6] << std::endl;
        std::cout << ret[8] << "," << ret[9] << "," << ret[10] << std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
    }
    return ret;
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
MakeProjectionMatrix( std::vector<Double> UBmat, std::vector<Double> rotateSteps, std::vector<Double> viewAxes ){
    std::vector<Double> empty;
    if (_MakeUBIMatrix(UBmat)){
        if (_MakeProjectionMatrix(  rotateSteps, viewAxes )){
        }else{
            return empty;
        }
    }else{
        return empty;
    }

    std::vector<Double> ret(16,0.0);
    for (UInt4 i=0; i<3; i++){
        for (UInt4 j=0; j<3; j++){
            ret[i*4 + j] = gsl_matrix_get(A,i,j);
        }
    }
    ret[15] = 1.0;
    if (isDEBUG){
        std::cout << "-------- Matrix [A]=[V][L][UVW] ----------------------" << std::endl;
        std::cout << ret[0] << "," << ret[1] << "," << ret[2] << std::endl;
        std::cout << ret[4] << "," << ret[5] << "," << ret[6] << std::endl;
        std::cout << ret[8] << "," << ret[9] << "," << ret[10] << std::endl;
        std::cout << "------------------------------------------------------" << std::endl;
    }
    return ret;
}







//////////////////////////////////////////////////////////
void UtsusemiSqeCalc2::
SetInstGeometry( std::string _instType ){
    if ((_instType!="Direct") and (_instType!="Inverted")){
        return;
    }
    if (_instType=="Direct") isDirectGeometry=true;
    if (_instType=="Inverted") isDirectGeometry=false;
}




//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
Projection( std::vector<Double> vaxis){
    ElementContainerMatrix* ecm = Put();
    if (ecm->PutPointer(0)->PutPointer(0)->PutHeaderPointer()->CheckKey(UTSUSEMI_KEY_HEAD_EF)==1){
        isDirectGeometry = false;
        UtsusemiMessage("UtsusemiSqeCalc2 >> Projection :: Instrument of given data is indirectGeometry");
    }else{
        UtsusemiMessage("UtsusemiSqeCalc2 >> Projection :: Instrument of given data is directGeometry");
    }
    if (isSetXtalParams){
        if (_XP->SetSampleInfoToData( ecm )){
        }
    }
    HeaderBase* hh = ecm->PutHeaderPointer();
    //isDEBUG=false;
    std::vector<Double> latticeConst = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_LATTICECONSTS);
    std::vector<Double> Uvec         = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_UVECT);
    std::vector<Double> Vvec         = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_VVECT);
    std::vector<Double> ubMat;
    std::string pri = UTSUSEMI_KEY_HEAD_SAMPLE_LATTICECONSTS;
    if (hh->CheckKey(UTSUSEMI_KEY_HEAD_SAMPLE_UBMATRIX)==1)
        ubMat = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_UBMATRIX);
    if (hh->CheckKey(UTSUSEMI_KEY_HEAD_SAMPLE_UBPRIORITY)==1)
        pri = hh->PutString(UTSUSEMI_KEY_HEAD_SAMPLE_UBPRIORITY);

    std::vector<Double> rotateSteps;
    if (hh->CheckKey(UTSUSEMI_KEY_HEAD_SAMPLE_ROTATESTEPS)==1)
        rotateSteps  = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_ROTATESTEPS);
    if ( (hh->CheckKey("SampleRotatePhi")==1) && (hh->CheckKey(UTSUSEMI_KEY_HEAD_SAMPLE_ROTATESTEPS)==0) ){
        rotateSteps.resize(2);
        rotateSteps[0]=1.0;
        rotateSteps[1]=hh->PutDouble("SampleRotatePhi");
    }
    //std::vector<Double> rotateSteps  = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_ROTATESTEPS);

    std::vector<Double> A;
    A.clear();
    if ((pri==UTSUSEMI_KEY_HEAD_SAMPLE_LATTICECONSTS)||(ubMat.empty()))
        A = MakeProjectionMatrix( latticeConst, Uvec, Vvec, rotateSteps, vaxis );
    else
        A = MakeProjectionMatrix( ubMat, rotateSteps, vaxis );
    if (A.empty()){
        UtsusemiError(MessageTag+"Projection >>> Projection failed.");
        return A;
    }

    Double Ei = 0.0;
    Double ki = 0.0;
    if (isDirectGeometry){
        Ei = hh->PutDouble(UTSUSEMI_KEY_HEAD_EI);
        ki = sqrt( UCC->EtoK2( Ei ) );
    }

    //clock_t t1 = clock();
    for (UInt4 i=0;i<ecm->PutTableSize();i++){
        ElementContainerArray *eca = ecm->PutPointer(i);
        HeaderBase *eca_h = eca->PutHeaderPointer();
        if ((eca_h->CheckKey(UTSUSEMI_KEY_HEAD_DETTYPE)==1)&&(eca_h->PutString(UTSUSEMI_KEY_HEAD_DETTYPE)==UTSUSEMI_KEY_HEAD_DETTYPE_MONITOR)) continue;
        for (UInt4 j=0;j<eca->PutTableSize();j++){
            ElementContainer *ec = eca->PutPointer(j);
            HeaderBase *ec_h = ec->PutHeaderPointer();

            Double Ef = 0.0;
            Double kf = 0.0;
            Double PA = 0.0;
            Double AA = 0.0;
            if (isDirectGeometry){
                std::vector<Double> vPA = ec_h->PutDoubleVector(UTSUSEMI_KEY_HEAD_PIXELPOLARANGLES);
                std::vector<Double> vAA = ec_h->PutDoubleVector(UTSUSEMI_KEY_HEAD_PIXELAZIMANGLES);
                PA = vPA[0]*M_PI/180.0;
                AA = vAA[0]*M_PI/180.0;
            }else{
                Ef = ec_h->PutDouble(UTSUSEMI_KEY_HEAD_EF);
                kf = sqrt( UCC->EtoK2( Ef ) );
                PA = ec_h->PutDouble(UTSUSEMI_KEY_HEAD_POLARANGLE);
                AA = ec_h->PutDouble(UTSUSEMI_KEY_HEAD_AZIMANGLE);
            }

            Double px = sin(PA)*cos(AA);
            Double py = sin(PA)*sin(AA);
            Double pz = cos(PA);
            //std::vector<Double> pv = ec_h->PutDoubleVector(UTSUSEMI_KEY_HEAD_PIXELPOSITION);
            //Double L2 = sqrt( pv[0]*pv[0]+pv[1]*pv[1]+pv[2]*pv[2] );
            //Double px = pv[0]/L2;
            //Double py = pv[1]/L2;
            //Double pz = pv[2]/L2;

            std::vector<Double> xx = ec->PutX();
            UInt4 size_xx = (UInt4)(xx.size());

            if (ec->CheckKey(UTSUSEMI_KEY_VIEWX)==1) ec->Remove(UTSUSEMI_KEY_VIEWX);
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWY)==1) ec->Remove(UTSUSEMI_KEY_VIEWY);
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWZ)==1) ec->Remove(UTSUSEMI_KEY_VIEWZ);
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWW)==1) ec->Remove(UTSUSEMI_KEY_VIEWW);

            std::vector<Double> tmp;
            ec->Add( UTSUSEMI_KEY_VIEWX,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );
            ec->Add( UTSUSEMI_KEY_VIEWY,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );
            ec->Add( UTSUSEMI_KEY_VIEWZ,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );
            ec->Add( UTSUSEMI_KEY_VIEWW,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );

            std::vector<Double> *Vxi = (*ec)(UTSUSEMI_KEY_VIEWX);
            std::vector<Double> *Vyi = (*ec)(UTSUSEMI_KEY_VIEWY);
            std::vector<Double> *Vzi = (*ec)(UTSUSEMI_KEY_VIEWZ);
            std::vector<Double> *Vwi = (*ec)(UTSUSEMI_KEY_VIEWW);
            Vxi->resize( (size_xx -1), 0.0 );
            Vyi->resize( (size_xx -1), 0.0 );
            Vzi->resize( (size_xx -1), 0.0 );
            Vwi->resize( (size_xx -1), 0.0 );

            for (UInt4 k=0;k<(xx.size()-1);k++){
                Double hw = (xx[k+1]+xx[k])/2.0;

                if (isDirectGeometry){
                    kf = sqrt( UCC->EtoK2( Ei-hw ) );
                }else{
                    ki = sqrt( UCC->EtoK2( Ef+hw ) );
                }

                Double qx = -kf*px;
                Double qy = -kf*py;
                Double qz = ki-kf*pz;

                Vxi->at(k) = qx*A[0] + qy*A[1] + qz*A[2] + hw*A[3];
                Vyi->at(k) = qx*A[4] + qy*A[5] + qz*A[6] + hw*A[7];
                Vzi->at(k) = qx*A[8] + qy*A[9] + qz*A[10] + hw*A[11];
                Vwi->at(k) = qx*A[12] + qy*A[13] + qz*A[14] + hw*A[15];

                /*
                Vxi->at(k) = qz*A[0] + qx*A[1] + qy*A[2] + hw*A[3];
                Vyi->at(k) = qz*A[4] + qx*A[5] + qy*A[6] + hw*A[7];
                Vzi->at(k) = qz*A[8] + qx*A[9] + qy*A[10] + hw*A[11];
                Vwi->at(k) = qz*A[12] + qx*A[13] + qy*A[14] + hw*A[15];
                */
            }
        }
    }
    _LastConv3DMatrixElements.clear();
    _LastConv3DMatrixElements.resize(9,0.0);
    for (UInt4 i=0; i<3; i++)
        for (UInt4 j=0; j<3; j++)
            _LastConv3DMatrixElements[i*3+j] = A[i*4+j];

    //std::cout << "Projection time=" << (Double)(clock()-t1) << std::endl;
    return A;
}

//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
Projection_QxQyQz( std::vector<Double> vaxis){
    ElementContainerMatrix* ecm = Put();
    if (isSetXtalParams){
        if (_XP->SetSampleInfoToData( ecm )){
        }
    }
    HeaderBase* hh = ecm->PutHeaderPointer();
    //isDEBUG=false;
    std::vector<Double> latticeConst = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_LATTICECONSTS);
    std::vector<Double> Uvec         = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_UVECT);
    std::vector<Double> Vvec         = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_VVECT);
    std::vector<Double> rotateSteps  = hh->PutDoubleVector(UTSUSEMI_KEY_HEAD_SAMPLE_ROTATESTEPS);

    std::vector<Double> A = MakeProjectionMatrix( latticeConst, Uvec, Vvec, rotateSteps, vaxis );

    //clock_t t1 = clock();
    for (UInt4 i=0;i<ecm->PutTableSize();i++){
        ElementContainerArray *eca = ecm->PutPointer(i);
        HeaderBase *eca_h = eca->PutHeaderPointer();
        if ((eca_h->CheckKey(UTSUSEMI_KEY_HEAD_DETTYPE)==1)&&(eca_h->PutString(UTSUSEMI_KEY_HEAD_DETTYPE)==UTSUSEMI_KEY_HEAD_DETTYPE_MONITOR)) continue;
        for (UInt4 j=0;j<eca->PutTableSize();j++){
            ElementContainer *ec = eca->PutPointer(j);
            HeaderBase *ec_h = ec->PutHeaderPointer();

            if ( (ec->CheckKey(UTSUSEMI_KEY_QX)==0) || (ec->CheckKey(UTSUSEMI_KEY_QY)==0) || (ec->CheckKey(UTSUSEMI_KEY_QZ)==0) || (ec->CheckKey(UTSUSEMI_KEY_HW)==0) ){
                UtsusemiError(MessageTag+"Projection_QxQyQz > There is no Qx, Qy, Qz information");
                return A;
            }

            std::vector<Double>* Qx = (*ec)(UTSUSEMI_KEY_QX);
            std::vector<Double>* Qy = (*ec)(UTSUSEMI_KEY_QY);
            std::vector<Double>* Qz = (*ec)(UTSUSEMI_KEY_QZ);
            std::vector<Double>* hw = (*ec)(UTSUSEMI_KEY_HW);
            std::vector<Double>* II = (*ec)(UTSUSEMI_KEY_INTENSITY);
            UInt4 size_q = (UInt4)(II->size());
            if ( (Qx->size()!=size_q)||(Qy->size()!=size_q)||(Qz->size()!=size_q) ){
                UtsusemiError(MessageTag+"Projection_QxQyQz > Size of Qx, Qy, Qz is invalid.");
                return A;
            }

            if (ec->CheckKey(UTSUSEMI_KEY_VIEWX)==1) ec->Remove(UTSUSEMI_KEY_VIEWX);
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWY)==1) ec->Remove(UTSUSEMI_KEY_VIEWY);
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWZ)==1) ec->Remove(UTSUSEMI_KEY_VIEWZ);
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWW)==1) ec->Remove(UTSUSEMI_KEY_VIEWW);

            std::vector<Double> tmp;
            ec->Add( UTSUSEMI_KEY_VIEWX,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );
            ec->Add( UTSUSEMI_KEY_VIEWY,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );
            ec->Add( UTSUSEMI_KEY_VIEWZ,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );
            ec->Add( UTSUSEMI_KEY_VIEWW,tmp,UTSUSEMI_KEY_MOMENTTRANSFER_UNIT );

            std::vector<Double> *Vxi = (*ec)(UTSUSEMI_KEY_VIEWX);
            std::vector<Double> *Vyi = (*ec)(UTSUSEMI_KEY_VIEWY);
            std::vector<Double> *Vzi = (*ec)(UTSUSEMI_KEY_VIEWZ);
            std::vector<Double> *Vwi = (*ec)(UTSUSEMI_KEY_VIEWW);
            Vxi->resize( size_q, 0.0 );
            Vyi->resize( size_q, 0.0 );
            Vzi->resize( size_q, 0.0 );
            Vwi->resize( size_q, 0.0 );

            for (UInt4 k=0;k<size_q;k++){
                Vxi->at(k) = (Qx->at(k))*A[0] + (Qy->at(k))*A[1] + (Qz->at(k))*A[2] + (hw->at(k))*A[3];
                Vyi->at(k) = (Qx->at(k))*A[4] + (Qy->at(k))*A[5] + (Qz->at(k))*A[6] + (hw->at(k))*A[7];
                Vzi->at(k) = (Qx->at(k))*A[8] + (Qy->at(k))*A[9] + (Qz->at(k))*A[10] + (hw->at(k))*A[11];
                Vwi->at(k) = (Qx->at(k))*A[12] + (Qy->at(k))*A[13] + (Qz->at(k))*A[14] + (hw->at(k))*A[15];
                /*
                Vxi->at(k) = (Qz->at(k))*A[0] + (Qx->at(k))*A[1] + (Qy->at(k))*A[2] + (hw->at(k))*A[3];
                Vyi->at(k) = (Qz->at(k))*A[4] + (Qx->at(k))*A[5] + (Qy->at(k))*A[6] + (hw->at(k))*A[7];
                Vzi->at(k) = (Qz->at(k))*A[8] + (Qx->at(k))*A[9] + (Qy->at(k))*A[10] + (hw->at(k))*A[11];
                Vwi->at(k) = (Qz->at(k))*A[12] + (Qx->at(k))*A[13] + (Qy->at(k))*A[14] + (hw->at(k))*A[15];
                */

            }
        }
    }
    _LastConv3DMatrixElements.clear();
    _LastConv3DMatrixElements.resize(9,0.0);
    for (UInt4 i=0; i<3; i++)
        for (UInt4 j=0; j<3; j++)
            _LastConv3DMatrixElements[i*3+j] = A[i*4+j];

    //std::cout << "Projection time=" << (Double)(clock()-t1) << std::endl;
    return A;
}




//////////////////////////////////////////////////////////
Int4 UtsusemiSqeCalc2::
Slice( std::vector<Double> Ax1range,std::vector<Double> Ax2range,std::vector<Double> Ax3range,std::vector<Double> Ax4range,std::vector<std::string> AxesType, std::vector<Double> Folding, bool isAverageMode ){

    std::vector< std::vector<Double> > ranges;
    ranges.push_back( Ax1range );
    ranges.push_back( Ax2range );
    ranges.push_back( Ax3range );
    ranges.push_back( Ax4range );

    // Set index for each Axis in "ranges"
    UInt4 indX  = 99;
    UInt4 indY  = 99;
    UInt4 indT1 = 99;
    UInt4 indT2 = 99;

    for (UInt4 i=0;i<AxesType.size();i++){
        if ((AxesType[i]=="X")||(AxesType[i]=="x")){
            indX = i;
        }
        if ((AxesType[i]=="Y")||(AxesType[i]=="y")){
            indY = i;
        }
        if ((AxesType[i]=="T")||(AxesType[i]=="t")){
            if (indT1==99){
                indT1 = i;
            }else{
                indT2 = i;
            }
        }
    }

    // Get ranges of each axis for slicing
    std::vector<Double> xrange = CalcRangeAsBinCenterZero( ranges[indX][0],ranges[indX][1],ranges[indX][2] );
    std::vector<Double> yrange = CalcRangeAsBinCenterZero( ranges[indY][0],ranges[indY][1],ranges[indY][2] );
    Double *T1range = new Double[2];
    T1range[0] = ranges[indT1][0];
    T1range[1] = ranges[indT1][1];
    Double *T2range = new Double[2];
    T2range[0] = ranges[indT2][0];
    T2range[1] = ranges[indT2][1];

    UInt4 n_X = (UInt4)xrange[2];
    UInt4 n_Y = (UInt4)yrange[2];

    // Set Folding parameters
    Double FoldingX = Folding[ indX ];
    Double FoldingY = Folding[ indY ];
    Double FoldingT1 = Folding[ indT1 ];
    Double FoldingT2 = Folding[ indT2 ];
    bool isFoldingFYminusFX = false; // Y=X line
    bool isFoldingFYplusFX = false;  // Y=-X line
    UInt4 indFX = 0;
    UInt4 indFY = 0;

    if ((Folding.size()==7)&&(Folding[4]>0.0)){
        if (Folding[4]==1){
            isFoldingFYminusFX = true;
        }else if (Folding[4]==2){
            isFoldingFYplusFX = true;
        }else if (Folding[4]==3){
            isFoldingFYminusFX = true;
            isFoldingFYplusFX = true;
        }else{
        }

        /*
        std::vector<UInt4> dam;
        for (UInt4 i = 5; i < 7; i++){
            if ((UInt4)Folding[i] == indX) indFX = (UInt4)Folding[i];
            else if ((UInt4)Folding[i] == indY) indFY = (UInt4)Folding[i];
            else dam.push_back(i);
        }
        for (UInt4 i = 0; i < dam.size(); i++){
            if (indFX == 9) indFX = dam[i];
            else if (indFY == 9) indFY = dam[i];
        }
        // indFX = (UInt4)Folding[5];
        // indFY = (UInt4)Folding[6];
        */
        indFX = (UInt4)Folding[5];
        indFY = (UInt4)Folding[6];
    }

#ifdef MULTH
    omp_set_num_threads( _NumOfMulTh );
#endif

    // Make and initialize temporal matrix for storing sliced data
    Double ***dArray, ***eArray, ***cArray;
    dArray = new Double**[ _NumOfMulTh ];
    eArray = new Double**[ _NumOfMulTh ];
    cArray = new Double**[ _NumOfMulTh ];
    for (UInt4 i=0;i<_NumOfMulTh;i++){
        dArray[i] = new Double*[ n_X ];
        eArray[i] = new Double*[ n_X ];
        cArray[i] = new Double*[ n_X ];

        for (UInt4 j=0;j<n_X;j++){
            dArray[i][j] = new Double[ n_Y ];
            eArray[i][j] = new Double[ n_Y ];
            cArray[i][j] = new Double[ n_Y ];
        }
    }

    for (UInt4 k=0;k<_NumOfMulTh;k++){
        for (UInt4 i=0;i<n_X;i++){
            for (UInt4 j=0;j<n_Y;j++){
                dArray[k][i][j]=eArray[k][i][j]=cArray[k][i][j]=0.0;
            }
        }
    }

    // Slicing
    ElementContainerMatrix *ecm = Put();

    bool isHistogram = true;
    if ((ecm->PutHeaderPointer()->CheckKey(UTSUSEMI_KEY_HEAD_ISHISTOGRAM)==1)&&(ecm->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_ISHISTOGRAM)==0))
        isHistogram = false;

    for (UInt4 i_eca=0;i_eca<ecm->PutSize();i_eca++){
        ElementContainerArray* eca = ecm->PutPointer(i_eca);
        // if ECA is masked, skip it.
        if (eca->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_MASKED)!=0) continue;

        if (isHistogram)
            if ((eca->PutHeaderPointer()->CheckKey(UTSUSEMI_KEY_HEAD_ISHISTOGRAM)==1)&&(eca->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_ISHISTOGRAM)==0))
                isHistogram = false;

        for (UInt4 i_ec=0;i_ec<eca->PutSize();i_ec++){
            ElementContainer* ec = eca->PutPointer(i_ec);
            // if EC is masked, skip it.
            if (ec->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_MASKED)!=0) continue;

            if (isHistogram)
                if ((ec->PutHeaderPointer()->CheckKey(UTSUSEMI_KEY_HEAD_ISHISTOGRAM)==1)&&(ec->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_ISHISTOGRAM)==0))
                    isHistogram = false;

            std::vector<Double> ii, ee;
            if (isHistogram){
                HistogramBinToPoint* HTP = new HistogramBinToPoint();
                HTP->SetTarget(ec);
                HTP->Convert();
                ii = HTP->PutY();
                ee = HTP->PutE();
                delete HTP;
            }else{
                ii = ec->PutY();
                ee = ec->PutE();
            }

            std::vector<std::vector<Double>*> ax(4);
            ax[0] = (*ec)(UTSUSEMI_KEY_VIEWX);
            ax[1] = (*ec)(UTSUSEMI_KEY_VIEWY);
            ax[2] = (*ec)(UTSUSEMI_KEY_VIEWZ);
            ax[3] = (*ec)(UTSUSEMI_KEY_VIEWW);

#pragma omp parallel for
#if (_OPENMP >= 200805)  // OpenMP 3.0 and later
            for (UInt4 ind=0;ind<ax[0]->size();ind++){
#else
            for (Int4 ind=0;ind<ax[0]->size();ind++){
#endif
                UInt4 multh = omp_get_thread_num();
                if (ee[ind]<0) continue;
                //UInt4 multh = 1;
                // Pick up one point data
                std::vector<Double> AX(4,0.0); // X=AX[indX],Y=AX[indY],T1=AX[indT1],T2=AX[indT2]
                AX[indX] = ax[ indX ]->at(ind);
                AX[indY] = ax[ indY ]->at(ind);
                AX[indT1] = ax[ indT1 ]->at(ind);
                AX[indT2] = ax[ indT2 ]->at(ind);

                // Folding along diagonal line
                if ((isFoldingFYminusFX)&&(AX[indFY]<AX[indFX])){
                    Double tmp = AX[indFX];
                    AX[indFX] = AX[indFY];
                    AX[indFY] = tmp;
                }
                if ((isFoldingFYplusFX)&&(AX[indFY]<(-AX[indFX]))){
                    Double tmp = -AX[indFX];
                    AX[indFX] = -AX[indFY];
                    AX[indFY] = tmp;
                }

                // Folding along each axis
                if (FoldingX==0.0) AX[indX] = fabs(AX[indX]);
                if (FoldingX>0.0) {
                    AX[indX] = fabs(AX[indX]) - (floor(fabs(AX[indX])/(2.0*FoldingX))*(2.0*FoldingX));
                    if (AX[indX]>FoldingX) AX[indX]=(2.0*FoldingX)-AX[indX];
                }
                if (FoldingY==0.0) AX[indY] = fabs(AX[indY]);
                if (FoldingY>0.0) {
                    AX[indY] = fabs(AX[indY]) - (floor(fabs(AX[indY])/(2.0*FoldingY))*(2.0*FoldingY));
                    if (AX[indY]>FoldingY) AX[indY]=(2.0*FoldingY)-AX[indY];
                }
                if (FoldingT1==0.0) AX[indT1] = fabs(AX[indT1]);
                if (FoldingT1>0.0) {
                    AX[indT1] = fabs(AX[indT1]) - (floor(fabs(AX[indT1])/(2.0*FoldingT1))*(2.0*FoldingT1));
                    if (AX[indT1]>FoldingT1) AX[indT1]=(2.0*FoldingT1)-AX[indT1];
                }
                if (FoldingT2==0.0) AX[indT2] = fabs(AX[indT2]);
                if (FoldingT2>0.0) {
                    AX[indT2] = fabs(AX[indT2]) - (floor(fabs(AX[indT2])/(2.0*FoldingT2))*(2.0*FoldingT2));
                    if (AX[indT2]>FoldingT2) AX[indT2]=(2.0*FoldingT2)-AX[indT2];
                }

                // Add a data point to matrix if placed in given range
                if ((T1range[0]<=AX[indT1])&&(AX[indT1]<T1range[1])){
                    if ((T2range[0]<=AX[indT2])&&(AX[indT2]<T2range[1])){
                        if ((xrange[0]<=AX[indX])&&(AX[indX]<=xrange[1])){
                            if ((yrange[0]<=AX[indY])&&(AX[indY]<=yrange[1])){
                              UInt4 i_X = (UInt4)( floor((AX[indX]-xrange[0])/ranges[indX][2]) );
                              UInt4 i_Y = (UInt4)( floor((AX[indY]-yrange[0])/ranges[indY][2]) );
                                if ((i_X<n_X)&&(i_Y<n_Y)){
                                    dArray[multh][i_X][i_Y] += ii[ind];
                                    eArray[multh][i_X][i_Y] += ee[ind]*ee[ind];
                                    cArray[multh][i_X][i_Y] += 1.0;
                                }else{
                                  //std::cout << " Out of range " << i_X <<"/"<< n_X <<  "," << i_Y <<"/"<<n_Y<< std::endl;
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    // Make results from temporal matrix
    datArray.clear();
    errArray.clear();
    XArray.clear();
    YArray.clear();

    for (UInt4 i=0;i<(n_X+1);i++){
        XArray.push_back( xrange[0] + ranges[indX][2]*(Double(i)) );
    }
    for (UInt4 i=0;i<(n_Y+1);i++){
        YArray.push_back( yrange[0] + ranges[indY][2]*(Double(i)) );
    }

    for (UInt4 i=0;i<n_X;i++){
        std::vector<Double> datY,datE;
        datY.clear();
        datE.clear();
        for (UInt4 j=0;j<n_Y;j++){
            Double Yav = 0.0;
            Double Eav = 0.0;
            Double Cav = 0.0;
            for (UInt4 k=0;k<_NumOfMulTh;k++){
                Yav += dArray[k][i][j];
                Eav += eArray[k][i][j];
                Cav += cArray[k][i][j];
            }

            if (Cav!=0.){
                if (isAverageMode){
                    datY.push_back( Yav/Cav );
                    datE.push_back( sqrt( Eav )/Cav );
                }else{
                    datY.push_back( Yav );
                    datE.push_back( sqrt( Eav ) );
                }
            }else{
                datY.push_back( MASKVALUE );
                datE.push_back( 0.0 );
            }
        }
        datArray.push_back(datY);
        errArray.push_back(datE);
    }

    // Delete temporal matrix
    for (UInt4 i=0;i<_NumOfMulTh;i++){
        for (UInt4 j=0;j<n_X;j++){
            delete [] dArray[i][j];
            delete [] eArray[i][j];
            delete [] cArray[i][j];
        }
        delete [] dArray[i];
        delete [] eArray[i];
        delete [] cArray[i];
    }
    delete [] dArray;
    delete [] eArray;
    delete [] cArray;

    return 0;
}

//////////////////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
PutDArray(){
    std::vector<Double> ret;
    ret.clear();
    for (UInt4 i=0;i<datArray.size();i++){
        for (UInt4 j=0;j<datArray[i].size();j++){
            ret.push_back( datArray[i][j] );
        }
    }
    return ret;
}

//////////////////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
PutEArray(){
    std::vector<Double> ret;
    ret.clear();
    for (UInt4 i=0;i<errArray.size();i++){
        for (UInt4 j=0;j<errArray[i].size();j++){
            ret.push_back( errArray[i][j] );
        }
    }
    return ret;
}

//////////////////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::
GetAxLimit(){

    Double vx_min,vx_max,vy_min,vy_max,vz_min,vz_max,vw_min,vw_max;
    vx_min = vy_min = vz_min = vw_min = 100000000.0;
    vx_max = vy_max = vz_max = vw_max = -100000000.0;
    Double min_tmp,max_tmp;

    ElementContainerMatrix* ecm = Put();
    if (ecm==NULL){
        std::vector<Double> empty_ret(8,0.0);
        return empty_ret;
    }
    for (UInt4 i=0;i<(ecm->PutTableSize());i++){
        ElementContainerArray *eca = ecm->PutPointer(i);
        for (UInt4 j=0;j<(eca->PutTableSize());j++){
            ElementContainer* ec = eca->PutPointer(j);
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWX)==1){
                min_tmp = ec->Min(UTSUSEMI_KEY_VIEWX);
                max_tmp = ec->Max(UTSUSEMI_KEY_VIEWX);
                if (min_tmp<vx_min) vx_min = min_tmp;
                if (max_tmp>vx_max) vx_max = max_tmp;
            }
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWY)==1){
                min_tmp = ec->Min(UTSUSEMI_KEY_VIEWY);
                max_tmp = ec->Max(UTSUSEMI_KEY_VIEWY);
                if (min_tmp<vy_min) vy_min = min_tmp;
                if (max_tmp>vy_max) vy_max = max_tmp;
            }
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWZ)==1){
                min_tmp = ec->Min(UTSUSEMI_KEY_VIEWZ);
                max_tmp = ec->Max(UTSUSEMI_KEY_VIEWZ);
                if (min_tmp<vz_min) vz_min = min_tmp;
                if (max_tmp>vz_max) vz_max = max_tmp;
            }
            if (ec->CheckKey(UTSUSEMI_KEY_VIEWW)==1){
                min_tmp = ec->Min(UTSUSEMI_KEY_VIEWW);
                max_tmp = ec->Max(UTSUSEMI_KEY_VIEWW);
                if (min_tmp<vw_min) vw_min = min_tmp;
                if (max_tmp>vw_max) vw_max = max_tmp;
            }
        }
    }

    std::vector<Double> ret;
    ret.clear();
    ret.push_back( vx_min );
    ret.push_back( vx_max );
    ret.push_back( vy_min );
    ret.push_back( vy_max );
    ret.push_back( vz_min );
    ret.push_back( vz_max );
    ret.push_back( vw_min );
    ret.push_back( vw_max );

    return ret;
}


Int4 UtsusemiSqeCalc2::
ConvertToD4Mat(std::string filepath, std::vector< Double> ax1,std::vector< Double> ax2,
               std::vector< Double> ax3,std::vector< Double> ax4, std::vector<Double> foldings, bool isAverageMode ){

    std::string::size_type length = filepath.size();
    std::string::size_type i0 = filepath.rfind(".txt");
    std::string::size_type i1 = filepath.rfind(".vbin");
    bool isBinary = true;
    if ((i0!=std::string::npos) && (i0==(length-4))){
        isBinary = false;
    }else if ((i1!=std::string::npos) && (i1==(length-5))){
        isBinary = true;
    }else{
        std::string msg = MessageTag+"ConvertToD4Mat > Error, file name is invalid."+filepath+"\n";
        msg += "                File name was required .txt or .vbin as extention.\n";
        msg += "                length,i0,i1="+stool->Int4ToString((Int4)length)+","+stool->Int4ToString((Int4)i0)+","+stool->Int4ToString((Int4)i1);
        UtsusemiError( msg );
        return -1;
    }

    std::vector< std::vector<Double> > ArList;
    ArList.clear();
    ArList.push_back( ax1 );  // Ax1range = < min of Ax1, max of Ax1, step of Ax1 >
    ArList.push_back( ax2 );
    ArList.push_back( ax3 );
    ArList.push_back( ax4 );

    std::vector<Int4> ArNum;  // ArNum have the number of histogram box.
    for (UInt4 i=0;i<4;i++){
        std::vector< Double > ret = CalcRangeAsBinCenterZero( ArList[i][0], ArList[i][1], ArList[i][2] );
        ArList[i][0] = ret[0];
        ArList[i][1] = ret[1];
        ArNum.push_back( (Int4)ret[2] );
    }

    std::string msg = MessageTag+"ConvertToD4Mat >> \n";
    for (UInt4 i=0;i<4; i++){
        msg += "X"+stool->UInt4ToString(i)+"="+stool->DoubleToString(ArList[i][0])+","+stool->DoubleToString(ArList[i][1])+","+stool->DoubleToString(ArList[i][2])+","+stool->DoubleToString(ArNum[i])+"\n";
    }
    UtsusemiMessage( msg );

    FILE* fp=NULL;
    std::ofstream fo;
    if (isBinary){
        if ((fp=fopen(filepath.c_str(),"wb"))==NULL){
            UtsusemiError(MessageTag+"ConvertToD4Mat >> Cannot open file="+filepath);
            return -1;
        }
    }else{
        fo.open( filepath.c_str() );
        if (!fo){
            UtsusemiError(MessageTag+"ConvertToD4Mat >> Error, cannot open file."+filepath);
            return -1;
        }
    }


    // save memory under about 400MB
    UInt4 partStep = (UInt4)( (400.0*1024.0*1024.0/8.0)/(ArNum[0]*ArNum[1]*ArNum[2]) );
    if (partStep==0) partStep = 1;
    UtsusemiMessage(MessageTag+"ConvertToD4Mat > Step ="+stool->UInt4ToString(partStep));

    ElementContainerMatrix *ecm = Put();

    std::vector<float> cAx1,cAx2,cAx3,cAx4,cII,cEE;
    cAx1.clear();
    cAx2.clear();
    cAx3.clear();
    cAx4.clear();
    cII.clear();
    cEE.clear();
    HistogramBinToPoint hbt;
    for (Int4 pind=0;pind<ArNum[3];pind+=partStep){

        Double partMin=ArList[3][0]+(Double)pind * ArList[3][2];
        if (partMin>ArList[3][1]) break;
        Double partMax=partMin+ArList[3][2]*(Double)partStep;
        if (partMax>ArList[3][1]) partMax = ArList[3][1];
        UtsusemiMessage(MessageTag+"ConvertToD4Mat > hw ="+stool->DoubleToString(partMin)+"-"+stool->DoubleToString(partMax));

        UInt4 total_num = ArNum[0]*ArNum[1]*ArNum[2]*partStep;
        std::vector<Int4> *matrixP = new std::vector<Int4>( total_num, -1 );

        std::vector<float> d_cnt;
        d_cnt.clear();
        Int4 cnt = -1;

        for (UInt4 psd=0;psd<ecm->PutTableSize();psd++){
            ElementContainerArray *eca = ecm->PutPointer(psd);
            if (eca->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_MASKED)!=0) continue;
            for (UInt4 pixel=0;pixel<eca->PutTableSize();pixel++){
                ElementContainer *ec = eca->PutPointer(pixel);
                if (ec->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_MASKED)!=0) continue;

                hbt.SetTarget( ec );
                hbt.Convert();
                std::vector<Double> intensity = hbt.PutY();
                std::vector<Double> err = hbt.PutE();

                //std::vector<Double> intensity = ec->PutY();
                //std::vector<Double> err = ec->PutE();
                std::vector< std::vector<Double> > Vax;
                Vax.clear();
                Vax.push_back( ec->Put(UTSUSEMI_KEY_VIEWX) );
                Vax.push_back( ec->Put(UTSUSEMI_KEY_VIEWY) );
                Vax.push_back( ec->Put(UTSUSEMI_KEY_VIEWZ) );
                Vax.push_back( ec->Put(UTSUSEMI_KEY_VIEWW) );
                for (UInt4 i=0;i<4;i++){
                    if (foldings[i]==1){
                        for (UInt4 j=0;j<Vax[i].size();j++){
                            Vax[i][j]=fabs(Vax[i][j]);
                        }
                    }
                }
                for (UInt4 i=0;i<intensity.size();i++){
                    if (err[i]<0.0) continue;
                    if (( Vax[0][i]>=ArList[0][0])&&( Vax[0][i]<=ArList[0][1] )) {
                        if (( Vax[1][i]>=ArList[1][0])&&( Vax[1][i]<=ArList[1][1] )) {
                            if (( Vax[2][i]>=ArList[2][0])&&( Vax[2][i]<=ArList[2][1] )) {
                                if (( Vax[3][i]>=partMin)&&( Vax[3][i]<partMax )) {
                                    UInt4 ix1 = (UInt4)( (Vax[0][i]-ArList[0][0])/ArList[0][2] );
                                    UInt4 ix2 = (UInt4)( (Vax[1][i]-ArList[1][0])/ArList[1][2] );
                                    UInt4 ix3 = (UInt4)( (Vax[2][i]-ArList[2][0])/ArList[2][2] );
                                    //UInt4 ix4 = (UInt4)( (Vax[3][i]-ArList[3][0])/ArList[3][2] );
                                    UInt4 ix4p = (UInt4)( (Vax[3][i]-partMin)/ArList[3][2] );

                                    UInt4 iii = ix4p + partStep*( ix3 + ArNum[2]*( ix2 + ArNum[1]*ix1 ) );
                                    if (matrixP->at(iii)==-1){
                                        cAx1.push_back( (float)(ArList[0][0] + ((float)(ix1)+0.5)*ArList[0][2] ));
                                        cAx2.push_back( (float)(ArList[1][0] + ((float)(ix2)+0.5)*ArList[1][2] ));
                                        cAx3.push_back( (float)(ArList[2][0] + ((float)(ix3)+0.5)*ArList[2][2] ));
                                        //cAx4.push_back( (float)(ArList[3][0] + ((float)(ix4)+0.5)*ArList[3][2] ));
                                        cAx4.push_back( (float)(partMin + ((float)(ix4p)+0.5)*ArList[3][2] ));

                                        cII.push_back((float)intensity[i]);
                                        cEE.push_back((float)(err[i]*err[i]));
                                        d_cnt.push_back(1.0);
                                        cnt++;
                                        matrixP->at(iii)=cnt;

                                    }else{
                                        Int4 ind = matrixP->at(iii);
                                        cII[ ind ] += (float)intensity[i];
                                        cEE[ ind ] += (float)(err[i]*err[i]);
                                        d_cnt[ ind ] += 1.0;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        for (UInt4 i=0;i<d_cnt.size();i++){
            if (isAverageMode){
                cII[i] = cII[i]/d_cnt[i];
                cEE[i] = sqrt( cEE[i] )/d_cnt[i];
            }else{
                cEE[i] = sqrt( cEE[i] );
            }
        }
        if (isBinary){
            float data[6];
            for (UInt4 i=0;i<cII.size();i++){
                data[0] = cAx1[i];
                data[1] = cAx2[i];
                data[2] = cAx3[i];
                data[3] = cAx4[i];
                data[4] = cII[i];
                data[5] = cEE[i];
                std::fwrite( data, sizeof(data), 1, fp );
            }
        }else{
            for (UInt4 i=0;i<cII.size();i++){
                fo << cAx1[i] << "," << cAx2[i] << "," << cAx3[i] << ",";
                fo << cAx4[i] << "," << cII[i] << "," << cEE[i] << std::endl;
            }
        }
        cAx1.clear();
        cAx2.clear();
        cAx3.clear();
        cAx4.clear();
        cII.clear();
        cEE.clear();
        delete matrixP;
    }

    if (isBinary){
        fclose( fp );
    }else{
        fo.close();
    }

    UtsusemiMessage(MessageTag+"ConvertToD4Mat > complete.");
    return 0;
}

Int4 UtsusemiSqeCalc2::
ConvertToD4Mat(std::string filepath,bool isAverageMode){
    if (isSetXtalParams){
        return ConvertToD4Mat( filepath,
                               _XP->PutAxRange(0,false),
                               _XP->PutAxRange(1,false),
                               _XP->PutAxRange(2,false),
                               _XP->PutAxRange(3,false),
                               _XP->PutFolding(),
                               isAverageMode );
    }
    std::string::size_type length = filepath.size();
    std::string::size_type i0 = filepath.rfind(".txt");
    std::string::size_type i1 = filepath.rfind(".vbin");
    bool isBinary = true;
    if ((i0!=std::string::npos) && (i0==(length-4))){
        isBinary = false;
    }else if ((i1!=std::string::npos) && (i1==(length-5))){
        isBinary = true;
    }else{
        std::string msg = MessageTag+"ConvertToD4Mat > Error, file name is invalid."+filepath+"\n";
        msg += "                File name was required .txt or .vbin as extention.\n";
        msg += "                length,i0,i1="+stool->Int4ToString((Int4)length)+","+stool->Int4ToString((Int4)i0)+","+stool->Int4ToString((Int4)i1);
        UtsusemiError( msg );
        return -1;
    }

    FILE* fp=NULL;
    std::ofstream fo;
    if (isBinary){
        if ((fp=fopen(filepath.c_str(),"wb"))==NULL){
            UtsusemiError(MessageTag+"ConvertToD4Mat > Cannot open file="+filepath);
            return -1;
        }
    }else{
        fo.open( filepath.c_str() );
        if (!fo){
            UtsusemiError(MessageTag+"ConvertToD4Mat > Cannot open file."+filepath);
            return -1;
        }
    }

    ElementContainerMatrix *ecm = Put();

    HistogramBinToPoint hbt;

    for (UInt4 psd=0;psd<ecm->PutTableSize();psd++){
        ElementContainerArray *eca = ecm->PutPointer(psd);
        if (eca->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_MASKED)!=0) continue;
        for (UInt4 pixel=0;pixel<eca->PutTableSize();pixel++){
            ElementContainer *ec = eca->PutPointer(pixel);
            if (ec->PutHeaderPointer()->PutInt4(UTSUSEMI_KEY_HEAD_MASKED)!=0) continue;

            hbt.SetTarget( ec );
            hbt.Convert();
            std::vector<Double> intensity = hbt.PutY();
            std::vector<Double> err = hbt.PutE();

            std::vector<Double>* Vx = (*ec)(UTSUSEMI_KEY_VIEWX);
            std::vector<Double>* Vy = (*ec)(UTSUSEMI_KEY_VIEWY);
            std::vector<Double>* Vz = (*ec)(UTSUSEMI_KEY_VIEWZ);
            std::vector<Double>* Vw = (*ec)(UTSUSEMI_KEY_VIEWW);

            if (isBinary){
                float data[6];
                for (UInt4 i=0;i<intensity.size();i++){
                    if (err[i]<0.0) continue;
                    data[0] = (float)(Vx->at(i));
                    data[1] = (float)(Vy->at(i));
                    data[2] = (float)(Vz->at(i));
                    data[3] = (float)(Vw->at(i));
                    data[4] = (float)(intensity[i]);
                    data[5] = (float)(err[i]);
                    std::fwrite( data, sizeof(data), 1, fp );
                }
            }else{
                for (UInt4 i=0;i<intensity.size();i++){
                    if (err[i]<0.0) continue;
                    fo << Vx->at(i) << "," << Vy->at(i) << "," << Vz->at(i) << ",";
                    fo << Vw->at(i) << "," << intensity[i] << "," << err[i] << std::endl;
                }
            }
        }
    }

    if (isBinary){
        fclose( fp );
    }else{
        fo.close();
    }

    UtsusemiMessage(MessageTag+"ConvertToD4Mat >> complete.");
    return 0;
}

std::vector<Double> UtsusemiSqeCalc2::
QConvFromReciprocalToRealSpace( std::vector<Double> Qvec, std::vector<Double> LC, std::vector<Double> Uv, std::vector<Double> Vv, std::vector<Double> RS, std::vector<Double> VA ){
    std::vector<Double> ret;
    if (Qvec.size()<3) return ret;
    if ((Qvec.size()%3)!=0) return ret;
    if (LC.size()<6) return ret;
    if (Uv.size()<3) return ret;
    if (Vv.size()<3) return ret;

    // Make Conversion Matrix 'U' from (Qx,Qy,Qz) -> (Vx,Vy,Vz)
    std::vector<Double> empty;
    if (_MakeUBIMatrix(LC,Uv,Vv))
        if (_MakeProjectionMatrix( RS, VA )){
        }else{
            return empty;
        }
    else{
        return empty;
    }

    gsl_matrix* C = gsl_matrix_alloc(3,3);
    _CalcInverseMatrix( A, C );

    // Use 'C' matrix on (Vx,Vy,Vz) to get (Qx,Qy,Qz)
    std::vector<Double> A;
    A.clear();
    for (UInt4 i=0;i<3;i++){
        for (UInt4 j=0;j<3;j++){
            A.push_back( gsl_matrix_get( C, i, j ) );
        }
    }

    ret.resize(Qvec.size(), 0.0);
    for (UInt4 i=0; i<(UInt4)(Qvec.size()/3); i++){
        UInt4 x = i*3;
        UInt4 y = i*3+1;
        UInt4 z = i*3+2;
        ret[z] = Qvec[x]*A[0] + Qvec[y]*A[1] + Qvec[z]*A[2];
        ret[x] = Qvec[x]*A[3] + Qvec[y]*A[4] + Qvec[z]*A[5];
        ret[y] = Qvec[x]*A[6] + Qvec[y]*A[7] + Qvec[z]*A[8];
    }

    gsl_matrix_free( C );

    return ret;
}

void UtsusemiSqeCalc2::Test(){
    // Dummy Data of Qx,Qy,Qz in instrument system
    std::vector<Double> Qx(3);
    std::vector<Double> Qy(3);
    std::vector<Double> Qz(3);
    std::vector<Double> hw(3);
    std::vector<Double> ii(3);
    std::vector<Double> ee(3);

    Qx[0] = 1.11;
    Qx[1] = 2.22;
    Qx[2] = 3.33;
    Qy[0] = 0.52;
    Qy[1] = 1.04;
    Qy[2] = 1.56;
    Qz[0] = -3.4;
    Qz[1] = -1.7;
    Qz[2] = 0.0;
    hw[0] = 10.05;
    hw[1] = 12.11;
    hw[2] = 13.22;
    ii[0] = 100.0;
    ii[1] = 101.0;
    ii[2] = 102.0;
    ee[0] = 1.0;
    ee[1] = 1.0;
    ee[2] = 1.0;

    ElementContainer ec;
    ec.Add( UTSUSEMI_KEY_QX, Qx );
    ec.Add( UTSUSEMI_KEY_QY, Qy );
    ec.Add( UTSUSEMI_KEY_QZ, Qz );
    ec.Add( UTSUSEMI_KEY_HW, hw );
    ec.Add( UTSUSEMI_KEY_INTENSITY, ii );
    ec.Add( UTSUSEMI_KEY_ERROR, ee );
    ec.SetKeys(UTSUSEMI_KEY_HW,UTSUSEMI_KEY_INTENSITY,UTSUSEMI_KEY_ERROR);
    HeaderBase* hh_ec = ec.PutHeaderPointer();
    hh_ec->Add(UTSUSEMI_KEY_HEAD_MASKED, 0);

    ElementContainerArray eca;
    eca.Add(ec);
    HeaderBase* hh_eca =eca.PutHeaderPointer();
    hh_eca->Add(UTSUSEMI_KEY_HEAD_MASKED, 0);

    ElementContainerMatrix ecm;
    ecm.Add(eca);

    HeaderBase* hh_ecm = ecm.PutHeaderPointer();
    hh_ecm->Add(UTSUSEMI_KEY_HEAD_EI,10.0);
    hh_ecm->Add(UTSUSEMI_KEY_HEAD_MASKED, 0);
    hh_ecm->Add(UTSUSEMI_KEY_HEAD_ISHISTOGRAM, 0 );

    // Make Sample Lattice parameters
    std::vector<Double> LC(6);
    LC[0] = 4.8;
    LC[1] = 8.4;
    LC[2] = 2.91;
    LC[3] = 90.0;
    LC[4] = 90.0;
    LC[5] = 90.0;

    // Make Sample Orientation parameters
    std::vector<Double> Uv(6),Vv(6);
    Uv[0] = 0.0;
    Uv[1] = 1.0;
    Uv[2] = 0.0;
    Vv[0] = 0.0;
    Vv[1] = 0.0;
    Vv[2] = 1.0;

    std::vector<Double> RS;  // empty.

    // Set Parameters in Header
    hh_ecm->Add(UTSUSEMI_KEY_HEAD_SAMPLE_LATTICECONSTS,LC);
    hh_ecm->Add(UTSUSEMI_KEY_HEAD_SAMPLE_UVECT,Uv);
    hh_ecm->Add(UTSUSEMI_KEY_HEAD_SAMPLE_VVECT,Vv);
    hh_ecm->Add(UTSUSEMI_KEY_HEAD_SAMPLE_ROTATESTEPS,RS);

    // Set Projection Parameters
    std::vector<Double> ViewAx(16);
    ViewAx[0] = 1.;
    ViewAx[1] = 0.;
    ViewAx[2] = 0.;
    ViewAx[3] = 0.;

    ViewAx[4] = 0.;
    ViewAx[5] = 1.;
    ViewAx[6] = 0.;
    ViewAx[7] = 0.;

    ViewAx[8] = 0.;
    ViewAx[9] = 0.;
    ViewAx[10] = 1.;
    ViewAx[11] = 0.;

    ViewAx[12] = 0.;
    ViewAx[13] = 0.;
    ViewAx[14] = 0.;
    ViewAx[15] = 1.;


    // Do Projection to View space : Calculation of Vx,Vy,Vz,Vw
    // Conversion from (Qx,Qy,Qz,hw) in instrument space
    //              to (Vx,Vy,Vz,Vw) in reciprocal lattice space
    SetTarget( &ecm );
    Projection_QxQyQz( ViewAx );

    // Check converted axes in reciprocal lattice space
    ElementContainer ec_ret = ecm.Put(0).Put(0);
    std::vector<Double> Vx = ec_ret.Put(UTSUSEMI_KEY_VIEWX);
    std::vector<Double> Vy = ec_ret.Put(UTSUSEMI_KEY_VIEWY);
    std::vector<Double> Vz = ec_ret.Put(UTSUSEMI_KEY_VIEWZ);
    std::vector<Double> Vw = ec_ret.Put(UTSUSEMI_KEY_VIEWW);

    for (UInt4 i=0; i<Vx.size(); i++){
        std::cout << "["<< Qx[i] << "," << Qy[i] << "," << Qz[i] << "," << hw[i] << "]  ->  [";
        std::cout << Vx[i] << "," << Vy[i] << "," << Vz[i] << "," << Vw[i] << "]" << std::endl;
    }

    //////////////////////////////////////////////////////////////
    // Make Conversion Matrix from (Vx,Vy,Vz) to (Qx,Qy,Qz)
    //////////////////////////////////////////////////////////////
    std::vector<Double> Qvec(3);
    for (UInt4 i=0; i<Vx.size(); i++){
        Qvec[0] = Vx[i];
        Qvec[1] = Vy[i];
        Qvec[2] = Vz[i];
        std::vector<Double> iQvec = QConvFromReciprocalToRealSpace( Qvec, LC, Uv, Vv, RS, ViewAx );
        std::cout << "["<< Qx[i] << "," << Qy[i] << "," << Qz[i] << "]  ->  [";
        std::cout << Vx[i] << "," << Vy[i] << "," << Vz[i] << "]  ->  [";
        std::cout << iQvec[0] << "," << iQvec[1] << "," << iQvec[2] << "]   ";
        if ((Qx[i]-iQvec[0])<0.0000000001)
            std::cout << " X is OK /  ";
        else
            std::cout << " X is Bad Results! /  " << std::endl;
        if ((Qy[i]-iQvec[1])<0.0000000001)
            std::cout << " Y is OK /  ";
        else
            std::cout << " Y is Bad Results! /  " << std::endl;
        if ((Qz[i]-iQvec[2])<0.0000000001)
            std::cout << " Z is OK /  ";
        else
            std::cout << " Z is Bad Results! /  " << std::endl;
        std::cout << std::endl;
    }
}

bool UtsusemiSqeCalc2::
SetSlicedElementContainerArray( ElementContainerArray* _eca, std::string _bin_key, std::string _xval_key, std::string _bin_unit, std::string _xval_unit, std::string _xrange_key ){
    //std::string _bin_key="X", std::string _xval_key="XVAL", _xrange_key=UTSUSEMI_KEY_HEAD_XRANGE
    UInt4 n_X=(UInt4)(XArray.size())-1;

    ElementContainerMatrix *ecm = Put();
    _eca->InputHeader( ecm->PutHeader() );
    HeaderBase* hh = _eca->PutHeaderPointer();
    if (hh->CheckKey(UTSUSEMI_KEY_HEAD_ISHISTOGRAM)==1)
        hh->OverWrite(UTSUSEMI_KEY_HEAD_ISHISTOGRAM,0);
    else
        hh->Add(UTSUSEMI_KEY_HEAD_ISHISTOGRAM,0);
    _eca->Allocate(n_X);
    for (UInt4 i=0;i<n_X;i++){
        ElementContainer* ec = new ElementContainer();
        ec->Add(_bin_key,YArray,_bin_unit);
        ec->Add(UTSUSEMI_KEY_INTENSITY, datArray[i], UTSUSEMI_KEY_ARB_UNIT);
        ec->Add(UTSUSEMI_KEY_ERROR, errArray[i], UTSUSEMI_KEY_ARB_UNIT);
        ec->SetKeys(_bin_key,UTSUSEMI_KEY_INTENSITY,UTSUSEMI_KEY_ERROR);

        std::vector<Double> xrange(2);
        xrange[0] = XArray[i];
        xrange[1] = XArray[i+1];
        ec->AddToHeader( _xrange_key, xrange );
        ec->AddToHeader( _xval_key, (xrange[0]+xrange[1])/2.0 );
        ec->AddToHeader( MLF_KEY_HEAD_XUNIT, _xval_unit );
        ec->AddToHeader( MLF_KEY_HEAD_UNIT_PRE+_xval_key, _xval_unit );
        _eca->Set(i,ec);
    }
    return true;
}
bool UtsusemiSqeCalc2::
LoadXtalParam( std::string file ){
    isSetXtalParams = _XP->LoadFile( file );
    return isSetXtalParams;
}
bool UtsusemiSqeCalc2::
SaveXtalParam( std::string file ){
    return _XP->SaveFile( file );
}
bool UtsusemiSqeCalc2::
SetLatticeConstants( PyObject* LC ){
    isSetXtalParams = _XP->SetLatticeConstants( LC );
    return isSetXtalParams;
}
bool UtsusemiSqeCalc2::
SetUVvector( PyObject* UV, PyObject* VV ){
    isSetXtalParams = _XP->SetUVvector( UV, VV );
    return isSetXtalParams;
}
bool UtsusemiSqeCalc2::
SetUBMatrix( PyObject* UBM ){
    isSetXtalParams = _XP->SetUBMatrix( UBM );
    return isSetXtalParams;
}
bool UtsusemiSqeCalc2::
SetRotationSteps( std::string steps ){
    isSetXtalParams = _XP->SetRotationSteps( steps );
    return isSetXtalParams;
}
bool UtsusemiSqeCalc2::
Projection( PyObject* arg_ax1, PyObject* arg_ax2, PyObject* arg_ax3, PyObject* arg_ax4 ){
    if (_XP->SetProjectionAxes( arg_ax1, arg_ax2, arg_ax3, arg_ax4 )){
    }else{
        return false;
    }
    return Projection();
}
bool UtsusemiSqeCalc2::
Projection(){
    std::vector<Double> A =Projection( _XP->PutViewAxes() );
    return true;
}
bool  UtsusemiSqeCalc2::
Slice( PyObject* Ax1, PyObject* Ax2, PyObject* Ax3, PyObject* Ax4, PyObject* DiagFolding, bool isAverageMode ){
    if (!(_XP->SetSliceAxes( Ax1, Ax2, Ax3, Ax4 )))
        return false;
    if (!(_XP->SetDiagFolding(  DiagFolding )))
        return false;
    return Slice(isAverageMode);
}
bool  UtsusemiSqeCalc2::
Slice(bool isAverageMode){
    std::vector<Double> ax1range = _XP->PutAxRange(0);
    std::vector<Double> ax2range = _XP->PutAxRange(1);
    std::vector<Double> ax3range = _XP->PutAxRange(2);
    std::vector<Double> ax4range = _XP->PutAxRange(3);
    std::vector<std::string> axType = _XP->PutAxType();
    std::vector<Double> Folding = _XP->PutFolding();
    std::vector<UInt4> DFold = _XP->PutDiagFolding();
    for (UInt4 i=0; i<DFold.size(); i++) Folding.push_back( Double(DFold[i]) );
    return Slice( ax1range,ax2range,ax3range,ax4range,axType, Folding, isAverageMode );
}
bool  UtsusemiSqeCalc2::
_ConvPyListToDoubleVect( PyObject *ax, std::vector<double> *ret, UInt4 s ){
    UInt4 size = (UInt4) PyList_Size( ax );
    for (UInt4 i=s; i<size; i++){
        if( PyFloat_CheckExact( PyList_GetItem( ax, i ) ) ){
            ret->push_back( PyFloat_AsDouble( PyList_GetItem( ax, i ) ) );
        }
#ifdef IS_PY3
        else if( PyLong_CheckExact( PyList_GetItem( ax, i ) ) ){
            ret->push_back( ( Double )( PyLong_AsLong( PyList_GetItem( ax, i ) ) ) );
        }
#else
        else if( PyInt_CheckExact( PyList_GetItem( ax, i ) ) ){
            ret->push_back( ( Double )( PyInt_AsLong( PyList_GetItem( ax, i ) ) ) );
        }
#endif
        else{
            return false;
        }
    }
    return true;
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::PutInnerMatrix( std::string M){
    gsl_matrix* T;
    bool isFound=true;
    if (M=="L") T=L;
    else if (M=="U") T=U;
    else if (M=="R") T=R;
    else if (M=="V") T=V;
    else if (M=="A") T=A;
    else isFound=false;

    std::vector<Double> ret(9,0.0);
    if (isFound){
        ret[0] = gsl_matrix_get( T, 0, 0 );
        ret[1] = gsl_matrix_get( T, 0, 1 );
        ret[2] = gsl_matrix_get( T, 0, 2 );
        ret[3] = gsl_matrix_get( T, 1, 0 );
        ret[4] = gsl_matrix_get( T, 1, 1 );
        ret[5] = gsl_matrix_get( T, 1, 2 );
        ret[6] = gsl_matrix_get( T, 2, 0 );
        ret[7] = gsl_matrix_get( T, 2, 1 );
        ret[8] = gsl_matrix_get( T, 2, 2 );
    }else{
        UtsusemiError(MessageTag+"PutInnerMatrix : Argument is invalid ( L, U, R, V or A )");
    }
    return ret;
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::ConvToUBMatrix( PyObject* LC, PyObject* UV, PyObject* VV, bool flagRound ){
    UtsusemiSqeCalcXtalParams* _xp = new UtsusemiSqeCalcXtalParams();
    std::vector<Double> ret;
    if (_xp->SetLatticeConstants( LC )){
        if (_xp->SetUVvector( UV, VV )){
            if (_MakeUBIMatrix( _xp->PutLatticeConsts(), _xp->PutUVector(), _xp->PutVVector() )){
                gsl_matrix* ub = gsl_matrix_alloc(3,3);
                _CalcInverseMatrix( UBI, ub);
                ret = _ConvMatrixToVect( ub );
                gsl_matrix_free(ub);
            }else{
                UtsusemiError(MessageTag+"ConvToUBMatrix : Failed to make UB matrix");
            }
        }else{
            UtsusemiError(MessageTag+"ConvToUBMatrix : Uvect or Vvect Argument is invalid");
        }
    }else{
        UtsusemiError(MessageTag+"ConvToUBMatrix : Lattice Constants Argument is invalid");
    }
    delete _xp;
    if (flagRound){
        for (UInt4 i=0; i<ret.size(); i++)
            if (fabs(ret[i])<DBL_EPSILON) ret[i]=0.0;
    }
    return ret;
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiSqeCalc2::ConvToUVvector( PyObject* LC, PyObject* UB, bool flagRound ){
    UtsusemiSqeCalcXtalParams* _xp = new UtsusemiSqeCalcXtalParams();
    std::vector<Double> ret;
    if ((_xp->SetLatticeConstants( LC ))&&(_xp->SetUBMatrix( UB ))){
        gsl_matrix* l = gsl_matrix_alloc(3,3);
        _CalcReciprocalLatticeMatrix( _xp->PutLatticeConsts(), l );

        std::vector<Double> ubi_v=InverseMatrix(_xp->PutUBMatrix());
        gsl_matrix* ubi = gsl_matrix_alloc(3,3);
        for (UInt4 i=0; i<3; i++)
            for (UInt4 j=0; j<3; j++)
                gsl_matrix_set(ubi,i,j,ubi_v[ i*3 + j ]);

        gsl_matrix* uv = gsl_matrix_alloc(3,3);
        gsl_blas_dgemm( CblasNoTrans, CblasNoTrans, 1.0, l, ubi, 0.0, uv ); // 1.0*[A][UB-1] -> uv

        //     0  1  2
        // 0 | v1 w1 u1 |           |u1|        |v1|
        // 1 | v2 w2 u2 |  => uview=|u2| ,vview=|v2|
        // 2 | v3 w3 u3 |           |u3|        |v3|
        gsl_matrix_view uview = gsl_matrix_submatrix( uv, 0,2,3,1 );
        gsl_matrix_view vview = gsl_matrix_submatrix( uv, 0,0,3,1 );

        gsl_matrix* li = gsl_matrix_alloc(3,3);
        _CalcInverseMatrix( l, li );

        //        | Uh |
        // Umat = | Uk |
        //        | Ul |
        gsl_matrix* Umat = gsl_matrix_calloc(3,1);
        gsl_matrix* Vmat = gsl_matrix_calloc(3,1);

        // | a00 a01 a02 | | u1 |   | Uh |
        // | a10 a11 a12 |x| u2 | = | Uk |
        // | a20 a21 a22 | | u3 |   | Ul |
        gsl_blas_dgemm( CblasNoTrans, CblasNoTrans, 1.0, li, &uview.matrix, 0.0, Umat );
        gsl_blas_dgemm( CblasNoTrans, CblasNoTrans, 1.0, li, &vview.matrix, 0.0, Vmat );

        double Uh = gsl_matrix_get(Umat,0,0);
        double Uk = gsl_matrix_get(Umat,1,0);
        double Ul = gsl_matrix_get(Umat,2,0);
        double Vh = gsl_matrix_get(Vmat,0,0);
        double Vk = gsl_matrix_get(Vmat,1,0);
        double Vl = gsl_matrix_get(Vmat,2,0);
        double Ulen=sqrt( Uh*Uh + Uk*Uk + Ul*Ul );
        double Vlen=sqrt( Vh*Vh + Vk*Vk + Vl*Vl );

        ret.push_back( Uh/Ulen );
        ret.push_back( Uk/Ulen );
        ret.push_back( Ul/Ulen );
        ret.push_back( Vh/Vlen );
        ret.push_back( Vk/Vlen );
        ret.push_back( Vl/Vlen );

        if (flagRound){
            for (UInt4 i=0; i<6; i++)
                if (fabs(ret[i])<DBL_EPSILON) ret[i]=0.0;
            double U_max=0.0;
            double V_max=0.0;
            for (UInt4 i=0; i<3;i++){
                if (U_max<ret[i]) U_max=ret[i];
                if (V_max<ret[3+i]) V_max=ret[3+i];
            }
            if (U_max!=0.0)
                for (UInt4 i=0; i<3;i++) ret[i] = ret[i]/U_max;
            if (V_max!=0.0)
                for (UInt4 i=3; i<6;i++) ret[i] = ret[i]/V_max;
        }

        gsl_matrix_free(l);
        gsl_matrix_free(li);
        gsl_matrix_free(ubi);
        gsl_matrix_free(uv);
        gsl_matrix_free(Umat);
        gsl_matrix_free(Vmat);

    }
    delete _xp;
    return ret;
}
