#include "DetectorInfoEditorBase.hh"
//////////////////////////////////////////////////////////
DetectorInfoTagDefineBase::
DetectorInfoTagDefineBase(){
}
//////////////////////////////////////////////////////////
DetectorInfoTagDefineBase::
~DetectorInfoTagDefineBase(){
}
//////////////////////////////////////////////////////////
const std::string DetectorInfoTagDefineBase::DINFO="/detectorInfo";
const std::string DetectorInfoTagDefineBase::DINFO_AT_INST="inst";
const std::string DetectorInfoTagDefineBase::DINFO_AT_VER="version";
const std::string DetectorInfoTagDefineBase::DINFO_AT_DATE="update";

const std::string DetectorInfoTagDefineBase::DINFO_INSTRINFO="instrumentInfo";
const std::string DetectorInfoTagDefineBase::DINFO_INSTRINFO_L1=UTSUSEMI_KEY_HEAD_L1;
const std::string DetectorInfoTagDefineBase::DINFO_INSTRINFO_SAMPLEPOSI="samplePosition";
const std::string DetectorInfoTagDefineBase::DINFO_INSTRINFO_TYPICALL2=UTSUSEMI_KEY_HEAD_TYPICALL2;
const std::string DetectorInfoTagDefineBase::DINFO_INSTRINFO_TYPICALDS=UTSUSEMI_KEY_HEAD_TYPICALDS;

const std::string DetectorInfoTagDefineBase::DINFO_TFP="tfp";
const std::string DetectorInfoTagDefineBase::DINFO_TFP_PARAM="param";
const std::string DetectorInfoTagDefineBase::DINFO_TFP_PARAM_AT_PIXID="pixId";

const std::string DetectorInfoTagDefineBase::DINFO_TFPCALCPARAMS="tfpCalcParams";
const std::string DetectorInfoTagDefineBase::DINFO_TFPCALCPARAMS_PSD="psd";
const std::string DetectorInfoTagDefineBase::DINFO_TFPCALCPARAMS_PSD_REF="reference";
const std::string DetectorInfoTagDefineBase::DINFO_TFPCALCPARAMS_PSD_REF_L2="L2";
const std::string DetectorInfoTagDefineBase::DINFO_TFPCALCPARAMS_PSD_REF_THETA="theta";

const std::string DetectorInfoTagDefineBase::DINFO_POSIINFO="positionInfo";
const std::string DetectorInfoTagDefineBase::DINFO_POSIINFO_POSI="position";
const std::string DetectorInfoTagDefineBase::DINFO_POSIINFO_POSI_AT_DETID="detId";
const std::string DetectorInfoTagDefineBase::DINFO_POSIINFO_POSI_AT_NUMAXIS="numAxis";

const std::string DetectorInfoTagDefineBase::DINFO_BANKINFO="bankInfo";
const std::string DetectorInfoTagDefineBase::DINFO_BANKINFO_BANK="bank";
const std::string DetectorInfoTagDefineBase::DINFO_BANKINFO_BANK_AT_ID="bankId";
const std::string DetectorInfoTagDefineBase::DINFO_BANKINFO_BANK_AT_NAME="name";

//////////////////////////////////////////////////////////
UtsusemiDetectorInfoInstInfo::
UtsusemiDetectorInfoInstInfo(){
    L1 = 0.0;
    SamplePosition.resize(3,0.0);
    TypicalL2 = 0.0;
    TypicalDS = 0.0;
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoInstInfo::
UtsusemiDetectorInfoInstInfo(const UtsusemiDetectorInfoInstInfo& obj ){
    L1 = obj.L1;
    SamplePosition.resize(3,0.0);
    copy( obj.SamplePosition.begin(),obj.SamplePosition.end(),SamplePosition.begin() );
    TypicalL2 = obj.TypicalL2;
    TypicalDS = obj.TypicalDS;
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoInstInfo::
~UtsusemiDetectorInfoInstInfo(){
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoTfp::
UtsusemiDetectorInfoTfp(){
    ClearList();
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoTfp::
UtsusemiDetectorInfoTfp( const UtsusemiDetectorInfoTfp& obj ){
    TfpList.clear();
    std::pair<Double,Double> init_p;
    init_p.first=0.0;
    init_p.second=0.0;
    TfpList.resize( obj.TfpList.size(), init_p );
    for (UInt4 i=0; i<TfpList.size(); i++){
        TfpList[i].first = obj.TfpList[i].first;
        TfpList[i].second = obj.TfpList[i].second;
    }
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoTfp::
~UtsusemiDetectorInfoTfp(){
}
//////////////////////////////////////////////////////////
void UtsusemiDetectorInfoTfp::
AddTfp(UInt4 pixelId, Double A, Double B){

    if ( (pixelId+1)>TfpList.size() ) {
        std::pair<Double,Double> init_p;
        init_p.first=0.0;
        init_p.second=0.0;
        TfpList.resize( (pixelId+1), init_p );
    }
    TfpList[pixelId].first = A;
    TfpList[pixelId].second= B;
}
//////////////////////////////////////////////////////////
void UtsusemiDetectorInfoTfp::
ClearList(){
    TfpList.clear();
}

//////////////////////////////////////////////////////////
UtsusemiDetectorInfoTfpCalcParams::
UtsusemiDetectorInfoTfpCalcParams(){
    Psd_ref_L2 = 0.0;
    Psd_ref_Theta = 0.0;
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoTfpCalcParams::
UtsusemiDetectorInfoTfpCalcParams( const UtsusemiDetectorInfoTfpCalcParams& obj){
    Psd_ref_L2 = obj.Psd_ref_L2;
    Psd_ref_Theta = obj.Psd_ref_Theta;
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoTfpCalcParams::
~UtsusemiDetectorInfoTfpCalcParams(){
}

//////////////////////////////////////////////////////////
UtsusemiDetectorInfoPosiInfo::
UtsusemiDetectorInfoPosiInfo(){
    numAxis_list.clear();
    position_list.clear();
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoPosiInfo::
UtsusemiDetectorInfoPosiInfo( const UtsusemiDetectorInfoPosiInfo& obj ){
    numAxis_list.resize( obj.numAxis_list.size(), 0 );
    copy( obj.numAxis_list.begin(), obj.numAxis_list.end(), numAxis_list.begin() );

    position_list.resize(obj.position_list.size());
    for (UInt4 i=0; i<obj.position_list.size(); i++){
        position_list[i].resize( obj.position_list[i].size(), 0.0 );
        copy( obj.position_list[i].begin(), obj.position_list[i].end(), position_list[i].begin() );
    }
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoPosiInfo::
~UtsusemiDetectorInfoPosiInfo(){
}
//////////////////////////////////////////////////////////
void UtsusemiDetectorInfoPosiInfo::
Clear(){
    numAxis_list.clear();
    position_list.clear();
}
//////////////////////////////////////////////////////////
void UtsusemiDetectorInfoPosiInfo::
AddPosi(UInt4 detId, UInt4 numAxis, std::vector<Double> posi ){
    if ( (detId+1)>numAxis_list.size() ){
        numAxis_list.resize( (detId+1), 0 );
        position_list.resize( (detId+1) );
    }
    numAxis_list[ detId ] = numAxis;
    position_list[ detId ] = posi;
}
//////////////////////////////////////////////////////////
bool UtsusemiDetectorInfoPosiInfo::
DelPosi(UInt4 detId, bool isForced){
    bool ret = false;
    if ( detId<( position_list.size() ) ){
        position_list[detId].clear();
        numAxis_list[detId]=0;
        while(true){
            if ((position_list.back().empty())&&(numAxis_list.back()==0)){
                position_list.pop_back();
                numAxis_list.pop_back();
            }
        }
        ret = true;
    }

    if (isForced) return true;
    return ret;
}
//////////////////////////////////////////////////////////
UInt4 UtsusemiDetectorInfoPosiInfo::
PutPosiNumAxis(UInt4 detId){
    if ( (detId+1)>( numAxis_list.size() ) ){
        return 0;
    }else{
        return numAxis_list[detId];
    }
}
//////////////////////////////////////////////////////////
std::vector<Double> UtsusemiDetectorInfoPosiInfo::
PutPosi(UInt4 detId){
  if ( (detId+1)>( position_list.size() ) ){
        std::vector<Double> empty;
        return empty;
    }else{
        return position_list[detId];
    }
}
//////////////////////////////////////////////////////////
std::vector<UInt4> UtsusemiDetectorInfoPosiInfo::
PutDetIdList(){
    std::vector<UInt4> ret;
    for (UInt4 detId=0;detId<(position_list.size());detId++)
        if (!(position_list[detId].empty()))
            ret.push_back(detId);
    return ret;
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoBankInfo::
UtsusemiDetectorInfoBankInfo(){
    bankNameList.clear();
    bankConts.clear();
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoBankInfo::
UtsusemiDetectorInfoBankInfo( const UtsusemiDetectorInfoBankInfo& obj ){
    bankNameList.resize( obj.bankNameList.size(), "" );
    bankConts.resize( obj.bankConts.size(), "" );
    for (UInt4 i=0; i<(bankNameList.size()); i++){
        bankNameList[i] = obj.bankNameList[i];
        bankConts[i] = obj.bankConts[i];
    }
}
//////////////////////////////////////////////////////////
UtsusemiDetectorInfoBankInfo::
~UtsusemiDetectorInfoBankInfo(){
}
//////////////////////////////////////////////////////////
void UtsusemiDetectorInfoBankInfo::
AddBank( UInt4 bankId, std::string name, std::string conts ){
    if ( (bankId+1)>bankNameList.size() ){
        bankNameList.resize( (bankId+1), "" );
        bankConts.resize( (bankId+1), "" );
    }
    bankNameList[ bankId ] = name;
    bankConts[ bankId ] = conts;
}
//////////////////////////////////////////////////////////
bool UtsusemiDetectorInfoBankInfo::
DelBank( UInt4 bankId ){
    if ( bankId<(bankNameList.size() ) ){
        bankNameList[bankId]="";
        bankConts[bankId]="";
        if ((bankNameList.back().empty())&&(bankConts.back().empty())){
            bankNameList.pop_back();
            bankConts.pop_back();
        }
        return true;
    }
    return false;
}
//////////////////////////////////////////////////////////
std::vector<std::string> UtsusemiDetectorInfoBankInfo::
PutBank( UInt4 bankId ){
    std::vector<std::string> ret;
    if ( bankId<(bankNameList.size() ) ){
        ret.push_back( bankNameList[bankId] );
        std::string conts = bankConts[bankId];
    StringTools _st;
        std::vector<std::string> conts_v = _st.SplitString(conts,",");
        for (std::vector<std::string>::iterator it=conts_v.begin(); it!=conts_v.end(); ++it)
            ret.push_back(*it);
    }
    return ret;
}




//////////////////////////////////////////////////////////
DetectorInfoEditorBase::
DetectorInfoEditorBase()
{
    Initialize();
}
//////////////////////////////////////////////////////////
DetectorInfoEditorBase::
DetectorInfoEditorBase( std::string dfile, bool workAsReader){
    Initialize();
    if ( Read(dfile) ){
        if (workAsReader)
            SetInfoAsReader();
        _Status=true;
    }else{
        _Status=false;
    }
}
//////////////////////////////////////////////////////////
DetectorInfoEditorBase::
~DetectorInfoEditorBase()
{
    Clear(0);
    delete _parser;
    delete _st;
    ClearReader(0);
}

const std::string DetectorInfoEditorBase::KEY_READ_XML="KEYREADXML";
const std::string DetectorInfoEditorBase::KEY_WRITE_XML="KEYWRITEXML";

const Int4 DetectorInfoEditorBase::DUMP_ALL=-1;
const Int4 DetectorInfoEditorBase::DUMP_INSTINFO=1;
const Int4 DetectorInfoEditorBase::DUMP_TFPINFO=2;
const Int4 DetectorInfoEditorBase::DUMP_TFPCALCPARAMS=3;
const Int4 DetectorInfoEditorBase::DUMP_POSIINFO=4;
const Int4 DetectorInfoEditorBase::DUMP_BANKINFO=5;

//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
Initialize(){
    _parser = new BoostXmlParser();
    _parser->SetQuiet( !(UtsusemiEnvGetDebugMode()) );
    _st = new StringTools();
    _MessageTag = "DetectorInfoEditorBase::";

    InstInfo = NULL;
    TfpInfo = NULL;
    TfpCalcParams = NULL;
    PosiInfo = NULL;
    BankInfo = NULL;

    InstCode = "";
    Version = "1.5";
    Update = "";

    _Status = false;
    _isReadXml = false;
    _detectorFilePath = "";
    _TimeFocParam = NULL;
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
Clear( UInt4 index ){
    if ((index==0)||(index==1)){
        if (InstInfo!=NULL){
            delete InstInfo;
        }
        InstInfo=NULL;
    }
    if ((index==0)||(index==2)){
        if (TfpInfo!=NULL){
            delete TfpInfo;
        }
        TfpInfo=NULL;
    }
    if ((index==0)||(index==3)){
        if (TfpCalcParams!=NULL){
            delete TfpCalcParams;
        }
        TfpCalcParams=NULL;
    }
    if ((index==0)||(index==4)){
        if (PosiInfo!=NULL){
            delete PosiInfo;
        }
        PosiInfo=NULL;
    }
    if ((index==0)||(index==5)){
        if (BankInfo!=NULL){
            delete BankInfo;
        }
        BankInfo=NULL;
    }
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
Read( std::string arg ){
    _Status = false;
    _detectorFilePath = "";
    _parser->Clear();
    if (_parser->Load( KEY_READ_XML, arg )){

        InstCode = _parser->PutContent( KEY_READ_XML, TAG.DINFO, TAG.DINFO_AT_INST );
        Version = _parser->PutContent( KEY_READ_XML, TAG.DINFO, TAG.DINFO_AT_VER );
        Update = _parser->PutContent( KEY_READ_XML, TAG.DINFO, TAG.DINFO_AT_DATE );

        ImportInstInfo();
        if (_Status){
            ImportTfpInfo();
            if (_Status){
                ImportTfpCalcParams();
                if (_Status){
                    ImportPosiInfo();
                    if (_Status){
                        ImportBankInfo();
                        if (!_Status){
                            UtsusemiError(_MessageTag+"::Read >> failed to import BankInfo ");
                        }
                    }else{
                        UtsusemiError(_MessageTag+"::Read >> failed to import PosiInfo ");
                    }
                }else{
                    UtsusemiError(_MessageTag+"::Read >> failed to import TfpCalcParams ");
                }
            }else{
                UtsusemiError(_MessageTag+"::Read >> failed to import TfpInfo ");
            }
        }else{
            UtsusemiError(_MessageTag+"::Read >> failed to import InstInfo ");
        }
    }else{
        //UtsusemiError( _MessageTag+"::Read >> failed to open file = "+filepath );
    }
    _isReadXml = _Status;
    if (!(_parser->isLoadXmlString())){
        if (_isReadXml)
            _detectorFilePath = arg;
        else
            UtsusemiError( _MessageTag+"::Read >> failed to open file = "+arg );
    }else
        if (!_isReadXml)
            UtsusemiError( _MessageTag+"::Read >> failed to read the given string.");

    return _Status;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
Write( std::string filepath ){
    if (_makeOutputXml()){
        _parser->Save( KEY_WRITE_XML, filepath );
    }else{
        UtsusemiError( "Write : false to make output XML from given information. " );
        return false;
    }
    return true;
}
//////////////////////////////////////////////////////////
std::string DetectorInfoEditorBase::
OutXml( bool withIndent ){
    if (_makeOutputXml()){
        return _parser->OutToString( KEY_WRITE_XML, "", withIndent );
    }else{
        UtsusemiError( "OutXml : false to make output XML from given information. " );
        return "";
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
ImportInstInfo(){
    _Status = false;
    if (_parser->hasPath( KEY_READ_XML,TAG.DINFO )){
        std::string instinfo_path = TAG.DINFO+"/"+TAG.DINFO_INSTRINFO;
        if (_parser->hasPath( KEY_READ_XML,instinfo_path )){
            Clear(1);
            InstInfo = new UtsusemiDetectorInfoInstInfo();
            std::string l1_path = instinfo_path + "/" + TAG.DINFO_INSTRINFO_L1;
            if (_parser->hasPath( KEY_READ_XML,l1_path ))
                InstInfo->L1 = _st->StringToDouble( _parser->PutContent( KEY_READ_XML,l1_path, "" ) );

            std::string sposi = instinfo_path + "/" + TAG.DINFO_INSTRINFO_SAMPLEPOSI;
            if (_parser->hasPath( KEY_READ_XML,sposi )){
                std::string tmp = _parser->PutContent( KEY_READ_XML,sposi, "" );
                std::vector<std::string> tmp_v =_st->SplitString( tmp, "," );
                if (tmp_v.size()!=3){
                    UtsusemiError( _MessageTag+"ImportInstInfo >> invalid sample position : "+tmp );
                    _Status = false;
                }else{
                    InstInfo->SamplePosition[0] = _st->StringToDouble( tmp_v[0] );
                    InstInfo->SamplePosition[1] = _st->StringToDouble( tmp_v[1] );
                    InstInfo->SamplePosition[2] = _st->StringToDouble( tmp_v[2] );
                }
            }
            std::string l2_path = instinfo_path + "/" + TAG.DINFO_INSTRINFO_TYPICALL2;
            if (_parser->hasPath( KEY_READ_XML,l2_path ))
                InstInfo->TypicalL2 = _st->StringToDouble( _parser->PutContent( KEY_READ_XML,l2_path, "" ) );
            std::string dS_path = instinfo_path + "/" + TAG.DINFO_INSTRINFO_TYPICALDS;
            if (_parser->hasPath( KEY_READ_XML,dS_path ))
                InstInfo->TypicalDS = _st->StringToDouble( _parser->PutContent( KEY_READ_XML,dS_path, "" ) );
            if (((InstInfo->L1)>0.0)&&((InstInfo->TypicalL2)>0.0)&&((InstInfo->TypicalDS)>0.0))
                _Status = true;
        }
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
_makeInstInfoXml(){
    // Make InstInfo
    if (InstInfo!=NULL){
        std::string l1_path=TAG.DINFO+"/"+TAG.DINFO_INSTRINFO+"/"+TAG.DINFO_INSTRINFO_L1;
        _parser->AddElement( KEY_WRITE_XML, l1_path, _st->DoubleToString( InstInfo->L1 ) );

        std::string cont = "";
        cont += _st->DoubleToString( InstInfo->SamplePosition[0] );
        cont += ",";
        cont += _st->DoubleToString( InstInfo->SamplePosition[1] );
        cont += ",";
        cont += _st->DoubleToString( InstInfo->SamplePosition[2] );
        std::string ss_path=TAG.DINFO+"/"+TAG.DINFO_INSTRINFO+"/"+TAG.DINFO_INSTRINFO_SAMPLEPOSI;
        _parser->AddElement( KEY_WRITE_XML, ss_path, cont );

        std::string l2_path = TAG.DINFO+"/"+TAG.DINFO_INSTRINFO + "/" + TAG.DINFO_INSTRINFO_TYPICALL2;
        _parser->AddElement( KEY_WRITE_XML, l2_path, _st->DoubleToString( InstInfo->TypicalL2 ) );
        std::string dS_path = TAG.DINFO+"/"+TAG.DINFO_INSTRINFO + "/" + TAG.DINFO_INSTRINFO_TYPICALDS;
        _parser->AddElement( KEY_WRITE_XML, dS_path, _st->DoubleToString( InstInfo->TypicalDS ) );
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
ImportTfpInfo(){
    _Status = false;
    if (_parser->hasPath( KEY_READ_XML,TAG.DINFO )){
        std::string tfp_path = TAG.DINFO+"/"+TAG.DINFO_TFP;
        if (_parser->hasPath( KEY_READ_XML,tfp_path )){
            Clear(2);
            TfpInfo = new UtsusemiDetectorInfoTfp();
            UInt4 num_tfp = _st->StringToUInt4( _parser->PutContent( KEY_READ_XML, tfp_path, "n" ) );
            std::string param_path = tfp_path+"/"+TAG.DINFO_TFP_PARAM;
            std::vector<std::string> param_list = _parser->PutAttValList( KEY_READ_XML, param_path, TAG.DINFO_TFP_PARAM_AT_PIXID );
            for (std::vector<std::string>::iterator it=param_list.begin(); it!=param_list.end(); ++it){
                std::string a_param_cont_path = param_path+","+TAG.DINFO_TFP_PARAM_AT_PIXID+"="+(*it);
                UInt4 pixId = _st->StringToUInt4( (*it) );
                std::string a_param_cont = _parser->PutContent( KEY_READ_XML, a_param_cont_path, "" );
                std::vector<std::string> param_list = _st->SplitString( a_param_cont, "," );
                if (param_list.size()==2){
                    TfpInfo->AddTfp( pixId, _st->StringToDouble( param_list[0] ), _st->StringToDouble( param_list[1] ) );
                }
            }
            _Status = true;
        }else{
            _Status = true;
        }
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
_makeTfpInfoXml(){
    // Make Time Focusing
    if (TfpInfo!=NULL){
        std::vector<std::string> att_p, val_p;
        std::string tfpinfo_path=TAG.DINFO+"/"+TAG.DINFO_TFP+"/"+TAG.DINFO_TFP_PARAM;
        UInt4 num_pix=0;
        for (UInt4 pix=0; pix<(*TfpInfo).TfpList.size(); pix++){
            if ( ( (*TfpInfo).TfpList[pix].first==0.0 )&&( (*TfpInfo).TfpList[pix].second==0.0 ) ){
            }else{
                std::string tfp_param_path = tfpinfo_path+","+TAG.DINFO_TFP_PARAM_AT_PIXID+"="+_st->UInt4ToString( pix );
                std::string conts = _st->DoubleToString( (*TfpInfo).TfpList[pix].first )+","+_st->DoubleToString( (*TfpInfo).TfpList[pix].first );
                _parser->AddElement( KEY_WRITE_XML, tfpinfo_path, conts );
                num_pix++;

                att_p.clear();
                val_p.clear();
                att_p.push_back( "i" );
                val_p.push_back( _st->UInt4ToString( num_pix ) );
                _parser->AddElement( KEY_WRITE_XML, tfp_param_path, att_p, val_p );
            }
        }

        att_p.clear();
        val_p.clear();
        att_p.push_back( "n" );
        val_p.push_back( _st->UInt4ToString( num_pix ) );
        _parser->AddElement( KEY_WRITE_XML, tfpinfo_path, att_p, val_p );
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
ImportTfpCalcParams(){
    _Status = false;
    if (_parser->hasPath( KEY_READ_XML,TAG.DINFO )){
        std::string tfp_path = TAG.DINFO+"/"+TAG.DINFO_TFPCALCPARAMS;
        if (_parser->hasPath( KEY_READ_XML,tfp_path )){
            Clear(3);
            TfpCalcParams = new UtsusemiDetectorInfoTfpCalcParams();
            std::string psd_ref_path = tfp_path+"/"+TAG.DINFO_TFPCALCPARAMS_PSD+"/"+TAG.DINFO_TFPCALCPARAMS_PSD_REF;
            if (_parser->hasPath( KEY_READ_XML,psd_ref_path )){
                std::string L2_path = psd_ref_path+"/"+TAG.DINFO_TFPCALCPARAMS_PSD_REF_L2;
                std::string th_path = psd_ref_path+"/"+TAG.DINFO_TFPCALCPARAMS_PSD_REF_THETA;
                TfpCalcParams->Psd_ref_L2 = _st->StringToDouble( _parser->PutContent( KEY_READ_XML, L2_path, "" ) );
                TfpCalcParams->Psd_ref_Theta = _st->StringToDouble( _parser->PutContent( KEY_READ_XML, th_path, "" ) );
                _Status = true;
            }
            if (_Status){
            }else{
                Clear(3);
                _Status = true;
            }
        }else{
            _Status = true;
        }
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
_makeTfpCalcParamsXml(){
    // Make Time Focusing Calculation Parameters
    if (TfpCalcParams!=NULL){
        std::string tfpcalc_path = TAG.DINFO+"/"+TAG.DINFO_TFPCALCPARAMS+"/"+TAG.DINFO_TFPCALCPARAMS_PSD+"/"+TAG.DINFO_TFPCALCPARAMS_PSD_REF;
        std::string l2_path = tfpcalc_path + "/" + TAG.DINFO_TFPCALCPARAMS_PSD_REF_L2;
        std::string th_path = tfpcalc_path + "/" + TAG.DINFO_TFPCALCPARAMS_PSD_REF_THETA;
        _parser->AddElement( KEY_WRITE_XML, l2_path, _st->DoubleToString( TfpCalcParams->Psd_ref_L2 ) );
        _parser->AddElement( KEY_WRITE_XML, th_path, _st->DoubleToString( TfpCalcParams->Psd_ref_Theta ) );
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
ImportPosiInfo(){
    _Status = false;
    if (_parser->hasPath( KEY_READ_XML,TAG.DINFO )){
        std::string posiInfo_path = TAG.DINFO+"/"+TAG.DINFO_POSIINFO;
        if (_parser->hasPath( KEY_READ_XML,posiInfo_path )){
            Clear(4);
            PosiInfo = new UtsusemiDetectorInfoPosiInfo();
            std::string posi_path = posiInfo_path+"/"+TAG.DINFO_POSIINFO_POSI;
            std::vector<std::string> posi_list = _parser->PutAttValList( KEY_READ_XML, posi_path, TAG.DINFO_POSIINFO_POSI_AT_DETID );
            for (std::vector<std::string>::iterator it=posi_list.begin(); it!=posi_list.end(); ++it){
                std::string a_posi_path = posi_path+","+TAG.DINFO_POSIINFO_POSI_AT_DETID+"="+(*it);
                if (_parser->hasPath( a_posi_path )){
                    UInt4 detId = _st->StringToUInt4( (*it) );
                    UInt4 numAx = _st->StringToUInt4( _parser->PutContent( KEY_READ_XML, a_posi_path, TAG.DINFO_POSIINFO_POSI_AT_NUMAXIS ) );
                    std::vector<std::string> posi_cont = _st->SplitString( _parser->PutContent( KEY_READ_XML, a_posi_path, "" ), "," );
                    std::vector<double> tmp( posi_cont.size(), 0.0 );
                    for (UInt4 i=0; i<posi_cont.size(); i++)
                        tmp[i] = _st->StringToDouble( posi_cont[i] );
                    PosiInfo->AddPosi( detId, numAx, tmp );
                    _Status = true;
                }
            }
            if (_Status){
            }else Clear(4);
        }
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
_makePosiInfoXml(){
    // Make Position Info
    if (PosiInfo!=NULL){
        std::vector<std::string> att_p, val_p;
        std::string posiinfo_path=TAG.DINFO+"/"+TAG.DINFO_POSIINFO;
        UInt4 num_posi = 0;
        for (UInt4 detId=0; detId< (*PosiInfo).position_list.size(); detId++){
            if ( (*PosiInfo).position_list[detId].empty() ){
            }else{
                std::string posi_path = posiinfo_path+"/"+TAG.DINFO_POSIINFO_POSI+",detId="+_st->UInt4ToString(detId);
                std::string conts = "";
                for (UInt4 i=0; i< ((*PosiInfo).position_list[detId].size()-1); i++)
                    conts += _st->DoubleToString( (*PosiInfo).position_list[detId][i] )+",";
                conts += _st->DoubleToString( (*PosiInfo).position_list[detId].back() );
                _parser->AddElement( KEY_WRITE_XML, posi_path, conts );

                att_p.clear();
                val_p.clear();
                att_p.push_back( "i" );
                val_p.push_back( _st->UInt4ToString( num_posi ) );
                att_p.push_back( "numAxis" );
                val_p.push_back( _st->UInt4ToString( (*PosiInfo).numAxis_list[detId] ) );
                _parser->AddElement( KEY_WRITE_XML, posi_path, att_p, val_p );
                num_posi++;
            }
        }
        att_p.clear();
        val_p.clear();
        att_p.push_back( "n" );
        val_p.push_back( _st->UInt4ToString( num_posi ) );
        _parser->AddElement( KEY_WRITE_XML, posiinfo_path, att_p, val_p );
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
ImportBankInfo(){
    _Status = false;
    if (_parser->hasPath( KEY_READ_XML,TAG.DINFO )){
        std::string bankInfo_path = TAG.DINFO+"/"+TAG.DINFO_BANKINFO;
        if (_parser->hasPath( KEY_READ_XML,bankInfo_path )){
            Clear(5);
            BankInfo = new UtsusemiDetectorInfoBankInfo();
            std::string bank_path = bankInfo_path+"/"+TAG.DINFO_BANKINFO_BANK;
            std::vector<std::string> bank_list = _parser->PutAttValList( KEY_READ_XML, bank_path, TAG.DINFO_BANKINFO_BANK_AT_ID );
            for (std::vector<std::string>::iterator it=bank_list.begin(); it!=bank_list.end(); ++it){
                std::string a_bank_path = bank_path+","+TAG.DINFO_BANKINFO_BANK_AT_ID+"="+(*it);
                if (_parser->hasPath( a_bank_path )){
                    UInt4 bankId = _st->StringToUInt4( (*it) );
                    std::string bankName = _parser->PutContent( KEY_READ_XML, a_bank_path, TAG.DINFO_BANKINFO_BANK_AT_NAME );
                    std::string bankCont = _parser->PutContent( KEY_READ_XML, a_bank_path, "" );
                    BankInfo->AddBank( bankId, bankName, bankCont );
                    _Status = true;
                }
            }
            if (_Status){
            }else Clear(4);
        }
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
_makeBankInfoXml(){
    // Make Bank Info
    if (BankInfo!=NULL){
        std::vector<std::string> att_p, val_p;
        std::string bankinfo_path=TAG.DINFO+"/"+TAG.DINFO_BANKINFO;
        UInt4 num_bank = 0;
        for (UInt4 bId=0; bId< (*BankInfo).bankNameList.size(); bId++){
            if ( (*BankInfo).bankConts[bId].empty() ){
            }else{
                std::string bank_path=bankinfo_path+"/"+TAG.DINFO_BANKINFO_BANK+","+TAG.DINFO_BANKINFO_BANK_AT_ID+"="+_st->UInt4ToString( bId );
                _parser->AddElement( KEY_WRITE_XML, bank_path, (*BankInfo).bankConts[bId] );
                att_p.clear();
                val_p.clear();
                att_p.push_back( "i" );
                val_p.push_back( _st->UInt4ToString( num_bank ) );
                att_p.push_back( TAG.DINFO_BANKINFO_BANK_AT_NAME );
                val_p.push_back( (*BankInfo).bankNameList[bId] );
                _parser->AddElement( KEY_WRITE_XML, bank_path, att_p, val_p );
                num_bank++;
            }
        }
        att_p.clear();
        val_p.clear();
        att_p.push_back( "n" );
        val_p.push_back( _st->UInt4ToString( num_bank ) );
        _parser->AddElement( KEY_WRITE_XML, bankinfo_path, att_p, val_p );
    }
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
Dump( Int4 index ){
    if ((index==DUMP_ALL)||(index==DUMP_INSTINFO)){
        std::cout << "======== InstrumentInfo ========" << std::endl;
        if (InstInfo==NULL) std::cout << "  Not Imported "<<std::endl;
        else{
            std::cout << "--- L1 = " << InstInfo->L1 << std::endl;
        }
    }

    if  ((index==DUMP_ALL)||(index==DUMP_TFPINFO)){
        std::cout << "==== Time Focusing Parameter ====" << std::endl;
        if (TfpInfo==NULL) std::cout << " Not Imported " << std::endl;
        else{
            for (UInt4 i=0; i<(*TfpInfo).TfpList.size(); i++){
                if (((*TfpInfo).TfpList[i].first==0.0)&&(((*TfpInfo).TfpList[i].second==0.0))){
                    std::cout << "pixId= " << i << " : not imported " << std::endl;
                }else{
                    std::cout << "pixId= " << i << " : " << (*TfpInfo).TfpList[i].first;
                    std::cout << ", " << (*TfpInfo).TfpList[i].second << std::endl;
                }
            }
        }
    }
    if  ((index==DUMP_ALL)||(index==DUMP_TFPCALCPARAMS)){
        std::cout << "===== Time Focusing Calculation parameters ====" << std::endl;
        if (TfpCalcParams==NULL) std::cout << " Not Imported " << std::endl;
        else{
            std::cout << "  PSD/ Reference L2    = " << TfpCalcParams->Psd_ref_L2 << std::endl;
            std::cout << "  PSD/ Reference Theta = " << TfpCalcParams->Psd_ref_Theta << std::endl;
        }
    }

    if  ((index==DUMP_ALL)||(index==DUMP_POSIINFO)){
        std::cout << "===== Position Info ====" << std::endl;
        if (PosiInfo==NULL) std::cout << " Not Imported " << std::endl;
        else{
            for (UInt4 i=0; i<(*PosiInfo).numAxis_list.size() ; i++){
                if ( (*PosiInfo).position_list[i].empty() ){
                }else{
                    std::cout << "DetId=" << i << " : ";
                    for (UInt4 j=0; j<( (*PosiInfo).position_list[i].size()-1); j++)
                        std::cout << (*PosiInfo).position_list[i][j] << ", ";
                    std::cout << (*PosiInfo).position_list[i].back() << std::endl;
                }
            }
        }
    }
    if  ((index==DUMP_ALL)||(index==DUMP_BANKINFO)){
        std::cout << "===== Bank Info ====" << std::endl;
        if (BankInfo==NULL) std::cout << " Not Imported " << std::endl;
        else{
            for (UInt4 bId=0; bId<(*BankInfo).bankNameList.size(); bId++){
                if ( (BankInfo->bankConts[bId])!="" ){
                    std::cout << "BankId = " << bId << ": ";
                    std::cout << "BankName=" << (*BankInfo).bankNameList[bId] << ": ";
                    std::cout << "Conts=" << (*BankInfo).bankConts[bId] << std::endl;
                }
            }
        }
    }
}






//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
_makeOutputXml(){
    _parser->Clear(KEY_WRITE_XML, true);
    if (!(_parser->CreateNewTree( KEY_WRITE_XML ))) {
        UtsusemiError( _MessageTag+"_makeOutputXml >> False to make new tree." );
        return false;
    }

    _makeInstInfoXml();
    _makeTfpInfoXml();
    _makeTfpCalcParamsXml();
    _makePosiInfoXml();
    _makeBankInfoXml();

    // Add Instrument and version of DetectorInfo
    std::vector<std::string> att_p, val_p;
    att_p.clear();
    val_p.clear();
    att_p.push_back( TAG.DINFO_AT_INST );
    val_p.push_back( InstCode );
    att_p.push_back( TAG.DINFO_AT_VER );
    val_p.push_back( Version );
    att_p.push_back( TAG.DINFO_AT_DATE );
    val_p.push_back( Update );
    _parser->AddElement( KEY_WRITE_XML, TAG.DINFO, att_p, val_p );

    return true;
}
//////////////////////////////////////////////////////////
UInt4 DetectorInfoEditorBase::
CheckMultiContInfo( std::string strAttr ){
    //transform( strAttr.cbegin(), strAttr.cend(), strAttr.begin(), toupper);
    if ( (strAttr.find("All")!=std::string::npos) || (strAttr.find("ALL")!=std::string::npos) || (strAttr.find("all")!=std::string::npos) ){
        return 1;
    }
    else if (strAttr.find("-")!=std::string::npos){
        return 2;
    }
    else{
        return 0;
    }
}

//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetInstInfoL1( Double _L1 ){
    if (InstInfo==NULL) InstInfo = new UtsusemiDetectorInfoInstInfo();
    InstInfo->L1 = _L1;
    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetInstInfoSamplePosition( Double _px, Double _py, Double _pz ){
    if (InstInfo==NULL) InstInfo = new UtsusemiDetectorInfoInstInfo();
    InstInfo->SamplePosition[0] = _px;
    InstInfo->SamplePosition[1] = _py;
    InstInfo->SamplePosition[2] = _pz;
    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetInstInfoTypicalL2( Double _L2 ){
    if (InstInfo==NULL) InstInfo = new UtsusemiDetectorInfoInstInfo();
    if (_L2>0.0) InstInfo->TypicalL2 = _L2;
    else return false;
    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetInstInfoTypicalDS( Double _dS ){
    if (InstInfo==NULL) InstInfo = new UtsusemiDetectorInfoInstInfo();
    if (_dS>0.0) InstInfo->TypicalDS = _dS;
    else return false;

    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetTfpInfo( UInt4 pixId, Double A, Double B ){
    if (TfpInfo==NULL) TfpInfo = new UtsusemiDetectorInfoTfp();
    if ( (pixId+1)>( (*TfpInfo).TfpList.size() ) ){
        std::pair<Double,Double> tmp;
        tmp.first = 0.0;
        tmp.second= 0.0;
        (*TfpInfo).TfpList.resize( (pixId+1), tmp );
    }
    (*TfpInfo).TfpList[pixId].first = A;
    (*TfpInfo).TfpList[pixId].second = B;

    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetTfpInfo( std::vector<UInt4> pixIdList, std::vector<Double> AList, std::vector<Double> BList ){
    if ( ( (pixIdList.size())==(AList.size()) )&&( (AList.size())==(BList.size()) ) ){
        for (UInt4 i=0; i<pixIdList.size(); i++)
            SetTfpInfo( pixIdList[i], AList[i], BList[i] );
    }else{
        return false;
    }
    return true;
}
//////////////////////////////////////////////////////////
std::vector<Double> DetectorInfoEditorBase::
PutTfpInfo( UInt4 pixId ){
    std::vector<Double> ret;
    if ( (TfpInfo==NULL)||((pixId+1)>(TfpInfo->TfpList.size())) ){
    }else{
        ret.push_back( TfpInfo->TfpList[pixId].first );
        ret.push_back( TfpInfo->TfpList[pixId].second );
    }
    return ret;
}

//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
DeleteTfpInfo( UInt4 pixId, bool isForced ){
    if (TfpInfo==NULL){
        if (isForced) {
            return true;
        }else{
            return false;
        }
    }
    if ( (pixId+1)>( (*TfpInfo).TfpList.size() ) ){
        if (isForced) {
            return true;
        }else{
            return false;
        }
    }
    (*TfpInfo).TfpList[pixId].first = 0.0;
    (*TfpInfo).TfpList[pixId].second = 0.0;

    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
DeleteTfpInfo( std::vector<UInt4> pixIdList, bool isForced ){
    for (UInt4 i=0; i<pixIdList.size(); i++)
        if (DeleteTfpInfo( pixIdList[i], isForced )){
        }else{
            UtsusemiError( _MessageTag+"DeleteTfpInfo >> No such pixId "+_st->UInt4ToString( pixIdList[i] ) );
            return false;
        }
    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetTfpCalcParamsPsdRef( Double psd_ref_l2, Double psd_ref_theta ){
    if (TfpCalcParams==NULL) TfpCalcParams = new UtsusemiDetectorInfoTfpCalcParams();
    TfpCalcParams->Psd_ref_L2 = psd_ref_l2;
    TfpCalcParams->Psd_ref_Theta = psd_ref_theta;
    return true;
}

//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetPositionInfo( UInt4 detId, UInt4 numAxis, std::vector<Double> params ){
    if (PosiInfo==NULL) PosiInfo = new UtsusemiDetectorInfoPosiInfo();
    PosiInfo->AddPosi( detId, numAxis, params );
    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
DeletePositionInfo( UInt4 detId, bool isForced ){
    if (PosiInfo!=NULL){
        return PosiInfo->DelPosi(detId, isForced);
    }else{
        if (isForced) return true;
        return false;
    }
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
DeletePositionInfo( std::vector<UInt4> detIdList, bool isForced ){
    if (PosiInfo==NULL){
        if (isForced){
            return true;
        }else{
            return false;
        }
    }
    for (UInt4 i=0; i<detIdList.size(); i++){
        if (DeletePositionInfo( detIdList[i], isForced )){
        }else{
            UtsusemiError( _MessageTag+"DeletePositionInfo >> No such detId ="+_st->UInt4ToString( detIdList[i] ) );
            return false;
        }
    }
    return true;

}
//////////////////////////////////////////////////////////
UInt4 DetectorInfoEditorBase::
PutPositionInfoNumAxis( UInt4 detId ){
  if (PosiInfo!=NULL){
    return PosiInfo->PutPosiNumAxis(detId);
  }
  return 0;
}
//////////////////////////////////////////////////////////
std::vector<Double> DetectorInfoEditorBase::
PutPositionInfoParams( UInt4 detId ){
    if (PosiInfo!=NULL){
        return PosiInfo->PutPosi(detId);
    }
    std::vector<Double> empty;
    return empty;
}
//////////////////////////////////////////////////////////
std::vector<UInt4> DetectorInfoEditorBase::
PutPositionInfoDetIdList(){
    if (PosiInfo!=NULL){
        return PosiInfo->PutDetIdList();
    }
    std::vector<UInt4> empty;
    return empty;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetBankInfo( UInt4 bankId, std::string bankName, std::string conts ){
    if (BankInfo==NULL) BankInfo = new UtsusemiDetectorInfoBankInfo();
    if ( (bankId+1)>( BankInfo->bankNameList.size() ) ){
        BankInfo->bankNameList.resize( (bankId+1), "" );
        BankInfo->bankConts.resize( (bankId+1), "" );
    }
    BankInfo->bankNameList[bankId] = bankName;
    BankInfo->bankConts[bankId] = conts;
    return true;
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
DeleteBankInfo( UInt4 bankId ){
  if (BankInfo!=NULL){
    return BankInfo->DelBank(bankId);
  }
  return false;
}
//////////////////////////////////////////////////////////
std::vector<std::string> DetectorInfoEditorBase::
PutBankInfo( UInt4 bankId ){
  if (BankInfo!=NULL){
      return BankInfo->PutBank(bankId);
  }
  std::vector<std::string> empty;
  return empty;
}
//////////////////////////////////////////////////////////
Double DetectorInfoEditorBase::
PutInstL1(){
    if (InstInfo==NULL) return -1.0;
    return InstInfo->L1;
}
//////////////////////////////////////////////////////////
std::vector<Double> DetectorInfoEditorBase::
PutInstSamplePosition(){
    std::vector<Double> ret;
    if (InstInfo==NULL) return ret;
    return InstInfo->SamplePosition;
}
//////////////////////////////////////////////////////////
Double DetectorInfoEditorBase::
PutInstTypicalL2(){
    if (InstInfo==NULL) return -1.0;
    Double ret = InstInfo->TypicalL2;
    if (ret==0.0){
        for (UInt4 detId=0; detId< (*PosiInfo).position_list.size(); detId++){
            if ( (*PosiInfo).position_list[detId].empty() ){
            }else{
                std::vector<Double> p=(*PosiInfo).position_list[detId];
                ret = sqrt(p[0]*p[0] + p[1]*p[1] + p[2]*p[2]);
                return ret;
            }
        }
    }
    return ret;
}
//////////////////////////////////////////////////////////
Double DetectorInfoEditorBase::
PutInstTypicalDS(){
    if (InstInfo==NULL) return -1.0;
    return InstInfo->TypicalDS;
}
//////////////////////////////////////////////////////////
std::vector<Int4> DetectorInfoEditorBase::
PutBankIdList(){
    std::vector<Int4> ret;
    if (BankInfo==NULL) return ret;
    for (Int4 bankid=0; bankid<(Int4)(BankInfo->bankConts.size()); bankid++){
        if (BankInfo->bankConts[ bankid ]!="")
            ret.push_back( bankid );
    }
    return ret;
}
//////////////////////////////////////////////////////////
std::vector<Int4> DetectorInfoEditorBase::
PutDetIdInBank( UInt4 bankId ){
    std::vector<Int4> ret;
    if (BankInfo==NULL) return ret;
    if (bankId<(BankInfo->bankConts.size())){
        std::string conts = BankInfo->bankConts[bankId];
        std::vector<std::string> cont_sp = _st->SplitString( conts, "," );
        for (std::vector<std::string>::iterator it=cont_sp.begin(); it!=cont_sp.end(); ++it){
            if ((*it).find("-")==std::string::npos)
                ret.push_back( _st->StringToInt4( (*it) ) );
            else{
                std::vector<std::string> rng = _st->SplitString( (*it),"-" );
                if (rng.size()==1) ret.push_back( _st->StringToInt4( (*it) ) ); // when value is minus
                else{
                    Int4 s_id = _st->StringToInt4( rng[0] );
                    Int4 e_id = _st->StringToInt4( rng[1] );
                    for (Int4 did=s_id; did<(e_id+1); did++){
                        ret.push_back( did );
                    }
                }
            }
        }
    }
    return ret;
}
//////////////////////////////////////////////////////////
std::vector<Int4> DetectorInfoEditorBase::
PutDetIdInBank( std::string bankName ){
    if (BankInfo!=NULL){
        for (UInt4 bankid=0; bankid<(BankInfo->bankConts.size()); bankid++){
            if (BankInfo->bankConts[bankid].find( bankName )>=0)
                return PutDetIdInBank( bankid );
        }
    }
    std::vector<Int4> empty_vec;
    return empty_vec;
}
//////////////////////////////////////////////////////////
std::vector<std::string> DetectorInfoEditorBase::
PutBankName(){
    if (BankInfo!=NULL)
        return BankInfo->bankNameList;
    std::vector<std::string> empty_vec;
    return empty_vec;
}
//////////////////////////////////////////////////////////
void DetectorInfoEditorBase::
ClearReader( UInt4 index ){
    if ((index==0)||(index==2)||(index==3)){
        if (_TimeFocParam!=NULL){
            delete _TimeFocParam;
        }
        _TimeFocParam=NULL;
    }
}
//////////////////////////////////////////////////////////
bool DetectorInfoEditorBase::
SetInfoAsReader(){
    ClearReader(0);
    if (TfpCalcParams!=NULL){
        _TimeFocParam = new std::vector< std::vector<Double> >(1);
        _TimeFocParam->at(0).push_back( InstInfo->L1 );
        _TimeFocParam->at(0).push_back( TfpCalcParams->Psd_ref_L2 );
        _TimeFocParam->at(0).push_back( TfpCalcParams->Psd_ref_Theta );
    }else if (TfpInfo!=NULL){
        _TimeFocParam = new std::vector< std::vector<Double> >(TfpInfo->TfpList.size());
        for (UInt4 i=0; i<(TfpInfo->TfpList.size()); i++){
            _TimeFocParam->at(i).push_back( (*TfpInfo).TfpList[i].first );
            _TimeFocParam->at(i).push_back( (*TfpInfo).TfpList[i].second );
        }
    }
    return true;
}
//////////////////////////////////////////////////////////
std::vector< std::vector<Double> >* DetectorInfoEditorBase::
PutTfp(){
    if (_TimeFocParam!=NULL)
        return _TimeFocParam;
    return NULL;
}
