#!/usr/bin/python3
# -*- coding: utf-8 -*-

import utsusemi.SAS.ana.Reduction.HistBaseCommands as HBC

# reserved words for return value in this.
DAT = None
ecs = None
EC = None
ret = None
PyObj = None
Iq = None
Tr = None

########################################
def GetHistogramSAS( runNo=0, Xrange="0.7 7.6 0.1 dL", Bank="SM", TimeSlice="-1 -1", frameBoundary=0.0, isTimeFoc=False, SolAngCor=True, paramFiles="- -" ):
    """
    Load EventData to produce histogram
    GetHistogramSAS( runNo=0, AxType="lambda", Xrange="0.7 7.7 0.05", Bank="SM", TimeSlice="-1 -1", frameBoundary=0.0, isTimeFoc=False )

    @param runNo     (int)     Run Number
    @param Xrange    (string)  "startX endX widthX type" separated with space (not comma) : type=[dL, dL/L, tof]
    @param Bank      (string)  "SM"...  Monitors by "MON1" or "MON2"
    @param TimeSlice (string)  "startTime endTime" separated with space: "-1 -1" means no-setting.
    @param frameBoundary (float)
    @param isTimeFoc (bool)    True or False
    @param SolAngCor  (bool)    True or False for Solid Angle Correction
    @param paramFiles (string) "WiringInfo DetectorInfo"
    @retval DAT (ElementContainerMatrix)
    """
    return HBC.GetHistogramSAS( runNo, Xrange, Bank, TimeSlice, str(frameBoundary), isTimeFoc, SolAngCor, paramFiles )

########################################
def GetMonHistSAS( runNo=0, MonNo=1, AxType="lambda", Xrange="0.7 7.6 0.1", MonIndex=0, TimeSlice="-1 -1", frameBoundary=0.0, CalibEffi=True ):
    """
    Load EventData to produce histogram of monitors
    GetMonHistSAS( runNo=0, MonNo=1, AxType="lambda", Xrange="0.7 7.7 0.05", MonIndex=0, TimeSlice="-1 -1", frameBoundary=0.0, CalibEffi=True )

    @param runNo     (int)     Run Number
    @param MonNo     (int)     Monitor Number 1 (up-stream) or 2 (down-stream)
    @param AxType    (string)  Type must be "tof', "lambda" or "lamda2"
    @param Xrange    (string)  "startX endX widthX" separated with space (not comma)
    @param MonIndex  (int)
    @param TimeSlice (string)  "startTime endTime" separated with space: "-1 -1" means no-setting.
    @param frameBoundary (float)
    @param CalibEffi (bool)    Monitor-detector efficiency calibration
    @retval EC (ElementContainer)
    """
    return HBC.GetMonHistSAS( runNo, MonNo, AxType, Xrange, MonIndex, TimeSlice, str(frameBoundary), CalibEffi )

########################################
def GetPulseHeightSAS( runNo=0, Xrange="0 5000 10", Bank="SM", TimeSlice="-1 -1", paramFiles="- -" ):
    """
    Make Pulse Height data of EventData
    GetPulseHeightSAS( runNo=0, Xrange="0 10000 10", Bank="SM", TimeSlice="-1 -1", paramFiles="- -" ):

    @param runNo     (int)     Run Number
    @param Xrange    (string)  "startX endX widthX type" separated with space (not comma) : type=[dL, dL/L, tof]
    @param Bank      (string)  "SM"...  Monitors by "MON1" or "MON2"
    @param TimeSlice (string)  "startTime endTime" separated with space: "-1 -1" means no-setting.
    @param paramFiles (string) "WiringInfo DetectorInfo"
    @retval ECM (ElementContainerMatrix)
    """
    frameBoundary="0.0"
    PHbin=5
    return HBC.GetPulseHeightSAS( runNo, Xrange, Bank,TimeSlice, frameBoundary, paramFiles, PHbin)

########################################
def GetMonHistByPhSAS( runNo=0, MonNo=1, Xrange="0.0 40000.0 10.0 tof", phRange="0 10000", MonIndex=0, TimeSlice="-1 -1", frameBoundary=0.0, CalibEffi=True ):
    """
    Load EventData to produce histogram of monitors limited by pulse height
    GetMonHistByPhSAS( runNo=0, MonNo=1, Xrange="0.0 40000.0 10.0 tof", phRange="0 10000", MonIndex=0, TimeSlice="-1 -1", frameBoundary=0.0, CalibEffi=True )

    @param runNo     (int)     Run Number
    @param MonNo     (int)     Monitor Number 1 (up-stream) or 2 (down-stream)
    @param Xrange    (string)  "startX endX widthX type" separated with space (not comma) : type=[dL, dL/L, tof]
    @param phRange   (string)  "lld hld"
    @param MonIndex  (int)     0.. digital  1.. analog
    @param TimeSlice (string)  "startTime endTime" separated with space: "-1 -1" means no-setting.
    @param frameBoundary (float)
    @param CalibEffi (bool)    Monitor-detector efficiency calibration
    @retval EC (ElementContainer)
    """
    return HBC.GetMonHistByPhSAS( runNo, MonNo, Xrange, phRange, MonIndex, TimeSlice, frameBoundary, CalibEffi )

########################################
# Dictionary for entry the name of functions
_functions = {
            "GetHistogramSAS":GetHistogramSAS,
            "GetMonHistSAS":GetMonHistSAS,
            "GetPulseHeightSAS":GetPulseHeightSAS,
            "GetMonHistByPhSAS":GetMonHistByPhSAS
            }
_functionsOrder = [
    "GetHistogramSAS",
    "GetMonHistSAS",
    "GetPulseHeightSAS",
    "GetMonHistByPhSAS"
    ]

