#!/usr/bin/python
# -*- coding: utf-8 -*-

import time
import os
import Manyo as mm
import Manyo.Utsusemi as mu
import math

import utsusemi.ana.Reduction.BaseCommands as BCM
import utsusemi.ana.Reduction.BaseCommandsInEla as BaseCom
import utsusemi.DNA.ana.Reduction.BaseCommandsDNA as BaseComDNA
import utsusemi.DNA.ana.Reduction.LS340ASMWLogReader as LS340ASMW
import utsusemi.DNA.ana.Reduction.LS350IROHA2LogReader as LS350IROHA2
import utsusemi.DNA.ana.Reduction.MakeRunListFromParamsIROHA2 as MakeRunListFromParamsIROHA2
# special reserved word for commands history
HISTORY = None

# reserved words for return value in this.
DAT = None
ecs = None
EC  = None
ret = None
ECA = None
#########################################
def CalcDATSumWithNormCurrent(ECM1=DAT,ECM2=DAT, ECM3=DAT, ECM4=DAT, ECM5=DAT, datanum=3):
    """
    @param ECM1	
    @param ECM2	
    @param ECM3	
    @param ECM4	
    @param ECM5	
    @retval ret_ecm        ElementContainerMatrix
    """
    ECMs = [ECM1,ECM2,ECM3,ECM4,ECM5]
    bc = 0.0
    import Com
    for ii in range(datanum):
        ecm = ECMs[ii]
        print (type(ecm),ii)
        if type(ecm) == mm.ElementContainerMatrix:
            if bc == 0.0:
                ret_ecm = ecm
            else:
                ret_ecm = Com.CalcContainers(ret_ecm,ecm,coef1=1.0,coef2=1.0,ope="+")
            bc += BCM.GetBeamCurrentFromECM(ecm,offset_sec=0.0,isCT8n=True,isTagCorrect=True)
    
    if (bc > 0.0):
        print ("Sum(bc) = ", bc)
        MutiplyConstant(ret_ecm,1/bc)

    return ret_ecm

#########################################
def GetDataOrg(runnumber=10,inform_xml="DI_daq0mod0to09.xml",daq=0):
    """
    Load EventData and convert to histogram.
    @param  runnumber  Run Number
    @param  inform_xml Detector Info xml file
    @param daq       DAQ CPU
    @retval DAT        ElementContainerMatrix
    """
    return BaseCom.GetDataOrg(runnumber,inform_xml,daq)

########################################
def CopyData(first_run=10,last_run=-1):
    """
    Copy EventData file from DAQ0 or DAQ1 into /data/XXX
    @param runno_from (int) First Run No of a measurement to be copied
    @param runno_to (int) Last Run No of a measurement to be copied
    @retval None
    """
    BaseCom.CopyData(first_run,last_run)

########################################
def MakeRunInfoList(StartRunNo=10053, EndRunNo=10057, outputfile="/home/dnaadmin/RunInfoList.txt"):
    """
    Make Run information list with protoncurrent, temperature and protron current.
    @param StartRunNo (int) Start Run No.
    @param EndRunNo (int) End Run No.
    @param outputfile (str) output file name.
    @retval None
    """
    MakeRunListFromParamsIROHA2.BCMakeRunList(int(StartRunNo),int(EndRunNo),outputfile,flag_proton=1, offset_sec=0.0, isCT8n=True, devTemp="LS350Loop1", devGonio="ComCryoRotStage")

##################################################################################

########################################
def GetHistogramHW(runNo=1956,HwParam="0.01/-1.00/1.00",LambdaParam="6.26/3.34",t0_offset=0.0,
                   useT0ModCorr=False,TimeParam="-1.0/-1.0",UseFastChopper=False,tofOffsetFile="",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @retval DAT     (ElementContainerMatrix)
    """
    t0_offset=str(t0_offset)
    return BaseComDNA.GetHistogramHW(runNo,HwParam,LambdaParam,t0_offset,useT0ModCorr,TimeParam,UseFastChopper,tofOffsetFile,isHistogram=isHistogram)

########################################
def GetHistogramHW333(runNo=1956,HwParam="0.01/-1.00/1.00",LambdaParam="6.26/3.34",t0_offset=0.0,
                      useT0ModCorr=False,TimeParam="-1.0/-1.0",UseFastChopper=False,tofOffsetFile="",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @retval DAT     (ElementContainerMatrix)
    """
    t0_offset=str(t0_offset)
    return BaseComDNA.GetHistogramHW333(runNo,HwParam,LambdaParam,t0_offset,useT0ModCorr,TimeParam,UseFastChopper,tofOffsetFile,isHistogram=isHistogram)

########################################
def GetHistogramHW311(runNo=1956,HwParam="0.001/-0.15/0.30",LambdaParam="4.173/4.15",t0_offset=6484.0,
                      useT0ModCorr=False,TimeParam="-1.0/-1.0",UseFastChopper=False,tofOffsetFile="",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @retval DAT     (ElementContainerMatrix)
    """
    t0_offset=str(t0_offset)
    return BaseComDNA.GetHistogramHW311(runNo,HwParam,LambdaParam,t0_offset,useT0ModCorr,TimeParam,UseFastChopper,tofOffsetFile,isHistogram=isHistogram)

########################################
def GetHistogramHWTrignet(runNo=1956,HwParam="0.01/-1.00/1.00",LambdaParam="6.26/3.34",offset="0.0/0.0",
                          useT0ModCorr=False,TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",tofOffsetFile="",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @retval DAT     (ElementContainerMatrix)
    """

    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=2
    elif slitWidth=="3":
        useTrignet=3
    else:
        useTrignet=1

    return BaseComDNA.GetHistogramHWTrignet(runNo,HwParam,LambdaParam,offset,useT0ModCorr,TimeParam,tofSource,useTrignet,ChopperFrequency,tofOffsetFile,isHistogram)

########################################
def GetHistogramHW333Trignet(runNo=1956,HwParam="0.01/-1.00/1.00",LambdaParam="6.26/3.34",offset="0.0/0.0",
                             useT0ModCorr=False,TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",tofOffsetFile="",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @retval DAT     (ElementContainerMatrix)
    """
    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=2
    elif slitWidth=="3":
        useTrignet=3
    else:
        useTrignet=1

    return BaseComDNA.GetHistogramHW333Trignet(runNo,HwParam,LambdaParam,offset,useT0ModCorr,TimeParam,tofSource,useTrignet,ChopperFrequency,tofOffsetFile,isHistogram)

########################################
def GetHistogramHWnoTrignet(runNo=1956,HwParam="0.01/-1.00/1.00",LambdaParam="6.26/3.34",offset="0.0/0.0",
                            useT0ModCorr=False,TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",tofOffsetFile="",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @retval DAT     (ElementContainerMatrix)
    """
    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=5
    elif slitWidth=="3":
        useTrignet=6
    else:
        useTrignet=4

    return BaseComDNA.GetHistogramHWnoTrignet(runNo,HwParam,LambdaParam,offset,useT0ModCorr,TimeParam,tofSource,useTrignet,ChopperFrequency,tofOffsetFile,isHistogram)

########################################
def GetHistogramHW333noTrignet(runNo=1956,HwParam="0.01/-1.00/1.00",LambdaParam="6.26/3.34",offset="0.0/0.0",
                               useT0ModCorr=False,TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",tofOffsetFile="",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @retval DAT     (ElementContainerMatrix)
    """
    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=5
    elif slitWidth=="3":
        useTrignet=6
    else:
        useTrignet=4

    return BaseComDNA.GetHistogramHW333noTrignet(runNo,HwParam,LambdaParam,offset,useT0ModCorr,TimeParam,tofSource,useTrignet,ChopperFrequency,tofOffsetFile,isHistogram)

########################################
def GetHistogram(runNo=1956,startTOF=0,endTOF=40000,binTOF=100,startPSD=-1,endPSD=-1,pixelNo=120):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    DAT=BaseComDNA.GetHistogram(runNo,0,startTOF,endTOF,binTOF,startPSD,endPSD,pixelNo)
    return DAT

########################################
def GetHistogramMon(runNo=1956, useEiConv=True, LambdaParam="6.321/3.000", t0_offset=0.0, background=0.0, useT0ModCorr=False, TimeParam="-1.0/-1.0", UseFastChopper=True,isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    t0_offset=str(t0_offset)
    DAT=BaseComDNA.GetHistogramMon(runNo,useEiConv,LambdaParam,t0_offset,background,useT0ModCorr,TimeParam,int(UseFastChopper)-1,0,isHistogram=isHistogram)
    return DAT

########################################
def GetHistogramMonAtSmp(runNo=1956, useEiConv=True, LambdaParam="6.321/3.000", t0_offset=0.0, background=0.0, useT0ModCorr=False, TimeParam="-1.0/-1.0", UseFastChopper=True,isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    t0_offset=str(t0_offset)
    DAT=BaseComDNA.GetHistogramMon(runNo,useEiConv,LambdaParam,t0_offset,background,useT0ModCorr,TimeParam,int(UseFastChopper)-1,0,isHistogram=isHistogram,monType="SMPMON")
    return DAT

########################################
def GetHistogramMonVan(runNo=1956, useEiConv=True, LambdaParam="6.321/3.000", t0_offset=0.0, background=0.0, useT0ModCorr=False, TimeParam="-1.0/-1.0", UseFastChopper=True,isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    t0_offset=str(t0_offset)
    DAT=BaseComDNA.GetHistogramMon(runNo,useEiConv,LambdaParam,t0_offset,background,useT0ModCorr,TimeParam,int(UseFastChopper)-1,0,isHistogram=isHistogram,monType="VANMON")
    return DAT

########################################
def GetHistogramMonTrignet(runNo=1956, LambdaParam="6.321/3.000",offset="0.0/0.0", background=0.0,useT0ModCorr=False, TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    useEiConv=False
    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=2
    elif slitWidth=="3":
        useTrignet=3
    else:
        useTrignet=1

    DAT=BaseComDNA.GetHistogramMon(runNo,useEiConv,LambdaParam,t0_offset,background,useT0ModCorr,TimeParam,True,useTrignet,tofSource,ChopperFrequency,isHistogram=isHistogram)
    return DAT

########################################
def GetHistogramMonnoTrignet(runNo=1956, LambdaParam="6.321/3.000", offset="0.0/0.0", background=0.0, useT0ModCorr=False, TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    useEiConv=False
    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=5
    elif slitWidth=="3":
        useTrignet=6
    else:
        useTrignet=4

    DAT=BaseComDNA.GetHistogramMon(runNo,useEiConv,LambdaParam,offset,background,useT0ModCorr,TimeParam,True,useTrignet,tofSource,ChopperFrequency,isHistogram=isHistogram)
    return DAT

########################################
def GetHistogramMonETrignet(runNo=1956, LambdaParam="6.321/3.000", offset="0.0/0.0", background=0.0,useT0ModCorr=False, TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    useEiConv=True
    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=2
    elif slitWidth=="3":
        useTrignet=3
    else:
        useTrignet=1
    DAT=BaseComDNA.GetHistogramMon(runNo,useEiConv,LambdaParam,offset,background,useT0ModCorr,TimeParam,True,useTrignet,tofSource,ChopperFrequency,isHistogram=isHistogram)
    return DAT

########################################
def GetHistogramMonEnoTrignet(runNo=1956, LambdaParam="6.321/3.000", offset="0.0/0.0", background=0.0, useT0ModCorr=False, TimeParam="-1.0/-1.0",tofSource="0.0/400.0",chopperInfo="-1/225.0",isHistogram=False):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,paraFlag,startTOF=-1,endTOF=-1,binTOF=-1,pixelNo=-1,startPSD=-1,endPSD=-1,envFile)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param pixelNo  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @retval DAT     (ElementContainerMatrix)
    """
    useEiConv=True
    if "," in chopperInfo:
        ci = chopperInfo.split(",")
    elif "/" in chopperInfo:
        ci = chopperInfo.split("/")
    else:
        ci = []

    if len(ci)>1:
        slitWidth=ci[0]
        ChopperFrequency=ci[1]
    else:
        slitWidth="-1"
        ChopperFrequency="225.0"

    if slitWidth=="1":
        useTrignet=5
    elif slitWidth=="3":
        useTrignet=6
    else:
        useTrignet=4

    DAT=BaseComDNA.GetHistogramMon(runNo,useEiConv,LambdaParam,offset,background,useT0ModCorr,TimeParam,True,useTrignet,tofSource,ChopperFrequency,isHistogram=isHistogram)
    return DAT

########################################

def CreateQEMap(dat=DAT,startQ=0.0,endQ=3.0,deltaQ=0.05):
    """
    Create Q-E map

    @param dat    (ElementContaienrMatrix)
    @param startQ (double) Q value of range
    @param endQ   (double) Q value of range
    @param deltaQ (double) binning of Q
    @retval ECA   (ElementContainerArray)
    """
    return BaseComDNA.CreateQEMap(dat,startQ,endQ,deltaQ)

########################################

def MonitorAbsorptionCorrection(ec=EC):
    """
    Abosorption Correction of N2 Monitor detector
    @param ec         (ElementContainer)
    @retval  EC     (ElementContainer)
    """
    return BaseComDNA.MonitorAbsorptionCorrection(ec)

##########################################

def OutputQEEC(runNo=197,dat=DAT):
    """
    OutputQE-ECM-ECA

    @param runNo    (int) Run Number
    @param dat    (ElementContaienrMatrix)
    @retval None
    """

    return BaseComDNA.OutputQEEC(runNo,dat)

##########################################

def OutputEC(ec=EC,tag="_"):
    """
    OutputEC

    @param ec    (ElementContainer)
    @param tag    (string)

    @retval None
    """

    return BaseComDNA.OutputEC(ec,tag)

##########################################

def OutputECA(eca=ECA,tag="_"):
    """
    OutputECA

    @param eca    (ElementContainerArray)
    @param tag    (string)

    @retval None
    """

    return BaseComDNA.OutputECA(eca,tag)

##########################################

def OutputECM(ecm=DAT,tag="_"):
    """
    OutputECM

    @param eca    (ElementContainerMatrix)
    @param tag    (string)

    @retval None
    """

    return BaseComDNA.OutputECM(ecm,tag)

##########################################

def OutputECAText(eca=ECA,tag="_"):
    """
    OutputECAText file

    @param eca    (ElementContainerArray)
    @param tag    (string)

    @retval None
    """

    return BaseComDNA.OutputECAText(eca,tag)
##########################################

def OutputECADAVEText(eca=ECA,tag="_"):
    """
    OutputECAText file as DAVE format

    @param eca    (ElementContainerArray)
    @param tag    (string)

    @retval None
    """

    return BaseComDNA.OutputECADAVEText(eca,tag)
##########################################

def OutputNeiECAText(eca=ECA,tag="_"):
    """
    Output ECA having non equal interval energy transfer as Text format

    @param eca    (ElementContainerArray)
    @param tag    (string)

    @retval None
    """

    return BaseComDNA.OutputNeiECAText(eca,tag)


##########################################

def OutputECText(ec=EC,tag="_"):
    """
    OutputECText file

    @param ec    (ElementContainer)
    @param tag    (string)

    @retval None
    """

    return BaseComDNA.OutputECText(ec,tag)

##########################################

def ReadEC(filename=".bst"):
    """
    ReadEC

    @param filename    (string)
    @retval EC    (ElementContaienr)
    """

    return BaseComDNA.ReadEC(filename)

##########################################

def ReadECA(filename=".bst"):
    """
    ReadECA

    @param filename    (string)
    @retval ECA    (ElementContaienrArray)
    """

    return BaseComDNA.ReadECA(filename)

##########################################

def ReadECM(filename=".bst"):
    """
    ReadECM

    @param filename    (string)
    @retval ECM    (ElementContaienrMatrix)
    """

    return BaseComDNA.ReadECM(filename)

##########################################

def SliceQE(dat=DAT,lowerQ=0.0,upperQ=1.0):
    """
    SliceQEMap

    @param dat    (ElementContaienrArray)
    @param lowerQ    (double) lowerQ
    @param upperQ    (double) upperQ
    @retval EC    (ElementContainer)
    """

    return BaseComDNA.SliceQE(dat,lowerQ,upperQ)

##########################################

def SliceQEall(dat=DAT,initialQ=0.0,finalQ=1.0,deltaQ=0.1):
    """
    SliceQEall

    @param dat    (ElementContaienrArray)
    @param initialQ    (double)
    @param finalQ    (double)
    @param deltaQ    (double)
    @retval ECA    (ElementContainerArray)
    """
    return BaseComDNA.SliceQEall(dat,initialQ,finalQ,deltaQ)

#########################################

def OutputPartOfEC(ec=EC,initial=-0.5,final=0.5):
    """
    OutputPartOfEC

    @param EC         (ElementContaienr)
    @param initial    (int)
    @param final      (int)
    @retval EC        (ElementContainer)
    """
    return BaseComDNA.OutputPartOfEC(ec,initial,final)

##########################################

def GetInformationOfECbin(ec=EC):
    """
    GetInformationOfECbin

    @param EC         (ElementContaienr)
    @retval None
    """
    return BaseComDNA.GetInformationOfECbin(ec)

##########################################

def ILambdaCorrDNA(dat=DAT, ec=EC, useMonEff=False):
    """
    I (Lambda) correction

    @param dat    (ElementContaienrMatrix)
    @param ec              (ElementContainer)
    @param use Monitor Efficienncy  (bool)
    @retval ECM   (ElementContainerMatrix)
    """

    return BaseComDNA.ILambdaCorrDNA(dat, ec, useMonEff)

##########################################
def ILambdaCorrUseMonAtSmpDNA(dat=DAT, ec=EC, monDatFile="monitor_data.txt", isHighReso=True, attenuateVal=0.0):
    """
    I (Lambda) correction by using data of the monitor at sample position for DNA

    @param dat    (ElementContaienrMatrix)
    @param ec     (ElementContainer)
    @param monDatFile (string) information file for composition of monitor
    @param isHighReso (bool) is high resolution or not
    @param attenuateVal (float) 0.0 -> use default value (4.088585)
    @retval ECM   (ElementContainerMatrix)
    """

    return BaseComDNA.ILambdaCorrUseMonAtSmpDNA(dat, ec, monDatFile, isHighReso, attenuateVal)

##########################################
def GetHistForILC(ec=EC,factor=0.000000000001):
    """
    Get EC for I (Lambda) correction(DNA)

    @param ec     (ElementContainer)
    @param factor     (double)
    @retval EC2     (ElementContainer)

    """
    return BaseComDNA.GetHistForILC(ec,factor)

##########################################

def SolidAngleCorrDNA(dat=DAT,useDetEff=False, useAbsoCorr=False,useEffCorr=False,sampletype="sample",sampleDataPath="test_sample_data.dat", DetEffDataPath=""):
    """
    I (Lambda) correction

    @param dat    (ElementContaienrMatrix)
    @param use Detector Efficiency  (bool)
    @param use SampleAbosorptionCorr(bool)
    @param use EfficiencyCorr       (bool)
    @param sampletype               (string) sample/cell
    @param sampleDataPath           (string)
    @retval ECM   (ElementContainerMatrix)
    """

    return BaseComDNA.SolidAngleCorrDNA(dat,useDetEff, useAbsoCorr,useEffCorr,sampletype,sampleDataPath,DetEffDataPath)

##########################################

def DoQEMask(dat,filename="mask.txt"):
    """
    MASKED flag controled by file.
    @param dat (ElementContainerArray)
    @param filename (string) filename of QEmask information 
    """

    BaseComDNA.DoQEMask(dat,filename)

#########################################
def PeakFitSimple(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,peak_back=10.0,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param peak_back          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval None
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple(ec,"g c",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,peak_back,xbin_start,xbin_end,useDomain,flug,True)

#########################################
def PeakFitGaussConst(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,peak_back=10.0,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param peak_back          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval None
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple(ec,"g c",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,peak_back,xbin_start,xbin_end,useDomain,flug,True)

#########################################
def PeakFitLorentzConst(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,peak_back=10.0,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param peak_back          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval None
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple(ec,"l c",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,peak_back,xbin_start,xbin_end,useDomain,flug,True)

#########################################
def PeakFitErrorFuncConst(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,peak_back=10.0,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param peak_back          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval None
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple(ec,"erf c",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,peak_back,xbin_start,xbin_end,useDomain,flug,True)

#########################################

def PeakFitTriangleConst(ec=EC,fit_boundary_ratio=0.8,hight=30.0,boundary=" ",background=1.0,xbin_start=10.0,xbin_end=60.0,useDomain=False):
    """
    PeakFitting script by using Levmar,Triangle AS Fit
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param hight              (float)
    @param boundary           (string)
    @param background         (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @param useDomain           (bool)
    """
    BaseComDNA.PeakFitTriangle(ec,fit_boundary_ratio,hight,boundary,background,xbin_start,xbin_end,useDomain)

#########################################

def PeakFitTrapezoidConst(ec=EC,fit_boundary_ratio=0.8,hight=30.0,boundary=" ",background=1.0,xbin_start=10.0,xbin_end=60.0,useDomain=False):
    """
    PeakFitting script by using Levmar,Triangle AS Fit
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param hight              (float)
    @param boundary           (string)
    @param background         (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @param useDomain           (bool)
    """
    BaseComDNA.PeakFitTrapezoid(ec,fit_boundary_ratio,hight,boundary,background,xbin_start,xbin_end,useDomain)

#########################################
def PeakFitSimple0(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval EC     (ElementContainer)
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple0(ec,"g",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,xbin_start,xbin_end,useDomain,flug)

#########################################
def PeakFitGauss(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval None
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple0(ec,"g",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,xbin_start,xbin_end,useDomain,flug)

#########################################
def PeakFitLorentz(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval None
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple0(ec,"l",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,xbin_start,xbin_end,useDomain,flug)

#########################################
def PeakFitErrorFunc(ec=EC,fit_boundary_ratio=0.8,peak_hight=9200.0,peak_position=2.1,peak_fwhm=0.4,xbin_start=1.7,xbin_end=3.0):
    """
    PeakFitting script by using Levmar, AS Fit, gauss+constant
    @param ec (ElementContainer)
    @param fit_boundary_ratio (float)
    @param peak_hight         (float)
    @param peak_position      (float)
    @param peak_fwhm          (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    @retval None
    """
    flug = True
    useDomain = True
    BaseComDNA.PeakFitSimple0(ec,"erf",fit_boundary_ratio,peak_hight,peak_position,peak_fwhm,xbin_start,xbin_end,useDomain,flug)


#########################################
def PeakSearchSimple(ec=EC,window_width=100,window_upper=80,lower_bin=0,upper_bin=4000):
    """
    PeakSearch script , AS PeakSearch
    @param ec (ElementContainer)
    @param window_width (int)
    @param window_upper (int)
    @param lower_bin (int)
    @param upper_bin (int)
    @return None
    """
    BaseComDNA.PeakSearchSimple(ec,window_width,window_upper,lower_bin,upper_bin)

#########################################
def PeakFitSimpleECA(eca=DAT,fit_boundary_ratio=0.8,xbin_start=0.0,xbin_end=1.0):
    """
    PeakFitting script by using Levmar, AS Fit
    @param eca (ElementContainerArray)
    @param fit_boundary_ratio (float)
    @param xbin_start         (float)
    @param xbin_end           (float)
    """
    useDomain = True
    BaseComDNA.PeakFitSimpleECA(eca,fit_boundary_ratio,xbin_start,xbin_end,useDomain)

#########################################

def PeakFit(ec=EC,function=" ",param=" ",param_ub=" ",param_lb=" ",fit_region=" ",useDomain=False,NumOfItteration=10000):
    """
    PeakFitting script by using Levmar, AS Fit
    @param ec (ElementContainer)
    @param function      (string)
    @param param         (string)
    @param param_lb      (string)
    @param param_ub      (string)
    @param useDomain       (bool)
    @param NumOfItteration (int)
    @retval EC     (ElementContainer)
    """
    plotflug = True
    BaseComDNA.PeakFit(ec,function,param,param_ub,param_lb,fit_region,useDomain,NumOfItteration,plotflug)

#########################################

def PeakFitNotUseWeight(ec=EC,function=" ",param=" ",param_ub=" ",param_lb=" ",fit_region=" ",useDomain=False,NumOfItteration=10000):
    """
    PeakFitting script by using Levmar, AS Fit
    @param ec (ElementContainer)
    @param function      (string)
    @param param         (string)
    @param param_lb      (string)
    @param param_ub      (string)
    @param useDomain       (bool)
    @param NumOfItteration (int)
    @retval EC     (ElementContainer)
    """
    plotflug = True
    BaseComDNA.PeakFitNotUseWeight(ec,function,param,param_ub,param_lb,fit_region,useDomain,NumOfItteration,plotflug)

##########################################
def NormByBeamWithParams(dat=DAT,runNo=627):
    """
    Normailze by Beam current from web and calculate summation of the number of shots and protons
    @param dat        (ElementContainer,-Array,-Matrix)
    @param runNo (int)
    """
    BaseComDNA.NormByBeamWithParams(dat,runNo)

#########################################
def MutiplyConstant(dat=DAT, factor=10000.0 ):
    """
    Multiplication
    @param dat        (ElementContainer,-Array,-Matrix)
    @param factor     (double) factor
    """
    BaseComDNA.MutiplyConstant(dat,factor)

##################################################################################
def GetHistogramDB(runNo=197,startTOF=-1,endTOF=-1,binTOF=-1,startPSD=-1,endPSD=-1,NumOfPixel=100, boundaryTof=0.0):
    """
    Load EventData and convert to histogram on Double Frame mode
    GetHistogram(runNo,paraFlag=0,startTOF=-1,endTOF=-1,binTOF=-1,startPSD=-1,endPSD=-1,
                  NumOfPixel=Def_NumOfPixel, boundaryTof=0.0)
    @param runNo    (int) Run Number
    @param startTOF (double) top TOF value to be selected
    @param endTOF   (double) end TOF value to be selected
    @param binTOF   (double) bin width of TOF
    @param startPSD (int) first PSD ID
    @param endPSD   (int) final PSD ID
    @param NumOfPixel  (int) the number of pixels on one PSD
    @param envFile  (string) Environ file for event data to histogram
    @param boundaryTof (double) TOF value of boundary between frames
    @retval DAT     (ElementContainerMatrix)
    """
    return BaseCom.GetHistogramDB(runNo,startTOF,endTOF,binTOF,startPSD,endPSD,NumOfPixel, boundaryTof)

#########################################
def GetMultiEi(runNo=0, expEi="0.0", PSD="75", PIXEL="52",NumOfPixel=100, Resolution=0.5,PlotFlag=0):
    """
    Get Incident Energy from raw event data
    @param runNo (int) run number
    @param expEi (string) Initial Ei or multi Ei with comma as separater
    @param PSD   (string) range of PSDs to be used for fitting
    @param PIXEL (string) range of Pixels to be used for fitting
    @param NumOfPixel (int) the number of pixels in one PSD
    @param Resolution (double) resolution [%] for tof binning
    @param PlotFlag (int) if 1, plot graph
    @retval None
    """
    BaseCom.GetEiFromEvent(runNo, expEi, PSD, PIXEL,NumOfPixel, Resolution,PlotFlag)

#########################################
def GetEi(dat, expEi=50.0, MonPsd=75,MonPixel=52, L1=18.03, T0shift=0.0, FittingType=0, Debug=0):
    """
    Get Incident Energy from Monitor Counter data
    @param  dat          (ElementContainerMatrix)
    @param  expEi        (Double)  expected incident energy
    @param  MonPsd       (int) PSD ID to be used as monitor
    @param  MonPixel     (int) Pixel No to be used as monitor
    @param  L1           (double) Distance between moderator to sample [m]
    @param  T0shift      (double) T0 offset [micro-sec]
    @param  FittingType  (UInt4)  Gaussian:0 
    @param  Debug        (UInt4)  OFF:0, ON:1      
    """ 

    BaseCom.GetEi(dat, expEi, MonPsd,MonPixel, L1, T0shift, FittingType, Debug)

#########################################
def GetHistogramOfEnergyTransfer(runNo=10,Ei=-1.0,dHW=-1.0,hw_min=-1.0,hw_max=-1.0,startTime=0.0,endTime=0.0):
    """
    Load EventData and convert to histogram of energy transfer
    GetHistogramOfEnergyTransfer(runNo,Ei,dHW,hw_min,hw_max,startTime,endTime)
    @param runNo    (int) Run Number
    @param Ei       (double) incident energy
    @param dHW      (double) delta of hw
    @param hw_min   (double) minimum of hw range
    @param hw_max   (double) maximum of hw range
    @param startTime (double)
    @param endTime   (double)
    @retval DAT      (ElementContainerMatrix)
    """
    return BaseCom.GetHistogramOfEnergyTransfer(runNo,Ei,dHW,hw_min,hw_max,startTime,endTime)

#########################################
def GetHistogramBydE(runNo=-1,Ei=0.0,Resolution=0.5,t0shift=0.0,L1=18.03,L2=2.5,NumOfPixel=100):
    """
    Load EventData and convert to histogram of the TOF region around Ei
    GetHistogramAroundEi(runNo,Ei=0.0,binTOF=-1,t0shift=0.0,L1=18.0,L2=2.5,paraFlag=1):
    @param runNo    (int) Run Number
    @param Ei       (double) Incident Energy
    @param Resolution   (double) Energy Resolution for tof binnings at Elastic Peak
    @param t0shift  (double) T0 offset [micro-sec]
    @param L1       (double)  Distance between moderator to sample [m]
    @param L2       (double)  between sample to PSD avarage [m]
    @retval DAT     ElementCotainerMatix as result
    """
    return BaseCom.GetHistogramBydE(runNo,Ei,Resolution,t0shift,L1,L2,NumOfPixel,0)

##########################################[inamura 170612]
def NormByBeamCurrent(dat=DAT,factor=1000.0,offset_sec=0.0, isCT8n=True ):
    """
    Normailze by Beam current from web and calculate summation of the number of shots and protons. Experimental info file, like SIK000XXX.xml file, must be placed in your ~/exp/xml directory.
    @param dat [Def:DAT] (ElementContainer,-Array,-Matrix)
    @param factor     (double) normalize factor data*factor/beamCurrent
    @param offset_sec (double) offset time
    @param isCT8n (bool) use CT8Neutron (corrected by muon target effect) or not
    """
    try:
        BaseCom.NormByBeamCurrent(dat, factor, offset_sec,isCT8n)
    except:
        msg = "Normalize by Proton Current is disable. Argument of Norm Factor must be 0 or minus"
        raise UserWarning(msg)
#def NormByBeamCurrent(dat=DAT,factor=1000.0 ):
#    """
#    Normailze by Beam current from web and calculate summation of the number of shots and protons. Experimental info file, like SIK000XXX.xml file, must be placed in your ~/exp/xml directory.
#    @param dat        (ElementContainer,-Array,-Matrix)
#    @param factor     (double) normalize factor data*factor/beamCurrent
#    """
#    BaseCom.NormByBeamCurrent(dat, factor)
    
#########################################
def TofToEnergyTransfer(dat=DAT,Ei=-1.0,L1=18.03,t0shift=0.0,deltaE=0.0,EminRate=0.4,EmaxRate=0.9):
    """
    Function to convert Time-of-Flight to Energy 
    
    @param dat      (ElementContainerMatrix)
    @param Ei       (double) Incident Energy [meV]: if -1, use Ei value in header
    @param L1       (double) Distance between moderator to sample [m]
    @param t0shift  (double) T0 shift [micro-sec]
    @param deltaE   (double) Delta Energy Transfer [meV]
    @param EminRate (double) range for minus E (<1.0)
    @param EmaxRate (double) range for plus E (<1.0)
    """
    BaseCom.TofToEnergyTransfer(dat,Ei,L1,t0shift,deltaE,EminRate,EmaxRate)

##########################################
def TofToEnergyTransferStrict(dat,Ei=-1.0,L1=18.03,t0shift=0.0,Emin=-10.0,Emax=10.0,deltaE=0.0):
    """
    Function to convert Time-of-Flight to Energy Transfer
    
    @param dat      (ElementContainerMatrix)
    @param Ei       (double) Incident Energy [meV]
    @param L1       (double) Distance between moderator to sample [m]
    @param t0shift  (double) T0 shift [micro-sec]
    @param Emin (double) minimum of E range
    @param Emax (double) maximum of E range
    @param deltaE   (double) Delta Energy Transfer [meV]
    """
    BaseCom.TofToEnergyTransferStrict(dat,Ei,L1,t0shift,Emin,Emax,deltaE)

##########################################
def TofToEnergy(dat=DAT,offset=0.0,L1=18.03,Emin=1.0,Emax=3000.0,deltaE=1.0):
    """
    Function to convert Time-of-Flight to Energy

    @param dat(ElementContaienrMatrix)
    @param offset  (double)
    @param L1      (double) [m]
    @param Emin    (double) [meV]
    @param Emax    (double) [meV]
    @param deltaE  (double) [meV]
    """
    BaseCom.TofToEnergy(dat,offset,L1,Emin,Emax,deltaE)

#########################################
def DoMask(dat=DAT,filename="maskTY.txt"):
    """
    MASKED flag controled by file.
    @param dat (ElementContainerMatrix)
    @param filename (string) filename of mask information 
    """
    BaseComDNA.DoMask(dat,filename)

#########################################
def SolidAngleCorrection(dat=DAT):
    """
    Correction for solid angle of pixel
    Each intensity is devided by solid angle * default value
    @param dat (ElementContainerMatrix)
    """

    BaseCom.SolidAngleCorrection(dat)

#########################################
def MakeWhiteVanCorr(dat, Emin=20.0, Emax=40.0):
    """
    Make data of White Vanadium correction

    @param dat (ElementContainerMatrix) target data 
    @retval None
    """
    BaseCom.MakeWhiteVanCorr(dat,Emin,Emax)

#########################################
def WhiteVanCorr(dat, runno=0 ):
    """
    White Vanadium correction

    @param dat (ElementContainerMatrix) target data 
    @param runno (int) RunNo for white vanadium
    @retval None
    """
    BaseCom.WhiteVanCorr( dat, runno )
    
#########################################
def KiKfCorrection(dat=DAT):
    """
    Calculate polar-angle and azim-angle of each Pixel from Pixel Position
    @param dat (ElementContainerMatrix)
    """
    BaseCom.KiKfCorrection(dat)

##########################################
def BoseFactorCorrection( dat=DAT,T=300.0):
    """
    Functon to do correction of Bose Factor

    @param dat (ElementContainerMatrix)
    @param T   (double)  [K]
    """
    BaseCom.BoseFactorCorrection( dat, T )

##########################################
def VisContOutputTextByPhi(dat, param_file="param.xml",phi=0.0,output_file="./"):
    """
    Output text data using VisualCont function

    @param dat (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param phi (float) angle phi
    @param output_file (string) Path to output file
    @return None
    """
    BaseCom.VisContOutputTextByPhi(dat, param_file,phi,output_file)

#########################################
def VisContOutputFileByPhi(dat, param_file="param.xml",phi=0.0,output_file="./", hasMuchMem=True):
    """
    Output data to a file using VisualContCommon function
    [inamura 110425]
    @param dat (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param phi (float) angle phi
    @param output_file (string) Path to output file
    @param hasMuchMem (Bool) if Ture, method to output occupies much memory.
    @return None
    """
    BaseCom.VisContOutputFileByPhi(dat, param_file,phi,output_file, hasMuchMem)

#########################################
def VisContMOutputFileByPhi(dat, param_file="param.xml",phi=0.0,output_file="./"):
    """
    Output data to a file using VisualContCommon function
    [inamura 110425]
    @param dat (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param phi (float) angle phi
    @param output_file (string) Path to output file
    @return None
    """
    BaseCom.VisContMOutputFileByPhi(dat, param_file,phi,output_file)

##########################################
def BoseFactCorrEC(dat=DAT,T=300.0):
    """
    Function to convert Time-of-Flight to Energy

    @param dat (ElementContaienr)
    @param T   (double) [K]
    @retval None
    """
    BaseCom.BoseFactCorr(dat,T)

##########################################
def ToPowder(dat=DAT,startQ=0.0,endQ=10.0,deltaQ=0.1):
    """
    Convert data from crystal to powder

    @param dat    (ElementContaienrMatrix)
    @param startQ (double) Q value of range
    @param endQ   (double) Q value of range
    @param deltaQ (double) binning of Q
    @retval ECM   (ElementContainerMatrix)
    """

    return BaseCom.ToPowder(dat,startQ,endQ,deltaQ)

#########################################
def FlexCalc(dat,Intensity="I",Error="Err"):
    """
    Flexible calculation of S(Q,hw) on ElementContainerMatrix.
    @param dat (ElementContainerMatrix)
    @param Intensity (string) the expression of Intensity calculation
    @param Error     (string) the expression of Error calculation
    @retval dat (ElementContainerMatrix)
    """

    return BaseCom.FlexCalc(dat,Intensity,Error)

#########################################
def GetProfile(dat=DAT, detID=-1):
    """
    Get Pulse Height Profile of PSD
    Each PSD pulse heignt profile is stored in header of ElementContaienrArray.
    @param  dat   (ElementContainerMatrix) Loaded ElementContainerMatrix.
    @param  detID (int) Detector ID (-1 : put all)
    @retval ECS    ElementContainer or list of ElementContainers of pulse height profile
    """
    return BaseCom.GetProfile(dat, detID)
    
##########################################[inamura 170612]
def NormByBeam(dat=DAT,start_day="2009/5/30", start_time="10:00:00", end_day="2009/5/30", end_time="11:00:00", factor=1.0, offset_sec=0.0, isCT8n=True ):
    """
    Normailze by Beam current from web and calculate summation of the number of shots and protons
    @param dat [Def:DAT] (ElementContainer,-Array,-Matrix)
    @param start_day  (string) the day of measurement start [yyyy/mm/dd]
    @param start_time (string) the time of measurement start [hh:mm:ss]
    @param end_day    (string) the day of measurement end [yyyy/mm/dd]
    @param end_time   (string) the time of measurement end [hh:mm:ss]
    @param factor     (float) normalize factor data*factor/beamCurrent
    @param offset_sec (float) offset second for Measurement Period to get protons
    @param isCT8n (bool) use CT8Neutron (corrected by muon target effect) or not
    @retval None
    """
    try:
        BaseCom.NormByBeam(dat,start_day,start_time,end_day,end_time,factor,offset_sec,isCT8n)
    except:
        msg = "Normalize by Proton Current is disable."
        raise UserWarning(msg)

#def NormByBeam(dat=DAT,start_day="2009/5/30", start_time="10:00:00", end_day="2009/5/30", end_time="11:00:00", factor=1.0 ):
#    """
#    Normailze by Beam current from web and calculate summation of the number of shots and protons
#    @param dat        (ElementContainer,-Array,-Matrix)
#    @param start_day  (string) the day of measurement start [yyyy/mm/dd]
#    @param start_time (string) the time of measurement start [hh:mm:ss]
#    @param end_day    (string) the day of measurement end [yyyy/mm/dd]
#    @param end_time   (string) the time of measurement end [hh:mm:ss]
#    @param factor     (double) normalize factor data*factor/beamCurrent
#    """
#    BaseCom.NormByBeam(dat,start_day,start_time,end_day,end_time,factor)
    
#########################################                            
def SumOfTOF(Target=DAT, PSD=1, tof_min=0, tof_max=40000):
    """
    Put summation of intensity along TOF
    @param  Target     ElementContainerMatrix
    @param  PSD        PSD for summation
    @param tof_min     Minimum point of TOF for summation
    @param tof_max     Maximum point of TOF for summation
    @retval EC         ElementContainer
    """
    return BaseCom.SumOfTOF(Target, PSD, tof_min, tof_max)


#########################################
def SumOfPixel(Target=DAT, PSD_min=1, PSD_max=1, Pixel_min=1, Pixel_max=1):
    """
    Put summation of intensity along Pixel
    @param Target     ElementContainerMatrix
    @param PSD_min    lower limit of PSD range
    @param PSD_max    upper limit of PSD range
    @param Pixel_min  lower limit of Pixel range
    @param Pixel_max  upper limit of Pixel range
    @retval EC
    """
    return BaseCom.SumOfPixel(Target, PSD_min, PSD_max, Pixel_min, Pixel_max)


#########################################
def PickUpRegion(Target=DAT, PSD_min=1, PSD_max=1, Pixel_min=1, Pixel_max=1):
    """
    Put summation of intensity along Pixel
    @param Target     ElementContainerMatrix
    @param PSD_min    lower limit of PSD range
    @param PSD_max    upper limit of PSD range
    @param Pixel_min  lower limit of Pixel range
    @param Pixel_max  upper limit of Pixel range
    @retval ECM       ElementContainerMatrix
    """
    return BaseCom.PickUpRegion(Target, PSD_min, PSD_max, Pixel_min, Pixel_max)

#########################################
def GetHistogram_Time(runNo=197,NumOfPixel=100,StartTime=0.0,EndTime=0.0):
    """
    Load EventData and convert to histogram.
    GetHistogram(runNo,NumOfPixel=100,StartTime=0.0,EndTime=0.0)
    @param runNo    (int) Run Number
    @param NumOfPixel  (int) the number of pixels on one PSD
    @param StartTime (double) [sec]
    @param EndTime   (double) [sec]
    @retval DAT     (ElementContainerMatrix)
    """
    return BaseCom.GetHistogram_Time(runNo,NumOfPixel,StartTime,EndTime)

#########################################
def GetHistogramBydE_Time(runNo=-1,Ei=0.0,Resolution=0.5,t0shift=0.0,L1=18.03,NumOfPixel=100,StartTime=0.0, EndTime=0.0):
    """
    Load EventData and convert to histogram of the TOF region around Ei
    GetHistogramBydE_Time(runNo,Ei=0.0,Resolution=0.5,t0shift=0.0,L1=18.0,
                          NumOfPixel=100,StartTime=0.0, EndTime=0.0)
    @param runNo    (int) Run Number
    @param Ei       (double) Incident Energy
    @param Resolution   (double) Energy Resolution for tof binnings at Elastic Peak [%]
    @param t0shift  (double) T0 offset [micro-sec]
    @param L1       (double)  Distance between moderator to sample [m]
    @param NumOfPixel (int) Number of pixels in one PSD
    @param StartTime (double) [sec]
    @param EndTime   (double) [sec]
    @retval DAT      (ElementContainerMatrix)
    """
    return BaseCom.GetHistogramBydE_Time(runNo,Ei,Resolution,t0shift,L1,NumOfPixel,StartTime, EndTime)

############################################
def SaveDataToManyoBinary(dat=DAT, path="./",filename="Sample.mdb"):
    """
    Save ElementContainerMatrix to manyo binary data file
    This is the binary format for storing ElementContainers using HDF5
    @param  dat [Def:DAT] (ElementContainer, -Array, -Matrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval None
    """
    BaseCom.SaveDataToManyoBinary(dat, path,filename)

############################################
def LoadDataFromManyoBinary(path="./",filename="Sample.mdb"):
    """
    Load ElementContainerMatrix from manyo binary data file
    which is the binary format for storing ElementContainers usign HDF5
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval data (ElementContaienr, -Array, -Matrix)
    """
    return BaseCom.LoadDataFromManyoBinary(path,filename)

#########################################
def SaveDataToNxspe(dat=DAT, path="./",filename="Sample",params="PSI:0.0,TEMP:300.0,USER:mlfuser"):
    """
    Save ElementContainerMatrix to NXSPE file.
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name without extension of .nxspe
    @param  params   (string) Other information : psi [deg], temperature[K] and user name
    @retval None
    """
    if dat.PutHeader().CheckKey(mu.UTSUSEMI_KEY_HEAD_EI) == 0:
        dat.PutHeaderPointer().Add(mu.UTSUSEMI_KEY_HEAD_EI, 0.0)
    BaseCom.SaveDataToNxspe(dat,path,filename,params)

#########################################
def LoadDataFromNxspe(path="./",filename="Sample.nxspe", num_of_pixels=120 ):
    """
    Make ElementContainerMatrix from nxspe.
    @param  path     (String) path to folder including spefile
    @param  spefile  (String) SPE file name (extention is .spe or .SPE)
    @param  num_of_pixels (Int) the number of pixels in a detector
    @retval DAT      (ElementContainerMatrix)
    """
    return BaseCom.LoadDataFromNxspe(path,filename, num_of_pixels)

#########################################
def SaveDataToSPE(dat=DAT, path="./",filename="Sample"):
    """
    Save ElementContainerMatrix to SPE and phx file.
    @param  dat      (ElementContainerMatrix)    
    @param  path     (String) path to data file
    @param  filename (String) data file name without extension (.phx and .spe)
    @retval None
    """
    return BaseCom.SaveDataToSPE(dat,path,filename)

#########################################
def SavePowderDataToText(dat=DAT,deltaQ=0.0, path="./",filename="Sample.txt"):
    """
    Save ElementContainerMatrix of powder to text file.
    @param  dat      (ElementContainerMatrix)
    @param  deltaQ   (double) bin width of Q
    @param  path     (String) path to data file
    @param  filename (String) text file name
    @retval None
    """
    BaseCom.SavePowderDataToText(dat,deltaQ, path,filename)

#########################################
def LoadDataFromDump(path="./",filename="Sample.dmp"):
    """
    Load ElementContainerMatrix from dump file made by ECMSerialize class
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval DAT        ElementContainer/Array/Matrix
    """

    return BaseCom.LoadDataFromDump(path,filename)


#########################################
def SaveDataToDump(dat=DAT, path="./",filename="Sample.dmp"):
    """
    Serialize ElementContainerMatrix to dump file using ECMSerialize class.
    @param  dat      (ElementContainerMatrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @retval None
    """

    BaseCom.SaveDataToDump(dat,path,filename)


#########################################
def GetDataFromNeXus(path="./",filename="Sample.nx"):
    """
    Load ElementContainer/Array/Matrix from NeXus file.
    @param  path     (String)   path to data file
    @param  filename (String)  data file name
    @retval DAT      ElementContainer/Array/Matrix
    """
    return BaseCom.GetDataFromNeXus(path,filename)


#########################################
def SaveDataToNeXus(dat=DAT, name="Data",path="./",filename="Sample.nx"):
    """
    Save ElementContainer/Array/Matrix to NeXus file.
    @param  dat       (ElementContainerMatrix)    
    @param  name      (String)   Title of saved data
    @param  path      (String)   path to data file
    @param  filename  (String)  data file name
    @retval None
    """
    BaseCom.SaveDataToNeXus(dat, name,path,filename)

#########################################
def SaveDataToDAVEgrp(dat=DAT,path="./",filename="Sample.grp",startQ=1.0,endQ=1.0,frcQ=1,binE=0.0):
    """
    Save grouped text (DAVE format) files of Q-sliced powder data
    @param  dat [Def:DAT] (ElementContainerMatrix)
    @param  path     (String) path to data file
    @param  filename (String) data file name
    @param  startQ   (double) minimum limit of Q-range
    @param  endQ     (double) maximum limit of Q-range
    @param  frcQ     (int) number of division
    @param  binE      (double) rebinning width of energy
    @retval None
    """
    BaseCom.SaveDataToDAVEgrp(dat,path,filename,startQ,endQ,frcQ,binE)

##########################################
def getBeamCurrent(start_day="2009/5/30", start_time="10:00:00", end_day="2009/5/30", end_time="11:00:00", offset_sec=0.0, isCT8n=True ):
    """
    get Beam current from web and calculate summation of the number of shots and protons
    @param start_day  (string) the day of measurement start [yyyy/mm/dd]
    @param start_time (string) the time of measurement start [hh:mm:ss]
    @param end_day    (string) the day of measurement end [yyyy/mm/dd]
    @param end_time   (string) the time of measurement end [hh:mm:ss]
    @param offset_sec (float) offset second for Measurement Period to get protons
    @param isCT8n (bool) use CT8Neutron ( corrected by muon target effect ) or not
    @retval beamCurrent        the number of protons [TelaProtons]
    """
    return BaseCom.getBeamCurrent(start_day, start_time, end_day, end_time, offset_sec, isCT8n)

#########################################
def GetHistogramAroundEi(runNo=-1,Ei=0.0,binTOF=1,t0shift=0.0,L1=18.03,L2=2.5,NumOfPixel=100):
    """
    Load EventData and convert to histogram of the TOF region around Ei
    GetHistogramAroundEi(runNo,Ei=0.0,binTOF=-1,t0shift=0.0,L1=18.0,L2=2.5,paraFlag=1):
    @param runNo    (int) Run Number
    @param Ei       (double) Incident Energy
    @param binTOF   (int) bin width of TOF
    @param t0shift  (double) T0 offset [micro-sec]
    @param L1       (double)  Distance between moderator to sample [m]
    @param L2       (double)  between sample to PSD avarage [m]
    @retval DAT     ElementCotainerMatix as result
    """
    return BaseCom.GetHistogramAroundEi(runNo,Ei,binTOF,t0shift,L1,L2,NumOfPixel,0)

#########################################
def GetHistogramBydE2(runNo=-1,Ei=0.0,Resolution=0.5,t0shift=0.0,hw_min=-10.0,hw_max=10.0,NumOfPixel=100):
    """
    Load EventData and convert to histogram of the TOF region around Ei
    GetHistogramBydE(runNo,Ei=0.0,binTOF=-1,t0shift=0.0,L1=18.0,L2=2.5,paraFlag=1):
    @param runNo    (int) Run Number
    @param Ei       (double) Incident Energy
    @param Resolution   (double) Energy Resolution for tof binnings at Elastic Peak [%]
    @param t0shift  (double) T0 offset [micro-sec]
    @param L1       (double)  Distance between moderator to sample [m]
    @param L2       (double)  between sample to PSD avarage [m]
    @param NumOfPixel (int) Number of pixels in one PSD
    @retval DAT     ElementCotainerMatix as result
    """
    return BaseCom.GetHistogramBydE2(runNo,Ei,Resolution,t0shift,hw_min,hw_max,NumOfPixel,0)
    
#########################################
def GetDammyHistogramAroundEi(Ei=0.0,deltaE=0.0,binTOF=1,t0shift=0.0,L1=18.03,L2=2.50):
    """
    Make dammy histogram of the TOF region around Ei for checking scan area.
    GetHistogramAroundEi(Ei=0.0,binTOF=-1,t0shift=0.0,L1=18.0,L2=2.5):
    @param Ei       (double) Incident Energy
    @param deltaE   (double) binning of energy transfer
    @param binTOF   (int) bin width of TOF
    @param t0shift  (double) T0 offset [micro-sec]
    @param L1       (double)  Distance between moderator to sample [m]
    @param L2       (double)  between sample to PSD avarage [m]
    @retval DAT     ElementCotainerMatix as result
    """
    return BaseCom.GetDammyHistogramAroundEi(Ei,deltaE,binTOF,t0shift,L1,L2)

#########################################
def GetData(runnumber=10,inform_xml="DI_all.xml",daq=-1,Multh=1):
    """
    Load EventData and convert to histogram.
    @param  runnumber  Run Number
    @param  inform_xml Detector Info xml file
    @param daq         Where event data are stored (0:CPU0, 1:CPU1, -1:/data/SIK)
    @param Multh       Use Multi-Thread = 1, Use Single-Thread =0
    @retval DAT        ElementContainerMatrix
    """
    return BaseCom.GetData(runnumber,inform_xml,daq,Multh)

#########################################
def GetDataPara(runnumber=10,mode=0,envFile="environ_ana.xml"):
    """
    Load EventData and convert to histogram using Main UI (multi-threading) and Sub Ana (server distributed)
    @param runnumber (int) Run Number.
    @param mode      (int) mod=0(PSD 100 bin), mod=1(PSD 1000 bin)
    @param envFile   (string) file name of set environments
    @retval DAT        ElementContainerMatrix
    """
    return BaseCom.GetDataPara(runnumber,mode,envFile)

#########################################
def SetMeasurementParams(dat=DAT, L1=18.03, Ei=100.0):
    """
    Set some parameters into the header of ElementContainerMatrix for data
    @param data    (ElementContainerMatrix) measurement data.
    @param L1      (double) The distance between target to the sample [m]
    @param Ei      (double) Incident Energy [meV]
    @retval None
    """
    BaseCom.SetMeasurementParams(dat,L1,Ei)
    
#########################################
def GetDummyDataForCheck(inform_xml="DI_all.xml",det_posi_xml="DetectorPosi2.xml",Ei=100.0,delta_e=1.0,delta_tof=5.0,L1=18.03,L2=2.50,t0shift=0.0 ):
    """
    Load Dammy data to check for measuring area
    @param inform_xml (string) Detector Info xml file
    @param det_poxi_xml (string) Detector Position xml file
    @param Ei (double) Incident Energy [meV]
    @param delta_e (double) bin width of energy transfer [meV]
    @param delta_tof (double) bin width of TOF [micro sec]
    @param L1 (double) L1
    @param L2 (double) L2
    @param t0shift (double) Shift Time at T0
    @retval DAT
    """
    return BaseCom.GetDummyDataForCheck(inform_xml,det_posi_xml,Ei,delta_e,delta_tof,L1,L2,t0shift)

#########################################
def SetTofByEiBeforGetData(File="DI_all.xml",Ei=100.0,delta_tof=5.0,L1=18.03,L2=2.50,t0shift=0.0,EminRate=0.4,EmaxRate=0.9):
    """
    Function to convert Time-of-Flight to Energy 
    
    @param File      (string) WiringFile(DetectorInfor.xml) or "para" to change several files for GetDataPara command. "para1k" to change for 1000 bin on PSD.
    @param  Ei       (double)  Incident Energy [meV]
    @param delta_tof (double) constant tof
    @param L1        (double)  Distance between moderator to sample [m]
    @param L2        (double)  between sample to PSD avarage [m]
    @param t0shift   (double) T0 shift [micro-sec]
    @param EminRate  (double)　range for minus E
    @param EmaxRate  (double)  range for plus E
    """
    BaseCom.SetTofByEiBeforGetData(File,Ei,delta_tof,L1,L2,t0shift,EminRate,EmaxRate)
    
#########################################
def SetTofToDefault(File="DI_all.xml",start_tof=0,end_tof=40000,delta_tof=100):
    """
    Function to convert Time-of-Flight to Energy
    
    @param File(string) WiringFile(DetectorInfor.xml) or "para" to change several files for GetDataPara command. "para1k" to change for 1000 bin on PSD.
    @param start_tof  (double) start tof
    @param end_tof    (double) end tof
    @param delta_tof  (double) bin width on tof
    """
    BaseCom.SetTofToDefault(File,start_tof,end_tof,delta_tof)

#########################################
def AppendRawDataFiles(OutRunNo=990000,outDataPath="",RunNo1=0,RunNo2=0,RunNo3=0,RunNo4=0,RunNo5=0,RunNo6=0):
    """
    append several EventData iles into one 
    
    @param OutRunNo (int) output run number
    @param outDataPath (string) output data path (default is empty)
    @param RunNo1   (int) run number 1
    @param RunNo2   (int) run number 2
    @param RunNo3   (int) run number 3
    @param RunNo4   (int) run number 4
    @param RunNo5   (int) run number 5
    @param RunNo6   (int) run number 6
    @retval None
    """
    BaseCom.AppendRawDataFiles(OutRunNo,outDataPath,RunNo1,RunNo2,RunNo3,RunNo4,RunNo5,RunNo6)

#########################################
def VisContMOutputFigure(dat, param_file="param.xml",output_file="./figure.png",zmin=-1.0,zmax=-1.0):
    """
    Output data to a file using VisualContCommon function
    [inamura 120417]
    @param dat (ElementContainerMatrix)
    @param param_file (string) Parameter XML file for VisualCont
    @param output_file (string) Path to output file name of figure.
    @param zmin (float) min value of plotting Z-range
    @param zmax (float) max value of plotting Z-range
    @return None
    """
    BaseCom.VisContMOutputFigure(dat, param_file,output_file,zmin,zmax)

#########################################
def CheckSampleTempByRunNumber(runNo,TCh):
    """
    @param runNo (int) Run No.
    @parm TCh (string) Sensor Channel on LS340
    @retval EC
    """
    #return BaseComDNA.CheckSampleTempByRunNumber(runNo,TCh)
    #return LS340ASMW.CheckSampleTempByRunNumber(runNo,TCh)
    if int(runNo) <= 9851:
        return LS340ASMW.CheckSampleTempByRunNumber(runNo,TCh)
    else:
        return LS350IROHA2.CheckSampleTempByRunNumber(runNo, TCh) 
#########################################
def DoMaskForEachEC(dat=DAT):
    """
    """

    return BaseComDNA.DoMaskForEachEC(dat)

#########################################

def MergeWithWeight(dat1=DAT,dat2=DAT):
    """
    @param dat1 (ElementContainerMatrix)
    @param dat2 (ElementContainerMatrix)
    @retval EC
    """

    return BaseComDNA.MergeWithWeight(dat1,dat2)

#########################################

def MergeWithWeightForList(dats=[]):
    """
    """

    return BaseComDNA.MergeWithWeightList(dats)

#########################################
# Dictionary for entry the name of functions
_functions = {
              "CopyData":CopyData,
              "GetHistogramHW":GetHistogramHW,
              "GetHistogramHW333":GetHistogramHW333,
              "GetHistogramHWTrignet":GetHistogramHWTrignet,
              "GetHistogramHW333Trignet":GetHistogramHW333Trignet,
              "GetHistogramHW311":GetHistogramHW311,

              "GetHistogramHWnoTrignet":GetHistogramHWnoTrignet,
              "GetHistogramHW333noTrignet":GetHistogramHW333noTrignet,

              "GetHistogram":GetHistogram,
              "GetHistogramMon":GetHistogramMon,
              "GetHistogramMonTrignet":GetHistogramMonTrignet,
              "GetHistogramMonnoTrignet":GetHistogramMonnoTrignet,
              "GetHistogramMonETrignet":GetHistogramMonETrignet,
              "GetHistogramMonEnoTrignet":GetHistogramMonEnoTrignet,
              "GetHistogramMonAtSmp":GetHistogramMonAtSmp,
              "GetHistogramMonVan":GetHistogramMonVan,

              "CreateQEMap":CreateQEMap,
              "MonitorAbsorptionCorrection":MonitorAbsorptionCorrection,

              "OutputQEEC":OutputQEEC,
              "OutputEC":OutputEC,
              "OutputECA":OutputECA,
              "OutputECM":OutputECM,
              "OutputECAText":OutputECAText,
              "OutputECADAVEText":OutputECADAVEText,
              "OutputNeiECAText":OutputNeiECAText,
              "OutputECText":OutputECText,
              "ReadEC":ReadEC,
              "ReadECA":ReadECA,
              "ReadECM":ReadECM,
              "SliceQE":SliceQE,
              "SliceQEall":SliceQEall,
              "OutputPartOfEC":OutputPartOfEC,
              "GetInformationOfECbin":GetInformationOfECbin,
              "ILambdaCorrDNA":ILambdaCorrDNA,
              "ILambdaCorrUseMonAtSmpDNA":ILambdaCorrUseMonAtSmpDNA,
              "GetHistForILC":GetHistForILC,
              "SolidAngleCorrDNA":SolidAngleCorrDNA,
              "DoQEMask":DoQEMask,

              "GetHistogramDB":GetHistogramDB,
              "GetHistogramOfEnergyTransfer":GetHistogramOfEnergyTransfer,
              "GetHistogram_Time":GetHistogram_Time,
              "GetHistogramBydE":GetHistogramBydE,
              "GetHistogramBydE_Time":GetHistogramBydE_Time,
              "GetMultiEi":GetMultiEi,
              "GetEi":GetEi,
              "SetMeasurementParams":SetMeasurementParams,
              "NormByBeamCurrent":NormByBeamCurrent,
              "DoMask":DoMask,
              "MakeWhiteVanCorr":MakeWhiteVanCorr,
              "WhiteVanCorr":WhiteVanCorr,
              "TofToEnergyTransfer":TofToEnergyTransfer,
              "TofToEnergyTransferStrict":TofToEnergyTransferStrict,
              "TofToEnergy":TofToEnergy,
              "KiKfCorrection":KiKfCorrection,
              "VisContOutputFileByPhi":VisContOutputFileByPhi,
              "VisContMOutputFileByPhi":VisContMOutputFileByPhi,
              "BoseFactorCorrection":BoseFactorCorrection,
              "BoseFactCorrEC":BoseFactCorrEC,
              "ToPowder":ToPowder,
              "SolidAngleCorrection":SolidAngleCorrection,
              "FlexCalc":FlexCalc,
              "SumOfTOF":SumOfTOF,
              "SumOfPixel":SumOfPixel,
              "PickUpRegion":PickUpRegion,
              "GetProfile":GetProfile,
              "NormByBeam":NormByBeam,
              "getBeamCurrent":getBeamCurrent,
              "SaveDataToManyoBinary":SaveDataToManyoBinary,
              "LoadDataFromManyoBinary":LoadDataFromManyoBinary,
              
              "SaveDataToSPE":SaveDataToSPE,
              "SaveDataToNxspe":SaveDataToNxspe,
              "LoadDataFromNxspe":LoadDataFromNxspe,
              "SaveDataToDAVEgrp":SaveDataToDAVEgrp,
              "SavePowderDataToText":SavePowderDataToText,
              "SaveDataToDump":SaveDataToDump,
              "LoadDataFromDump":LoadDataFromDump,
              "SaveDataToNeXus":SaveDataToNeXus,
              "GetDataFromNeXus":GetDataFromNeXus,
              "AppendRawDataFiles":AppendRawDataFiles,
              "VisContMOutputFigure":VisContMOutputFigure,
              "CheckSampleTempByRunNumber":CheckSampleTempByRunNumber,
              "DoMaskForEachEC":DoMaskForEachEC,
              "MergeWithWeight":MergeWithWeight,
              "MergeWithWeightForList":MergeWithWeightForList,
              "MutiplyConstant":MutiplyConstant,
              "MakeRunInfoList":MakeRunInfoList,
              "CalcDATSumWithNormCurrent":CalcDATSumWithNormCurrent
          }
#"PeakFitSimple":PeakFitSimple,
#"PeakFitGaussConst":PeakFitGaussConst,
#"PeakFitLorentzConst":PeakFitLorentzConst,
#"PeakFitErrorFuncConst":PeakFitErrorFuncConst,
#"PeakFitTriangleConst":PeakFitTriangleConst,
#"PeakFitTrapezoidConst":PeakFitTrapezoidConst,
#"PeakFitSimple0":PeakFitSimple0,
#"PeakFitGauss":PeakFitGauss,
#"PeakFitLorentz":PeakFitLorentz,
#"PeakFitErrorFunc":PeakFitErrorFunc,
#"PeakSearchSimple":PeakSearchSimple,
#"PeakFitSimpleECA":PeakFitSimpleECA,
#"PeakFit":PeakFit,
#"PeakFitNotUseWeight":PeakFitNotUseWeight,

_functionsOrder = [
    "GetHistogramHW",
    "GetHistogramHW333",
    "GetHistogramHWTrignet",
    "GetHistogramHW333Trignet",
    "GetHistogramHW311",
    
    "GetHistogramHWnoTrignet",
    "GetHistogramHW333noTrignet",
    
    "GetHistogram",
    "GetHistogramMon",
    "GetHistogramMonTrignet",
    "GetHistogramMonnoTrignet",
    "GetHistogramMonETrignet",
    "GetHistogramMonEnoTrignet",
    "GetHistogramMonAtSmp",
    "GetHistogramMonVan",
    
    "CreateQEMap",
    "MonitorAbsorptionCorrection",
    
    "OutputQEEC",
    "OutputEC",
    "OutputECA",
    "OutputECM",
    "OutputECAText",
    "OutputECADAVEText",
    "OutputNeiECAText",
    "OutputECText",
    "ReadEC",
    "ReadECA",
    "ReadECM",
    "SliceQE",
    "SliceQEall",
    "OutputPartOfEC",
    "GetInformationOfECbin",
    "ILambdaCorrDNA",
    "ILambdaCorrUseMonAtSmpDNA",
    "GetHistForILC",
    "SolidAngleCorrDNA",
    "DoQEMask",

    "SetMeasurementParams",
    "NormByBeamCurrent",
    "DoMask",
    "KiKfCorrection",
    "VisContMOutputFileByPhi",
    "BoseFactorCorrection",
    "BoseFactCorrEC",
    "ToPowder",
    "SolidAngleCorrection",
    "FlexCalc",
    "SumOfTOF",
    "SumOfPixel",
    "PickUpRegion",
    "NormByBeam",

    "SaveDataToManyoBinary",
    "LoadDataFromManyoBinary",
    "SaveDataToNxspe",
    "LoadDataFromNxspe",
    "SaveDataToDAVEgrp",
    "SaveDataToSPE",
    "SavePowderDataToText",
    "SaveDataToDump",
    "LoadDataFromDump",
    "SaveDataToNeXus",
    "GetDataFromNeXus",
    "AppendRawDataFiles",
    "VisContMOutputFigure",
    "CheckSampleTempByRunNumber",
    "DoMaskForEachEC",
    "MergeWithWeight",
    "MergeWithWeightForList",
    "MutiplyConstant",
    "MakeRunInfoList",
    "CalcDATSumWithNormCurrent"
    ]

