#include "AdvPeakSearch.hh"

// class name
const string AdvPeakSearch::className = string("AdvPeakSearch");

/**
 *  default constructor
 */
AdvPeakSearch::AdvPeakSearch() {
    string memberName=string("PeakSerach()");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
};

/**
 *  constructor
 *
 *  \param[in] src  an element container as source
 */
AdvPeakSearch::AdvPeakSearch(ElementContainer *src) : AdvOperationBase(src) {
    string memberName=string("PeakSerach(ElementContainer *)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}

/**
 *  constructor
 *
 *  \param[in] src    an element container as source
 *  \param[in] method an algorithm for smoothing while peak search
 */
#ifndef SWIGPYTHON
AdvPeakSearch::AdvPeakSearch(ElementContainer *src,       AdvMethod *method        ) : AdvOperationBase(src, method) {
    string memberName=string("PeakSerach(ElementContainer *, AdvMethod *)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}
#endif

/**
 *  constructor
 *
 *  \param[in] src  an element container as source
 *  \param[in] type an algorithm for smoothing while peak search
 */
AdvPeakSearch::AdvPeakSearch(ElementContainer *src, const AdvMethodType &methodType) : AdvOperationBase(src, methodType) {
    string memberName=string("PeakSerach(ElementContainer *, AdvMethodType &)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}

/**
 *  constructor
 *
 *  \param[in] src        an element container as source
 *  \param[in] methodName an algorithm for smoothing while peak search
 */
AdvPeakSearch::AdvPeakSearch(ElementContainer *src, const string     &methodName) : AdvOperationBase(src, methodName) {
    string memberName=string("PeakSerach(ElementContainer *, string &)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}

/**
 *  constructor
 *
 *  \param[in] src        an element container as source
 *  \param[in] methodName an algorithm for smoothing while peak search
 *  \param[in] xLower     the lower bound of the domain
 *  \param[in] xUpper     the upper bound of the domain
 */
#ifndef SWIGPYTHON
AdvPeakSearch::AdvPeakSearch(ElementContainer *src,       AdvMethod *method,         const Double xLower, const Double xUpper) : AdvOperationBase(src, method, xLower, xUpper) {
    string memberName=string("PeakSerach(ElementContainer *, AdvMethod *, Double, Double)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}
#endif

AdvPeakSearch::AdvPeakSearch(ElementContainer *src, const AdvMethodType &methodType, const Double xLower, const Double xUpper) : AdvOperationBase(src, methodType, xLower, xUpper) {
    string memberName=string("PeakSerach(ElementContainer *, AdvMethodType &, Double, Double)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}

AdvPeakSearch::AdvPeakSearch(ElementContainer *src, const string &methodName,     const Double xLower, const Double xUpper) : AdvOperationBase(src, methodName, xLower, xUpper) {
    string memberName=string("PeakSerach(ElementContainer *, string &, Double, Double)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}

#ifndef SWIGPYTHON
AdvPeakSearch::AdvPeakSearch(ElementContainer *src,        AdvMethod *method,        const UInt4  lower,  const UInt4  upper) : AdvOperationBase(src, method,     lower, upper) {
    string memberName=string("PeakSerach(ElementContainer *, AdvMethod *, UInt4, UInt4)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}
#endif

AdvPeakSearch::AdvPeakSearch(ElementContainer *src, const AdvMethodType &methodType, const UInt4  lower,  const UInt4  upper) : AdvOperationBase(src, methodType, lower, upper) {
    string memberName=string("PeakSerach(ElementContainer *, AdvMethodType &, UInt4, UInt4)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}

AdvPeakSearch::AdvPeakSearch(ElementContainer *src, const string &methodName,     const UInt4  lower,  const UInt4  upper) : AdvOperationBase(src, methodName, lower, upper) {
    string memberName=string("PeakSerach(ElementContainer *, string &, UInt4, UInt4)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, "exit\n");
}

AdvPeakSearch::~AdvPeakSearch() {
};

void AdvPeakSearch::execute() {

    string memberName=string("execute()");

    if (! this->method->checkParam(*(this->source), this->domain, this->param) ) {
        return;
    }

    this->method->toInnerForm(*(this->source), this->domain, this->param);
    this->method->fit();
    //this->method->eval();

    // incluease-decrease table
    vector< vector<Double> > trend=this->method->getTrend();
    //this->resultY=new vector<Double>(trend[0].begin, trend[0].end());
    DebugMessage(className, memberName, "getTrend\n");

    // search peaks and sholder
    string xKey=this->source->PutXKey();
    string yKey=this->source->PutYKey();
    this->peakData=new AdvPeakData();
    Double x, xl, xu, h, w;
    for (UInt4 i=0; i<trend[0].size()-1; ++i) {
        if        ( isPeak(trend, i) ) {       // found a peak
            // the position of the peak
            x=this->source->Put(xKey, i+1);
            // the height of the peak
            h=(trend[0].at(i) + trend[0].at(i+1))/2.0;

            xl=searchHalfHeightInIncreaseRegion(trend, this->source->Put(xKey), i,   h);
            xu=searchHalfHeightInDecreaseRegion(trend, this->source->Put(xKey), i+1, h);
            w=max(x-xl, xu-x);

            DebugMessage(className, memberName, "%-2s x=%10.5f, xl=%10.5f, xr=%10.5f, h=%23.15e w=%10.5f\n", "p", x, xl, xu, h, w);
            peakData->add(PEAK, x, h, w, xl, xu);
        } else if (isSholderInIncreaseRegion(trend, i)) {
            x=this->source->Put(xKey, i+1);
            h=(trend[0].at(i) + trend[0].at(i+1))/2.0;

            xl=searchHalfHeightInIncreaseRegion(trend, this->source->Put(xKey), i,   h);
            xu=x;
            w=x-xl;

            DebugMessage(className, memberName, "%-2s x=%10.5f, xl=%10.5f, xr=%10.5f, h=%23.15e w=%10.5f\n", "si", x, xl, xu, h, w);
            peakData->add(SHOLDER_I, x, h, w, xl, xu);
        } else if (isSholderInDecreaseRegion(trend, i)) {
            x=this->source->Put(xKey, i+1);
            h=(trend[0].at(i) + trend[0].at(i+1))/2.0;

            xl=x;
            xu=searchHalfHeightInDecreaseRegion(trend, this->source->Put(xKey), i+1, h);
            w=xu-x;

            DebugMessage(className, memberName, "%-2s x=%10.5f, xl=%10.5f, xr=%10.5f, h=%23.15e w=%10.5f\n", "sd", x, xl, xu, h, w);
            peakData->add(SHOLDER_D, x, h, w, xl, xu);
        }
    }
}

/**
 *  @param[in]  trend  increase-decrease teble
 *  @param[in]  pos    index on the table
 *  @retval true if peak condition is sattisfied
 */
Bool AdvPeakSearch::isPeak(vector< vector<Double> > &trend, const UInt4 pos) const {
    return (trend[1].at(pos) > 0.0 && 0.0 > trend[1].at(pos+1));
}

/**
 *  @param[in]  trend  increase-decrease teble
 *  @param[in]  pos    index on the table
 *  @retval true if solder condition is sattisfied
 */
Bool AdvPeakSearch::isSholderInIncreaseRegion(vector< vector<Double> > &trend, const UInt4 pos) const {
    return (0.0 < trend[1].at(pos) && 0.0 < trend[1].at(pos+1) && trend[2].at(pos) < 0.0 && 0.0 < trend[2].at(pos+1) ) ;
}

/**
 *  @param[in]  trend  increase-decrease teble
 *  @param[in]  pos    index on the table
 *  @retval true if solder condition is sattisfied
 */
Bool AdvPeakSearch::isSholderInDecreaseRegion(vector< vector<Double> > &trend, const UInt4 pos) const {
    return ( 0.0 > trend[1].at(pos) && 0.0 > trend[1].at(pos+1) && trend[2].at(pos) > 0.0 && 0.0 > trend[2].at(pos+1) );
}

/**
 *  @param[in]  trend  increase-decrease teble
 *  @param[in]  pos    index on the table
 *  @raram[in]  h      peak height
 *  @retval true if half-value condition is sattisfied at the pos
 */
Bool AdvPeakSearch::isHalfHeightInIncreaseRegion(vector< vector<Double> > &trend, const UInt4 pos, const Double h) const {
    return  (trend[0].at(pos-1) < h/2.0 && h/2.0 < trend[0].at(pos));
}

Bool AdvPeakSearch::isHalfHeightInDecreaseRegion(vector< vector<Double> > &trend, const UInt4 pos, const Double h) const {
    return  (trend[0].at(pos-1) > h/2.0 && h/2.0 > trend[0].at(pos));
}

/**
 *  @param[in]  trend  increase-decrease teble
 *  @param[in]  pos    index on the table
 *  @raram[in]  h      peak height
 *  @retval true if the height at the bottom of a vallery that is higher than the half height of the peak
 */
Bool AdvPeakSearch::isHigherValley(vector< vector<Double> > &trend, const UInt4 pos, const Double h) const {
    return ( h/2.0 < trend[0].at(pos) && h/2.0 < trend[0].at(pos+1) &&
             trend[1].at(pos) < 0.0   && 0.0 < trend[1].at(pos+1)   &&
             trend[2].at(pos) > 0.0   && trend[2].at(pos+1) > 0.0       );
}

Bool AdvPeakSearch::isHigherSholderInIncreaseRegion(vector< vector<Double> > &trend, const UInt4 pos, const Double h) const {
    return  (trend[0].at(pos-1) < h/2.0 && h/2.0 < trend[0].at(pos) && isSholderInIncreaseRegion(trend, pos));
}

Bool AdvPeakSearch::isHigherSholderInDecreaseRegion(vector< vector<Double> > &trend, const UInt4 pos, const Double h) const {
    return  (trend[0].at(pos-1) > h/2.0 && h/2.0 > trend[0].at(pos) && isSholderInDecreaseRegion(trend, pos));
}

Bool AdvPeakSearch::isHigherEndInIncreaseRegion(vector< vector<Double> > &trend, const UInt4 pos, const Double h) const {
    return  (pos == 1 && h/2 < trend[0].at(pos-1) && trend[0].at(pos-1) < trend[0].at(pos));
}

Bool AdvPeakSearch::isHigherEndInDecreaseRegion(vector< vector<Double> > &trend, const UInt4 pos, const Double h) const {
    return  (pos == trend[0].size()-2 &&  trend[0].at(pos) > trend[0].at(pos+1) && trend[0].at(pos+1) > h/2.0);
}


/**
 *  seatch a half value point in the increase region of the peak
 *
 *  @param[in] trend   increase and decread table
 *  @raram[in] srcX    bin
 *  @raram[in] initPos the index of the staritng bin for search
 *  @param[in] h       the heigtht of the peak
 */
Double AdvPeakSearch::searchHalfHeightInIncreaseRegion(vector< vector<Double> > &trend, vector<Double> srcX, const UInt4 initPos, const Double h) {

    string memberName="searchHalfHeightInIncreaseRegion(vector< vector<Double> > &, vector<Double>, const UInt4, const Double)";
    Double x=srcX.at(0);
     // search a half height point in increase region
    for (UInt4 j=initPos; j>0; --j) {
        if ( isHalfHeightInIncreaseRegion(trend, j, h) ) {
            DebugMessage(className, memberName, "j=%u h\n", j);
            x=srcX.at(j);
            break;
        } else if (isHigherValley(trend, j, h)                 ) {
            DebugMessage(className, memberName, "j=%u v\n", j);
            x=srcX.at(j);
            break;
        } else if (isHigherSholderInIncreaseRegion(trend, j, h)) {
            DebugMessage(className, memberName, "j=%u s\n", j);
            x=srcX.at(j);
            break;
        } else if (isHigherEndInIncreaseRegion(trend, j, h)    ) {
            DebugMessage(className, memberName, "j=%u e\n", j);
            x=srcX.at(j-1);
            break;
        }
    }

    return x;
}

/**
 *  search a half value point in the decrease region of the peak
 *
 *  @param[in] trend increase and decrease table
 *  @param[in] srcX  bin
 *  @param[in] initPos the index of the starting bin for search
 *  @param[in] h       the heigtht of the peak
 */
Double AdvPeakSearch::searchHalfHeightInDecreaseRegion(vector< vector<Double> > &trend, vector<Double> srcX, const UInt4 initPos, const Double h) {

    string memberName="searchHalfHeightInDecreaseRegion(vector< vector<Double> > &, vector<Double>, const UInt4, const Double)";
    Double x=srcX.at(0);
    // search a half height point in decrease region
    for (UInt4 j=initPos; j< trend[0].size()-1U; ++j) {
        if ( isHalfHeightInDecreaseRegion(trend, j, h) ) {
            DebugMessage(className, memberName, "j=%u h\n", j);
            x=srcX.at(j+1);
            break;
        } else if (isHigherValley(trend, j, h)                 ) {
            DebugMessage(className, memberName, "j=%u v\n", j);
            x=srcX.at(j+1);
            break;
        } else if (isHigherSholderInDecreaseRegion(trend, j, h)) {
            DebugMessage(className, memberName, "j=%u s\n", j);
            x=srcX.at(j+1);
            break;
        } else if (isHigherEndInDecreaseRegion(trend, j, h)    ) {
            DebugMessage(className, memberName, "j=%u e\n", j);
            x=srcX.at(j+2);
            break;
        }
    }
    return x;
}

