#ifndef SIMULATEDANNEALING_ARGS
#define SIMULATEDANNEALING_ARGS

#include "Header.hh"
#include "ElementContainer.hh"
#include "ElementContainerArray.hh"

#include "AdvMessage.hh"
#include "AdvDomain.hh"
#include "AdvParamSet.hh"

#include <gsl/gsl_rng.h>
#include <gsl/gsl_siman.h>

#include "AdvSimulatedAnnealingConsts.hh"

class AdvFuncBase;
class AdvSimulatedAnnealing;

class AdvSimulatedAnnealingArgs : public AdvMessage {

    friend class AdvSimulatedAnnealing;

    public:
        /** constructor */
        AdvSimulatedAnnealingArgs();

        /** destructor */
        ~AdvSimulatedAnnealingArgs();

        void SetCurParam(Double* v, Int4 n) { cur_param.resize(n); for (Int4 i=0;i<n;++i) cur_param[i]=v[i]; }
        void SetCurParamBest() { best_param = cur_param; }

    private:
        static const std::string className; // =std::string("AdvSimulatedAnnealingArgs");

        /** a list of fitting functions */
        std::vector<AdvFuncBase*> funcList;

        /** a matrix of fitting functions (for multi-fit) */
        std::vector< std::vector<AdvFuncBase*> > funcMatrix;

        /* the number of fitting parameters */
        Int4    nParam;

        /* the number of reference data */
        std::vector<Int4>  nRef;

        /* the initial values of the fitting parameters */
        std::vector<Double> param;
        std::vector<Double> cur_param;

        /* the values of the reference data */
        Int4    n_data;
        std::vector< std::vector<Double> >  ref;
        std::vector< std::vector<Double> >  err;

        std::vector< std::vector<Double> >  srcBin;
        std::vector< std::vector<Double> >  srcX;

        std::vector<Double> best_param;

        std::vector< std::vector<Double> >  bestY;

        Int4 ncall;
        Double min_chi2;

        /* the lower bounes of the fitting parameters */
        std::vector<Double> lb;

        /* the upper bounes of the fitting parameters */
        std::vector<Double> ub;

        /** constrain type */
        AdvSimulatedAnnealingConsts::Constrain  constrain;

        /** flag for Least Square method with data weight */
        Bool useDataWeights;

        /* output interval */
        Int4 outputInterval;

        /** GSL: random number generator */
        gsl_rng *rand;

        /** GSL: parameter for gsl_siman_solve function */
        gsl_siman_params_t  siman_params;

        AdvParamSet setDefaultParam();
        AdvParamSet setDefaultParam(ElementContainer &src);
        AdvParamSet setDefaultParam(ElementContainerArray &src);

        /** check parameters */
        Bool checkParam(ElementContainer &src, AdvDomain &domain, AdvParamSet &param);
        Bool checkParam(ElementContainerArray &src, std::vector<AdvDomain> &domainArray, AdvParamSet &param);

        /* check parameter */
        Bool checkConstrainType(const AdvParamSet &param);
        Bool checkUseDataWeights(const AdvParamSet &param);
        Bool checkNumberOfFittingParam(const AdvParamSet &pram);
        Bool checkBoxies(const AdvParamSet &param);

        /** translate to inner form */
        void toInnerForm(ElementContainer &src, AdvDomain &domain, AdvParamSet &param);
        void toInnerForm(ElementContainerArray &src, std::vector<AdvDomain> &domain, AdvParamSet &param);
        /** translate into the inner form */
        void toInnerForm(const AdvParamSet &param);

        AdvParamSet getBestParam() const;

        std::vector<Double> getInitialParam()  { return param; }
        std::vector<Double> getLowerBounds()   { return lb; }
        std::vector<Double> getUpperBounds()   { return ub; }

        /* translate to inner form */
        AdvSimulatedAnnealingConsts::Constrain  getConstrain(const AdvParamSet &param);

        enum CHI2MODE {
            USE_GIVEN_ERRORS,
            USE_Y_INVERSE,
            USE_LOG,
            NO_WEIGHT
        };

        enum CHI2MODE  chi2mode;


        std::vector<AdvFuncBase*>           setFuncList(const AdvParamSet &param);
        std::vector< std::vector<AdvFuncBase*> > setFuncMatrix(const AdvParamSet &param);
        Int4               setNumberOfRequiredParam(const AdvParamSet &param);
        Int4               setNumberOfFittingParam(const AdvParamSet &param);
        // Int4               setNumberOfReferenceData(const ElementContainer &src, const AdvDomain &domain);
        // Double            *setFittingParamArray(const AdvParamSet &param);
        // Double            *setReferenceDataArray(ElementContainer &src, const AdvDomain &domain, const AdvParamSet &param);
        // Double            *setDataWeightArray(ElementContainer &src, const AdvDomain &domain);

        /**  */
        // std::vector< std::vector<Double> > getExtCoeffMatrix(const Double *A, const Double *b, const Int4 nEq, const Int4 nParam);
};

#endif //  SIMULATEDANNEALING_ARGS
