#ifndef CONTAINER_FOR_VECTOR
#define CONTAINER_FOR_VECTOR

#include <algorithm>
#include <functional>
#include <stdexcept>
#include <cmath>

#include "StringTools.hh"

#include "ContainerBaseTemplate.hh"

template<typename _SCALER_TYPE >
struct ValueSizeComp {
    Bool operator() (std::pair<const std::string, _SCALER_TYPE >& item1, std::pair<const std::string, _SCALER_TYPE >& item2) {
        return item1.second.size() < item2.second.size();
    };
};

/** Container For STL-Vector.
 *
 *  @see ContainerForScaler for add, replace, erase, empty, size, containe
 *
 *  @author TANIMORI Souichirou, AdvanceSoft Corp.
 *  @version 0.0
 *  @since   0.0
 */
template<typename _ELEMENT_TYPE>
class AdvContainerForVector : virtual public ContainerForScaler< std::vector< _ELEMENT_TYPE > > {
    public:
        static const std::string className; // = std::string("");

    public:
        /** constructor */
        AdvContainerForVector< _ELEMENT_TYPE >() { this->clear(); };

        /** constructor */
        ~AdvContainerForVector< _ELEMENT_TYPE >() { this->clear(); };

        /** get the size of the specified std::vector with the key.
         */
        UInt4 getVectorSize(const std::string& key) ;

        /** get the i-th. element of a vecttor.
         *  @param[in] key  the key for a vector
         *  @param[in] i    the index for the element
         */
        _ELEMENT_TYPE get(const std::string& key, const UInt4 i) throw(std::invalid_argument, std::out_of_range) ;

        /** replace the value with the specified key to new value.
         *  @param[in] key    the key for a value to be repalced
         *  @param[in] i      the index for the element of the value
         *  @param[in] value  the new value to be reparaced into
         */
        void replace(const std::string& key, const UInt4 i, const _ELEMENT_TYPE& value);

        /** output entry to the standard output.
         *  @param[in] indent
         */
        void dump(const UInt4 indent=0, const UInt4 indentDepth=4, const std::string& keyTitile="key", const std::string& sizeTitle="size", const std::string& valueTitle="value");
};

template<typename _ELEMENT_TYPE >
UInt4 AdvContainerForVector< _ELEMENT_TYPE >::getVectorSize(const std::string& key) {
    UInt4 retval=0;
    if (! this->contain(key)) {
        std::cerr<< "Warning: AdvContainerForVector::getVectorSize: " << __FILE__ << ":" << __LINE__ << ": not found the value with key \"" << key << "\"" << std::endl;
    } else {
        retval = this->cstd::map[key].size() ;
    }
    return retval;
};

template<typename _ELEMENT_TYPE >
_ELEMENT_TYPE AdvContainerForVector< _ELEMENT_TYPE >::get(const std::string& key, const UInt4 i) throw(std::invalid_argument, std::out_of_range) {
    if (! this->contain(key) ) {
        std::cerr << "Error: AdvContainerForVector::get: " << __FILE__ << ":" << __LINE__ << ": not found the value with key=\"" << key << "\"" << std::endl;
        throw std::invalid_argument(std::string("AdvContainerForVector::get: not found the entry with the specified key \"" + key + "\"\n"));
    }
    if ( i >= this->cstd::map.size() ) {
        StringTools tools;
        std::cerr << "Error: AdvContainerForVector::get: " << __FILE__ << ":" << __LINE__ << ": index is out of range: key=\"" << key << "\", index=" << i << std::endl;
        throw std::out_of_range(std::string("AdvContainerForVector::get: the index is out of range.  i=\"") + tools.UInt4ToString(i) + std::string("\"\n"));
    }
    return this->cstd::map[key].at(i);
};

template<typename _ELEMENT_TYPE >
void AdvContainerForVector< _ELEMENT_TYPE >::replace(const std::string& key, const UInt4 i, const _ELEMENT_TYPE& value) {
    if (! this->contain(key) ) {
        std::cerr << "Warning: AdvContainerForVector::get: " << __FILE__ << ":" << __LINE__ << ": not found the value with key=\"" << key << "\"" << std::endl;
    } else if ( i >= this->cstd::map.size() ) {
        std::cerr << "Warning: AdvContainerForVector::get: " << __FILE__ << ":" << __LINE__ << ": index is out of range: key=\"" << key << "\", index=" << i << std::endl;
    } else {
        this->cstd::map[key].at(i)=value;
    }
};

template< typename _ELEMENT_TYPE >
void AdvContainerForVector< _ELEMENT_TYPE >::dump(const UInt4 indent, const UInt4 indentDepth, const std::string& keyTitle, const std::string& sizeTitle, const std::string& valueTitle) {

    if (! this->cstd::map.empty() ) {
        size_t maxKeyWidth  = 0;
        maxKeyWidth = std::max(keyTitle.size(), max_element(this->cstd::map.begin(), this->cstd::map.end(), KeyLengthComp< std::vector< _ELEMENT_TYPE > >())->first.size());

        size_t maxSizeWidth = 0;
        maxSizeWidth = max_element(this->cstd::map.begin(), this->cstd::map.end(), ValueSizeComp< std::vector<_ELEMENT_TYPE> >())->second.size();
        maxSizeWidth = std::max(sizeTitle.size(), maxSizeWidth > 0 ? static_cast<size_t>(floor(log10(static_cast<Double>(maxSizeWidth)))+1) : 1);

        OutputTypeTitle ott;
        ott.title(indentDepth*indent, typeid(std::vector< _ELEMENT_TYPE >) );
        ott.header(indentDepth*(indent+1), maxKeyWidth, keyTitle, maxSizeWidth, sizeTitle, valueTitle);

        for_each(this->cstd::map.begin(), this->cstd::map.end(), OutputVectorEntry< _ELEMENT_TYPE >(indentDepth*(indent+1), maxKeyWidth, maxSizeWidth));
    }
};
 
#endif // CONTAINER_FOR_VECTOR
