#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
U2IFJupyter.py
uGao u2dplot I/F on Jupyter and on the dark mode
"""
from __future__ import print_function
import os
import Manyo

from uGao.u2dplotBase import U2Params, D2Chart, DataConverter, format_int_range
from uGao.uGaoColorMap import COLOR_MAPS
from uGao.uGaoUtil import IFEvtProp, MASKVALUE

import matplotlib
ISJUPYTER = False
if matplotlib.get_backend() == "inline":
    # print("U2IFJupyter : backend = inline")
    ISJUPYTER = True
    import matplotlib.pyplot as plt
else:
    # print("U2IFJupyter : backend is not inline")
    from matplotlib.figure import Figure
    from matplotlib.backends.backend_agg import FigureCanvasAgg as FigCanvas

#########################################
# U2IFJupyter
#########################################
class U2IFJupyter(object):
    """
    U2IFJupyter class

    This enables to plot on Jupyter and to save plot file using u2dplot of uGao

    import uGao.U2IFJupyter as mp
    p = mp.U2IFInDark(sizeX=10.0, sizeY=5.0)
    p.SetKeys("index", "index", "Intensity")
    p.SetData(dat)
    p.SetRange(False, 0.0, 40.0)
    p.SetTitles("Test in Dark", "")
    p.SetLabels("QA", "hw")
    p.SaveFile("test_InDark.png")
    # On Jupyter
    p.Show()
    """

    #########################################
    def __init__(self, data=None, orderNo=1, sizeX=7.0, sizeY=7.0, dpi=100):
        """
        Constructor
        @param data(list of numpy arrays or ElementContainerMatrix or -Array)
        @param orderNo(int) Plot number
        @param sizeX (float) size of X axis direction
        @param sizeY (float) size of Y axis direction
        @param dpi (float) dot per inch
        @retval None
        """
        if ISJUPYTER:
            # print("U2IFJupyter : ISJUPYTER is True")
            self.plt = plt
            self.fig = plt.figure(figsize=(sizeX, sizeY))
            self.canvas = self.fig.canvas
        else:
            # print("U2IFJupyter : ISJUPYTER is False")
            self.fig = Figure((sizeX, sizeY), dpi)
            self.canvas = FigCanvas(self.fig)
        self.order = orderNo
        self.params = U2Params()
        self.sliceMode = False
        self.ifi = IFEvtProp(orderNo)
        self.ifi.AddProperty('params', lambda: self.params)
        self.ifi.AddProperty('slice_mode', lambda: self.sliceMode)

        self.d2 = None

        self.ndata = None
        self.data = data
        self.keys = (None, None, None)
        self.intRange = None

    ###################################################
    def Show(self):
        """
        """
        if ISJUPYTER:
            if self.ndata is None:
                if not self.SetData():
                    msg = "U2IFJupyter::SaveFile >> data is invalid."
                    raise UserWarning(msg)

            if self.d2 is not None:
                del self.d2

            self.d2 = D2Chart(self, self.canvas, self.ndata, True)

            self.plt.show()
        else:
            pass

    #########################################
    def SetKeys(self, xkey, ykey, intkey=None, rkey=None):
        """キー文字列設定処理

        Parameters:
            xkey (str or None): X Key
            ykey (str or None): Y Key
            intkey (str or None): Int Key
            rkey (str or None): Range Key
        """
        if xkey is not None and not isinstance(xkey, str):
            raise PlotException('Common', 'C025', ('xkey',))
        if ykey is not None and not isinstance(ykey, str):
            raise PlotException('Common', 'C025', ('ykey',))
        if intkey is not None and not isinstance(intkey, str):
            raise PlotException('Common', 'C025', ('intkey',))
        if rkey is not None and not isinstance(rkey, str):
            raise PlotException('Common', 'C025', ('rkey',))

        self.keys = (xkey, ykey, intkey, rkey)

    #########################################
    def SetIntRange(self, range0, range1):
        """強度積算範囲設定処理

        self.intRangeに積算範囲を設定する。
        引数のどちらかがNoneの時は全範囲になるようにself.intRangeをNoneにする。

        Parameters:
            range0 (float or None): 積算範囲始値
            range1 (float or None): 積算範囲終値
        """
        self.intRange = format_int_range(range0, range1)

    #########################################
    def SetData(self, data=None):
        """
        Set Data
        @param data(EleentContainerArray, -Matrix or list of arrays)
        @retval None
        """
        if data is None:
            if self.data is None:
                return False
            data = self.data

        if isinstance(data, Manyo.ElementContainerMatrix) or isinstance(data, Manyo.ElementContainerArray):
            self.ndata = DataConverter().ConvertData(data, self.keys, self.intRange)
        elif data is not None:
            self.ndata = data
        else:
            msg = "U2IFJupyter::SetData >> ChangeData argument is invalid."
            raise UserWarning(msg)
        return True

    #########################################
    def ReadData(self, filename):
        """
        """
        print("ReadData not work yet")

    #########################################
    def SetTitles(self, title, comment=None):
        """
        Set title strings
        @param title(string)
        @retval None
        """
        self.params.title = str(title)
        if comment is not None:
            self.params.comment = comment

    #########################################
    def SetTransPosition(self, trans):
        """
        軸の転置指定
        @param  title (string) グラフのタイトル
        @param  comment (string) コメント
        @retval 無し
        """
        # 引数の型チェック
        if isinstance(trans, bool):
            if self.params.transPosition != trans:
                params.transPosition = trans
        else:
            msg = "U2IFJupyter::SetTransPosition >> argument is invalid."
            raise UserWarning(msg)

    #########################################
    def SetXScale(self, autoFlag, r_min, r_max):
        """
        Set Xaxis range
        @param autoFlag(bool)
        @param r_min(float)
        @param r_max(float)
        @retval None
        """
        self.params.xscale.autoRange = autoFlag
        if not autoFlag:
            self.params.xscale.range = (float(r_min), float(r_max))

    #########################################
    def SetYScale(self, autoFlag, r_min, r_max):
        """
        Set Yaxis range
        @param autoFlag(bool)
        @param r_min(float)
        @param r_max(float)
        @retval None
        """
        self.params.yscale.autoRange = autoFlag
        if not autoFlag:
            self.params.yscale.range = (float(r_min), float(r_max))

    #########################################
    def SetLabels(self, xlabel, ylabel):
        """
        軸のラベルを指定
        @param  xlabel (string or None) X軸のラベル
        @param  ylabel (string or None) Y軸のラベル
        @retval 無し
        """
        if xlabel is not None:
            self.params.xscale.label = xlabel
        if ylabel is not None:
            self.params.yscale.label = ylabel

    #########################################
    def SetHGrid(self, grid, color=None, interval=None):
        """
        水平グリッドの指定
        @param  grid (string) グラフのタイトル
        @param  color (int) 0:White, 1:Yellow 2:Red, 3:Black, 4:Blue, 5:Green, 6:Cyan, 7:Purple
        @param  interval (float) 間隔
        @retval 無し
        """
        # 引数の型チェック
        if isinstance(grid, bool):
            self.params.hgrid = grid
            # カラーが指定されていれば
            if color is not None:
                self.params.gridColor = color
            # 間隔が指定されていれば
            if color is not None:
                self.params.hgridInterval = interval

    #########################################
    def SetVGrid(self, grid, color=None, interval=None):
        """
        垂直グリッドの指定
        @param  grid (string) グラフのタイトル
        @param  color (int) 0:White, 1:Yellow 2:Red, 3:Black, 4:Blue, 5:Green, 6:Cyan, 7:Purple
        @param  interval (float) 間隔
        @retval 無し
        """
        # 引数の型チェック
        if isinstance(grid, bool):
            self.params.vgrid = grid
            # カラーが指定されていれば
            if color is not None:
                self.params.gridColor = color
            # 間隔が指定されていれば
            if color is not None:
                self.params.vgridInterval = interval

    #########################################
    def SetColorMap(self, cmap):
        """
        Set color map
        @param cmap(int or string)
        @retval None
        """
        if isinstance(cmap, int):
            if cmap >= 0 and cmap < len(COLOR_MAPS):
                self.params.colorMap = cmap
                return
        elif isinstance(cmap, str):
            for i, a_map in enumerate(COLOR_MAPS):
                if cmap == a_map:
                    self.params.colorMap = i
                    return

        msg = "U2IFJupyter::SetColorMap >> invalid argument."
        raise UserWarning(msg)

    #########################################
    def SetSmooth(self, flag, times=1):
        """
        Set smoothing parameters
        @param flag(int) if flag<=0, Smoothing is off
        @param times(int) the number of times
        @retval None
        """
        if float(flag) <= 0:
            self.params.smoothing = [False, 1, 1]
        else:
            self.params.smoothing = [True, int(flag), int(times)]

    #########################################
    def SetBankSeparator(self, flag=False):
        """
        Bank Separatorの指定
        @param flag   Showing bank separators or not
        @retval None
        """
        # 引数の型チェック
        if isinstance(flag, bool):
            self.params.bankseparator = flag
        else:
            msg = "U2IFJupyter::SetBankSeparator >> invalid argument."
            raise UserWarning(msg)

    #########################################
    def SetRange(self, autoFlag, r_min=None, r_max=None):
        """
        Set Zaxis(Intensity) range
        @param augoFlag(bool)
        @param r_min(float)
        @param r_max(float)
        @retval None
        """
        self.params.autoRange = autoFlag
        if autoFlag:
            if r_min is None and r_max is not None:
                self.params.range = (None, float(r_max))
            elif r_min is not None and r_max is None:
                self.params.range = (float(r_min), None)
            else:
                self.params.range = None

        else:
            self.params.range = (float(r_min), float(r_max))

    #########################################
    def SetLog(self, zlog):
        """
        Set log scale on Z axis
        @param zlog(bool)
        @retval None
        """
        if isinstance(zlog, bool):
            self.params.logScale = zlog
        else:
            msg = "U2IFJupyter::SetLog >> invalid argument."
            raise UserWarning(msg)

    #########################################
    def SetFigureSize(self, px, py):
        """
        Set Size of Figure
        @param px(float) size of X direction
        @param py(float) size of Y direction
        @retval None
        """
        if ISJUPYTER:
            return
        self.fig = Figure((px, py), 100)
        self.canvas = FigCanvas(self.fig)

    #########################################
    def SaveFile(self, filename):
        """
        Save figure file
        @param filename(string)
        @retval None
        """
        if self.ndata is None:
            if not self.SetData():
                msg = "U2IFJupyter::SaveFile >> data is invalid."
                raise UserWarning(msg)

        if self.d2 is not None:
            del self.d2

        self.d2 = D2Chart(self, self.canvas, self.ndata, True)
        if os.path.isdir(os.path.dirname(filename)):
            pass
        else:
            filename = os.path.join(os.getcwd(), filename)

        self.fig.savefig(filename)
    #########################################

    def SaveAsText(self, filename):
        """
        データファイルをテキスト形式で保存処理
        This function is copied from u2dplot::D2Frame::SaveAsTxt.
        @param  filename (string) ファイルのパス
        @retval 無し
        """
        import codecs
        root, ext = os.path.splitext(filename)
        if ext == "":
            filename = root + ".txt"
        elif ext != ".txt":
            raise PlotException('Common', 'C043', ("txt",))
        try:
            # テキストファイルを開く
            fw = codecs.open(filename, 'w', 'utf-8')
        except:
            raise PlotException('Common', 'C018', (filename, ))
        # グラフのタイトルをコメントとして書き込む
        fw.write("# Title: " + self.params.title + os.linesep)
        # グラフのコメントを書き込む
        lines = self.params.comment.split('\n')
        if len(lines) > 0:
            fw.write("# Comment: " + lines[0] + os.linesep)
        if len(lines) > 1:
            fw.write("#          " + lines[1] + os.linesep)
        if len(lines) > 2:
            fw.write("#          " + lines[2] + os.linesep)

        # 行タイトル
        fw.write("#       X           Y         Int" + os.linesep)
        # データを展開
        if self.ndata is None:
            if not self.SetData():
                msg = "U2IFJupyter::SaveAsText >> data is invalid."
                raise UserWarning(msg)
        Z, X, Y = self.ndata
        # テキストファイルに書き込み
        for i in range(len(Z[0])):
            for j in range(len(Z)):
                xx = (X[j][i] + X[j][i + 1]) / 2.0
                yy = (Y[j][i] + Y[j + 1][i]) / 2.0
                value = Z[j][i]
                if value >= MASKVALUE:
                    value *= 1.1
                line = "%.5e %.5e %.5e" % (xx, yy, value)
                fw.write(line + os.linesep)
            fw.write(os.linesep)
        fw.close()
    #########################################

    def OnXYlimChanged(self, axes):
        """XY軸変更コールバック関数

        Parameters:
            axes (matplotlib.axes.Axes)
        """
        params = self.ifi.GetProperty('params')
        if params.transPosition:
            params.xscale.range = list(axes.get_ylim())
            params.yscale.range = list(axes.get_xlim())
        else:
            params.xscale.range = list(axes.get_xlim())
            params.yscale.range = list(axes.get_ylim())
        self.ifi.NotifyEvent(self, 'change_displayed_params')


if __name__ == '__main__':
    import Cmm
    ecm = Cmm.LoadDataFromDump("./", "Load373.dmp")
    p = U2IFJupyter(ecm, 1)

    p.SetKeys(None, None, None)
    p.SetFigureSize(5.0, 5.0)
    p.SetRange(False, 0, 40.0)
    p.SetTitles("U2IFJupyter Test", "YAHOOO!")
    p.SetLabels("|Q| (1/Ang.)", "Energy (meV)")
    p.SetColorMap("haxby")
    p.SaveFile(os.path.join(os.getcwd(), "U2IFJupyter.png"))
