#include "UtsusemiUnitConverter.hh"
const Double PI=MLF_PI;
const Double Mn=MLF_Mn;            // Neutron mass [kg]
const Double PLANCK=MLF_PLANCK;    // Planck constant [Js]
const Double HBAR=MLF_HBAR;        // Planck/2*PI [Js]
const Double kB=MLF_kB;            // Boltzmann constant [J/K]
const Double EV2J=MLF_EV2J;        // 1 [eV] = 1.6 [J]
const Double J2EV=MLF_J2EV;        // 1 [J] = 1/1.6 [eV]
const Double MEV2J=MLF_MEV2J;      // 1meV=1.e-03eV
const Double J2MEV=MLF_J2MEV;

//////////////////////////////////////////////////////////
UtsusemiUnitConverter::
UtsusemiUnitConverter()
{

    ////cMn    = 1.67495e-27;          // [kg]
    ////cnvJtoE = 1.0/(1.60219e-22);   // [J]->[meV]

    /// used before 0.3.3.0
    //cMn      = 1.674927351e-27;    // [kg]
    //cPlanck  = 6.62606957e-34;     // [Js] or [m*m*kg/s]
    //chver  = cPlanck/2./M_PI;

    //cnvJtoE = 6.24150934e21;       // [J]->[meV]


    /// Use MlfPhyisicalConstants (0.3.3.1)
    cMn = MLF_Mn;
    chver = MLF_HBAR;
    cPlanck = chver*2.*MLF_PI;
    cnvJtoE = MLF_J2MEV;

    cLEtoT = sqrt( cMn/2.0*cnvJtoE )*1000000.;
    cVtoE  = (cMn/2.0)*cnvJtoE*1.0e12;
    cEtoV  = 1.0/sqrt(cVtoE);
    cK2toE = chver*chver/(2.0*cMn)*1.0e20*cnvJtoE;
    cVtoLambda = cPlanck/cMn*10000.; // [meter/micro-sec]/[kg] -> [Ang]

    //cLEtoT = 2286.2869;     // [micro sec]/[m]*[meV]
    //cVtoE  = 5.227108e-6;   // [meV]/[(m/s)**2]
    //cEtoV  = 437.39051;     // [m/s]/[sqrt(meV)]
    //cK2toE = 2.0720814;     //
    //cJ2meV = 6.24151e21;
}

//////////////////////////////////////////////////////////
UtsusemiUnitConverter::
~UtsusemiUnitConverter()
{
}

//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
LEtoT( double L, double E )
{
    return cLEtoT*L/sqrt(E);
}

//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
VtoE( double V )
{
    return cVtoE*V*V;
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
VtoK( double V )
{
    return sqrt(cVtoE*V*V/cK2toE);
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
Vmm_msToK()
{
    return sqrt(cVtoE/cK2toE)/1000.0;
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
Vmm_msToK( double V )
{
    return VtoK(V)/1000.0;
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
EtoV( double E )
{
    return cEtoV*sqrt( E );
}

//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
KtoE( double K )
{
    return cK2toE*K*K;
}

//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
EtoK2( double E )
{
    return E/cK2toE;
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
TLtoLambda( double _tof, double _L )
{
    return cVtoLambda/(_L/_tof);
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
VtoLambda( double V ){
    return cVtoLambda/V;
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
EtoLambda( double E ){
    return VtoLambda( cEtoV*sqrt( E ) );
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
KtoLambda( double K ){
    return 2*M_PI/K;
}
//////////////////////////////////////////////////////////
double UtsusemiUnitConverter::
LambdatoE( double Lam ){
    // cVtoLambda/(cEtoV*sqrt(E)) = Lambda
    // sqrt(E) = cVtoLambda/(cEtoV*Lambda)
    return pow(cVtoLambda/(cEtoV*Lam), 2.0);
}
