#ifndef UTSUSEMISETMASK
#define UTSUSEMISETMASK

#include "UtsusemiHeader.hh"

////////////////////////////////////
// UtsusemiSetMask
///////////////////////////////////

//! Sets mask flag at data container on Utsusemi environment
/*!
 * Mask flags is under the mask rule of Manyo-Library
 * - when the HeaderBase of container includes "MASKED" with (Int4) 1
 * - when the error value is minus
 *
 * This UtsusemiSetMask can set mask to ElementContainerMatrix using Mask File.
 * Mask File format1
 * - Ignores line which first charactor is #
 * - text format with space separator
 *    <detId> or <detId>.<pixNo>
 *    detId is value of key "DETID" in HeaderBase of ElementContainerArray
 *    pixNo is the index number of ElementContainer stored in ElementContainerArray
 *
 * Example of mask.txt
 * @code
 * # INSTRUMENT SIK
 * # DATATYPE ELEMENTCONTAINERMATRIX
 * # TIME 2014 3 10
 * 0.0 0.99
 * 1.0 1.99
 * 10
 * 11.0 11.1
 * @endcode
 *
 * Mask File format2
 * xml format
 * Example of mask.xml
 * @code
 * <maskInfo>
 *   <masklist n="2">
 *     <mask i="0">
 *       <pixelid> 100-200 </pixelid>
 *       <detector/>
 *       <pixelno/>
 *       <axis key=""/>
 *     </mask>
 *     <mask i="1">
 *       <pixelid/>
 *       <detector> 32-223 </detector>
 *       <pixelno>All</pixelno>
 *       <axis key="TOF">0:500,1000:1250</axis>
 *     </mask>
 *   </masklist>
 * </maskInfo>
 * @endcode
 *
 * Mask files must be placed in
 *   - ${UTSUSEMI_BASE_DIR}/${UTSUSEMI_INST_CODE}/ana/xml
 *   - ${UTSUSEMI_USR_DIR}/ana/xml
 */

class UtsusemiSetMask :
  public ManipulatorBase< ElementContainerMatrix >
{
private:
    BoostXmlParser* _Parser;
    std::string _AxisKey;
    StringTools* stools;
    std::string _MessageTag;
    void Initialize();
    std::string _MaskTxtPath;
    bool _isMaskXml;

protected:
    bool ExecuteXmlMask();
    bool ExecuteTxtMask();

public:
    UtsusemiSetMask();
        //!< Constructor
        /*!<
         */
    UtsusemiSetMask( ElementContainerMatrix* ecm );
      //!< Constructor
        /*!<
         *    @param ecm      (ElementContainerMatrix*) target data container
         */
    UtsusemiSetMask( ElementContainerMatrix* ecm, std::string filename );
      //!< Constructor
        /*!<
         *    @param ecm      (ElementContainerMatrix*) target data container
         *    @param filename (std::string)  Maskfile format1: xxx.txt, format2: xxx.xml
         */
    UtsusemiSetMask( ElementContainerMatrix* ecm, std::string filename, std::string key );
        //!< Constructor
        /*!<
         *    @param ecm      (ElementContainerMatrix*) target data container
         *    @param filename (std::string)  Maskfile format1: xxx.txt, format2: xxx.xml
         *    @param key      (std::string)  key of axis in data container on using format2
         */
    ~UtsusemiSetMask();
        //!< Destructor
        /*!<
         */

    bool ReadFile( std::string filename );
        //!< Reads Mask File
        /*!< bool ReadFile( std::string filename )
         *
         *   @param filename   (std::string) full path to mask file
         *   @retval true    : succeeded
         *   @retval false   : failed
         */
    void SetAxisKey( std::string a_key );
        //!< Sets Axis key in data container on using format2
        /*!< void SetAxisKey( std::string a_key )
         *
         *   @param key (std::string) key
         *   @retval None
         */
    bool Execute();
        //!< Executes Set Mask flags
        /*!< bool Execute()
         *
         *   @param None
         *   @retval true    : succeeded
         *   @retval false   : failed
         */
    void MaskRange( ElementContainer* ec, std::string key, std::vector<Double> axrange );
        //!< Masks given axis range
        /*!< void MaskRange( ElementContainer* ec, std::string key, std::vector<Double> axrange )
         *
         *   @param ec      (ElementContainer*) target data container
         *   @param key     (std::string) key of axis to be masked
         *   @param axrange (std::vector<Double>) [mask1_start,mask1_end,mask2_start,mask2_end,...]
         *   @retval None
         */
    void UnmaskAll();
        //!< Deletes all masked flag
        /*!< void UnmaskAll()
         *
         *   @param None
         *   @retval None
         */
    bool PutListOfTxtMask( std::vector< std::vector< std::vector<Double>* >* > *list );
};
#endif
