#ifndef UTSUSEMIDETECTOREFFICIENCYCORRECTION
#define UTSUSEMIDETECTOREFFICIENCYCORRECTION

#include "HeaderBase.hh"
#include "ElementContainer.hh"
#include "ElementContainerArray.hh"
#include "ElementContainerMatrix.hh"
#include "UtsusemiHeader.hh"
#include "MlfDetectorEfficiencyCorrection.hh"
#include "UtsusemiAnaEnvironReader.hh"

////////////////////////////////////////
// UtsusemiDetectorEfficiencyCorrection
////////////////////////////////////////

//! Detector Efficiency correction
/*!
 *  Usage: correction using approximate values stored in DetEffi.dat
 *      DF = UtsusemiDetectorEfficiencyCorrection()
 *      if DF.LoadDataFile( "path/to/DetEffi.dat" ):
 *          DF.Execute( dat )
 *
 *  Usage: Make DetEffi.dat
 *      DF = UtsusemiDetectorEfficiencyCorrection()
 *      if DF.Initialize( "path/to/DetectorInfo.xml", "SIK-SUS304-3He-PSD" ):
 *          DF.MakeApproximateData( "path/to/DetEffi.dat", <Ef_min>, <Ef_max>, <Ef_num>, <th_min>, <th_max>, <th_num> )
 *
 *  This class is based on MlfDetectorEfficiencyCorrection at rev301 or later
 *
 */


class UtsusemiDetectorEfficiencyCorrection
    :public MlfDetectorEfficiencyCorrection
{

private:
    std::string _MessageTag;

public:
    UtsusemiDetectorEfficiencyCorrection();
        //!< Constructor
        /*!<
         *    UtsusemiDetectorEfficiencyCorrection()
         *    @param None
         *    @return None
         */
    UtsusemiDetectorEfficiencyCorrection( std::string _detInfo, std::string _detName );
        //!< Constructor
        /*!<
         *    UtsusemiDetectorEfficiencyCorrection( std::string _detInfo, std::string _detName="" )
         *
         *    @param _detInfo (std::string) path to DetectorInfo.xml which must include <detectorStructure>.
         *    @param _detName (std::string) Detector Name like "SUS304-3He-PSD". This name must be included <detectorStructure>.
         *    @return None
         */
    UtsusemiDetectorEfficiencyCorrection( UInt4 _runNo, std::string _detName, UInt4 _modeNo=0 );
        //!< Constructor
        /*!<
         *    UtsusemiDetectorEfficiencyCorrection( UInt4 _runNo, std::string _detName, UInt4 _modeNo=0 );
         *
         *    @param _runNo   (UInt4)  runNo to find the suitable DetectorInfo.xml
         *    @param _detName (std::string) Detector Name like "SUS304-3He-PSD". This name must be included <detectorStructure>.
         *    @param _modNo   (UInt4)  mode number to choose DetectorInfo.xml
         *    @return None
         */
    UtsusemiDetectorEfficiencyCorrection( std::string _datFile );
        //!< Constructor
        /*!<
         *    UtsusemiDetectorEfficiencyCorrection( std::string _datFile )
         *
         *    @param _datFile (std::string) path to file storing approxmate correction values as matrix format.
         *    @return None
         */
    ~UtsusemiDetectorEfficiencyCorrection();
        //!< Destructor
        /*!<
         */
    bool Initialize( std::string _detInfo, std::string _detName );
        //!< Sets DetectorInfo and detNmae
        /*!<
         *    Sets DetectorInfo (including  <detectorStructure>) and detName
         *    to set parameters for calculations of detector efficiency.
         *
         *    @param _detInfo (std::string) path to DetectorInfo.xml which must include <detectorStructure>.
         *    @param _detName (std::string) Detector Name like "SUS304-3He-PSD". This name must be included <detectorStructure>.
         *    @retval  true   suceeded.
         *    @retval  false  failed.
         */
    bool Initialize( UInt4 _runNo, std::string _detName, UInt4 _modeNo=0 );
        //!< Sets DetectorInfo and detNmae
        /*!<
         *    Sets DetectorInfo (including  <detectorStructure>) and detName
         *    to set parameters for calculations of detector efficiency.
         *
         *    @param _runNo   (UInt4)  runNo to find the suitable DetectorInfo.xml
         *    @param _detName (std::string) Detector Name like "SUS304-3He-PSD". This name must be included <detectorStructure>.
         *    @param _modNo   (UInt4)  mode number to choose DetectorInfo.xml
         *    @retval  true   suceeded.
         *    @retval  false  failed.
         */
    bool LoadDataFile( std::string _datFile, std::string additionalPath="" );
        //!< Loads data from file storing approximate values of detector efficiency
        /*!<
         *    @param _datFile       (std::string)  path of data file to be saved
         *    @param additionalPath (std::string)  path of folder to be search _datFile
         *    @retval true  : succeeded.
         *    @retval false : failed.
         */
};

#endif
