#ifndef UTSUSEMICHECKDATAPROCESS
#define UTSUSEMICHECKDATAPROCESS

#include "UtsusemiHeader.hh"

/////////////////////////////////////
// UtsusemiCheckDataProcess
/////////////////////////////////////

//! Treat the process information of data reduction steps
/*!
 *  This enables to check the data reduction process of the data container.
 *  The header of the data container includes the information that what data reduction steps are done on the data.
 *  The data reduction functions can treat the process information by this class to avoid doing same data reduction on the data.
 */
class UtsusemiCheckDataProcess
{
private:
    std::string _ProcessTag;
    bool _CheckProcess( HeaderBase *hh, std::string process, bool isExact=true );
    void _AddProcess( HeaderBase *hh, std::string process );
    void _RemoveProcess( HeaderBase *hh, std::string process, bool isExact=true );
public:
    UtsusemiCheckDataProcess();
        //!< Constructor
        /*!<
         */
    ~UtsusemiCheckDataProcess();
        //!< Destructor
        /*!<
         */

    bool CheckProcess( ElementContainerMatrix *ecm, std::string process, bool isExact=true ){ return _CheckProcess( ecm->PutHeaderPointer(), process, isExact); }
        //!< Checks the process of the data (ElementContainerMatrix)
        /*!<
         *   @param ecm (ElementCotainerMatrix) The target data container.
         *   @param process (std::string) The key of the process.
         *   @param isExact (bool) If the given process key is required to be exactly correct, this is true. If there si ambiguity of the key, false.
         *   @retval true The process has already been done on the data
         *   @retval false The process has not yet
         */
    bool CheckProcess( ElementContainerArray *eca, std::string process, bool isExact=true ){ return _CheckProcess( eca->PutHeaderPointer(), process, isExact); }
        //!< Checks the process of the data (ElementContainerArray)
        /*!<
         *   @param ecm (ElementCotainerArray) The target data container.
         *   @param process (std::string) The key of the process.
         *   @param isExact (bool) If the given process key is required to be exactly correct, this is true. If there si ambiguity of the key, false.
         *   @retval true The process has already been done on the data
         *   @retval false The process has not yet
         */
    bool CheckProcess( ElementContainer *ec, std::string process, bool isExact=true ){ return _CheckProcess( ec->PutHeaderPointer(), process, isExact); }
        //!< Checks the process of the data (ElementContainer)
        /*!<
         *   @param ecm (ElementCotainer) The target data container.
         *   @param process (std::string) The key of the process.
         *   @param isExact (bool) If the given process key is required to be exactly correct, this is true. If there si ambiguity of the key, false.
         *   @retval true The process has already been done on the data
         *   @retval false The process has not yet
         */

    void AddProcess( ElementContainerMatrix *ecm, std::string process ){ _AddProcess( ecm->PutHeaderPointer(), process ); }
        //!< Add the process of the data to the data container (ElementContainerMatrix)
        /*!<
         *   @param ecm (ElementCotainerMatrix) The target data container.
         *   @param process (std::string) The key of the process.
         *   @retval None
         */
    void AddProcess( ElementContainerArray *eca, std::string process ){ _AddProcess( eca->PutHeaderPointer(), process ); }
        //!< Add the process of the data to the data container (ElementContainerArray)
        /*!<
         *   @param ecm (ElementCotainerArray) The target data container.
         *   @param process (std::string) The key of the process.
         *   @retval None
         */
    void AddProcess( ElementContainer *ec, std::string process ){ _AddProcess( ec->PutHeaderPointer(), process ); }
        //!< Add the process of the data to the data container (ElementContainer)
        /*!<
         *   @param ecm (ElementCotainer) The target data container.
         *   @param process (std::string) The key of the process.
         *   @retval None
         */
    void RemoveProcess( ElementContainerMatrix *ecm, std::string process, bool isExact=true ){ _RemoveProcess( ecm->PutHeaderPointer(), process, isExact); }
        //!< Remove the process of the data from the data container (ElementContainerMatrix)
        /*!<
         *   @param ecm (ElementCotainerMatrix) The target data container.
         *   @param process (std::string) The key of the process.
         *   @param isExact (bool) If the given process key is required to be exactly correct, this is true. If there si ambiguity of the key, false.
         *   @retval None
         */
    void RemoveProcess( ElementContainerArray *eca, std::string process, bool isExact=true ){ _RemoveProcess( eca->PutHeaderPointer(), process, isExact); }
        //!< Remove the process of the data from the data container (ElementContainerArray)
        /*!<
         *   @param ecm (ElementCotainerArray) The target data container.
         *   @param process (std::string) The key of the process.
         *   @param isExact (bool) If the given process key is required to be exactly correct, this is true. If there si ambiguity of the key, false.
         *   @retval None
         */
    void RemoveProcess( ElementContainer *ec, std::string process, bool isExact=true ){ _RemoveProcess( ec->PutHeaderPointer(), process, isExact); }
        //!< Remove the process of the data from the data container (ElementContainer)
        /*!<
         *   @param ecm (ElementCotainer) The target data container.
         *   @param process (std::string) The key of the process.
         *   @param isExact (bool) If the given process key is required to be exactly correct, this is true. If there si ambiguity of the key, false.
         *   @retval None
         */
};
#endif
