#ifndef SASDETECTOREFFICIENCYCORRECTION
#define SASDETECTOREFFICIENCYCORRECTION

#include "HeaderBase.hh"
#include "ElementContainer.hh"
#include "ElementContainerArray.hh"
#include "ElementContainerMatrix.hh"
#include "UtsusemiHeader.hh"
#include "MlfDetectorEfficiencyCorrection.hh"
#include "UtsusemiAnaEnvironReader.hh"

////////////////////////////////////////
// SASDetectorEfficiencyCorrection
////////////////////////////////////////

//! Detector Efficiency correction
/*!
 *  Usage: correction using approximate values stored in DetEffi.dat
 *      DF = SASDetectorEfficiencyCorrection()
 *      if DF.LoadDataFile( "path/to/DetEffi.dat" ):
 *          DF.Execute( dat )
 *
 *  Usage: Make DetEffi.dat
 *      DF = SASDetectorEfficiencyCorrection()
 *      if DF.Initialize( "path/to/DetectorInfo.xml", "SUS304-3He-PSD-SAS-6atm" ):
 *          DF.MakeApproximateData( "path/to/DetEffi.dat", <Ef_min>, <Ef_max>, <Ef_num>, <th_min>, <th_max>, <th_num> )
 *
 *  This class is based on MlfDetectorEfficiencyCorrection at rev301 or later
 *
 */


class SASDetectorEfficiencyCorrection
    :public MlfDetectorEfficiencyCorrection
{

private:
    std::string _MessageTag;
    UInt4 _NumOfMulTh;
public:
    SASDetectorEfficiencyCorrection();
    SASDetectorEfficiencyCorrection( std::string _detInfo, std::string _detName );
    SASDetectorEfficiencyCorrection( UInt4 _runNo, std::string _detName, UInt4 _modeNo=0 );
    SASDetectorEfficiencyCorrection( std::string _datFile );
        //!< Constructor
        /*!<
         *    SASDetectorEfficiencyCorrection()
         *    SASDetectorEfficiencyCorrection( std::string _detInfo, std::string _detName="" )
         *    SASDetectorEfficiencyCorrection( UInt4 _runNo, std::string _detName, UInt4 _modeNo=0 );
         *    SASDetectorEfficiencyCorrection( std::string _datFile )
         *
         *    @param _detInfo (std::string) path to DetectorInfo.xml which must include <detectorStructure>.
         *    @param _detName (std::string) Detector Name like "SUS304-3He-PSD". This name must be included <detectorStructure>.
         *    @param _runNo   (UInt4)  runNo to find the suitable DetectorInfo.xml
         *    @param _modNo   (UInt4)  mode number to choose DetectorInfo.xml
         *    @param _datFile (std::string) path to file storing approxmate correction values as matrix format.
         *    @return None
         */
    ~SASDetectorEfficiencyCorrection();
        //!< Destructor
        /*!<
         */
    bool Initialize( std::string _detInfo, std::string _detName );
    bool Initialize( UInt4 _runNo, std::string _detName, UInt4 _modeNo=0 );
        //!< Sets DetectorInfo and detNmae
        /*!<
         *    Sets DetectorInfo (including  <detectorStructure>) and detName
         *    to set parameters for calculations of detector efficiency.
         *
         *    @param _detInfo (std::string) path to DetectorInfo.xml which must include <detectorStructure>.
         *    @param _detName (std::string) Detector Name like "SUS304-3He-PSD". This name must be included <detectorStructure>.
         *    @param _runNo   (UInt4)  runNo to find the suitable DetectorInfo.xml
         *    @param _modNo   (UInt4)  mode number to choose DetectorInfo.xml
         *    @retval  true   suceeded.
         *    @retval  false  failed.
         */
    bool LoadDataFile( std::string _datFile, std::string additionalPath="" );
        //!< Loads data from file storing approximate values of detector efficiency
        /*!<
         *    @param _datFile       (std::string)  path of data file to be saved
         *    @param additionalPath (std::string)  path of folder to be search _datFile
         *    @retval true  : succeeded.
         *    @retval false : failed.
         */
    void Execute( ElementContainerMatrix* _ecm, UInt4 _bankId );
    Double GetEfficiencyApproximateVal(double lambda, double theta);
};

#endif
