#include "AdvVectorComplexTool.hh"

const string AdvVectorComplexTool::className=string("AdvVectorComplexTool");

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

////
Bool AdvVectorComplexTool::checkSrc(const Double* src, const UInt4 size) {
    string memberName = string("checkSrc(const Double*, UInt4)");

    if (src == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the source is NULL\n");
        return false;
    }
    if (size == 0) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the size of the source is equal to 0\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkSrc(const vector<Double>& src) {
    string memberName = string("checkSrc(const vector<Double>&)");

    if (src.empty()) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the source is empty\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkSrc(const gsl_vector* src) {
    string memberName = string("checkSrc(const gsl_vector>*)");
    if (src == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the source is NULL\n");
        return false;
    }
    if (src->size == 0) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the size of the source is equal to 0\n");
        return false;
    }
    return true;
}

////

Bool AdvVectorComplexTool::checkSrc(const Double* real, const Double* imag, const UInt4 size) {
    string memberName = string("checkSrc(const Double*, const Double*)");

    if (real == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the real part is NULL.\n");
        return false;
    }
    if (imag == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the imagenary part is NULL.\n");
        return false;
    }
    if (size == 0) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the size of two arrays is equal to 0.\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkSrc(const vector<Double>& real, const vector<Double>& imag) {
    string memberName = string("checkSrc(vector<Double>&, vector<Double>&)");

    if (real.empty()) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the real part is empty\n");
        return false;
    }
    if (imag.empty()) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the imagenary  part is empty\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkSrc(const gsl_vector* real, const gsl_vector* imag) {
    string memberName = string("checkSrc(const gsl_vector*, const gsl_vector*)");

    if (real == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the real part is NULL.\n");
        return false;
    }
    if (imag == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the imagenary  part is NULL.\n");
        return false;
    }
    if (real->size != imag->size) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the the sies of the real part is not eqal to the size of the imagenary part.\n");
        return false;
    }
    return true;
}

///

Bool AdvVectorComplexTool::checkSrc(const gsl_vector_complex* src) {
    string memberName = string("checkSrc(const gsl_vector_complex*)");
    
    if (src == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid gaument: the pointer to the souce is NULL.\n");
        return false;
    }
    if (src->size == 0) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid gaument: the size of the souce is equal to 0\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkSrc(const complex<Double>* src, const UInt4 size) {
    string memberName = string("checkSrc(const complex<Double>*, UInt4)");
    
    if (src == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid gaument: the pointer to the souce is NULL.\n");
        return false;
    }
    if (size == 0) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid gaument: the size of the souce is equal to 0\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkSrc(const vector< complex<Double> >& src) {
    string memberName = string("checkSrc(const vector< complex<Double> >&)");
    
    if (src.empty()) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid gaument: the souce is empty.\n");
        return false;
    }
    return true;
}

///

Bool AdvVectorComplexTool::checkDest(gsl_vector_complex* dest, const UInt4 srcSize) {
    string memberName = string("checkDest(gsl_vector_complex*, const UInt4)");

    if (dest == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the destination is NULL.\n");
        return false;
    }
    if (dest->size == 0) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the capacity of the destination is equal to 0.\n");
        return false;
    }
    if (dest->size < srcSize) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the capacity of the destination is smaller than the size of source.\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkDest(vector< complex<Double> >& dest, const UInt4 srcSize) {
    string memberName = string("checkDest(gsl_vector_complex*, const UInt4)");

    if (dest.capacity() == 0) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the capacity to the destination is equal to 0.\n");
        return false;
    }
    if (dest.capacity() < srcSize) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the capacity of the destination is smaller than the size of source.\n");
        return false;
    }
    return true;
}

Bool AdvVectorComplexTool::checkDest(complex<Double>* dest, const UInt4 srcSize) {
    string memberName = string("checkDest(vomplex<Double>*,  UInt4)");

    if (dest == NULL) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the pointer to the destination is NULL.\n");
        return false;
    }
    if (sizeof(dest)/sizeof(dest[0]) < srcSize) {
        this->errorMessage(className, memberName, __FILE__, __LINE__, "invalid argument: the capacity of the destination is smaller than the size of source.\n");
        return false;
    }
    return true;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AdvVectorComplexTool::initGslVectorComplex(const AdvVectorComplexTool::Part part, const Double* src, const UInt4 size, gsl_vector_complex* dest) {
    string memberName = string("initGslVectorComplex(AdvVectorComplexTool::Part, const Double*, const UInt4, gsl_vector_complex*)");

    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    gsl_vector_complex_set_zero(dest);
    gsl_vector_view view = (part==AdvVectorComplexTool::REAL) ? gsl_vector_complex_real(dest) : gsl_vector_complex_real(dest);
    for (UInt4 i=0; i<size; ++i) {
        gsl_vector_set(&view.vector, i, src[i]);
    }
}

void AdvVectorComplexTool::initGslVectorComplex(const AdvVectorComplexTool::Part part, const vector<Double>& src, gsl_vector_complex* dest) {
    string memberName = string("initGslVectorComplex(AdvVectorComplexTool::Part, const vector<Double>&, gsl_vector_complex*)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, src.size())) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    gsl_vector_complex_set_zero(dest);
    gsl_vector_view view = (part==AdvVectorComplexTool::REAL) ? gsl_vector_complex_real(dest) : gsl_vector_complex_real(dest);
    for (UInt4 i=0; i<src.size(); ++i) {
        gsl_vector_set(&view.vector, i, src.at(i));
    }
}

void AdvVectorComplexTool::initGslVectorComplex(const AdvVectorComplexTool::Part part, const gsl_vector* src, gsl_vector_complex* dest) {
    string memberName = string("initGslVectorComplex(AdvVectorComplexTool::Part, const gsl_vector*, gsl_vector_complex*)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, src->size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    } 

    gsl_vector_complex_set_zero(dest);
    gsl_vector_view view = (part==AdvVectorComplexTool::REAL) ? gsl_vector_complex_real(dest) : gsl_vector_complex_real(dest);
    gsl_vector_memcpy(&view.vector, src);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

gsl_vector_complex* AdvVectorComplexTool::createGslVectorComplex(const AdvVectorComplexTool::Part part, const Double* src, const UInt4 size) {
    string memberName = string("createGslVectorComplex(AdvVectorComplexTool::Part, const Double*, UInt4)");

    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    gsl_vector_complex* retval = gsl_vector_complex_alloc(size);
    this->initGslVectorComplex(part, src, size, retval);
    return retval;
}

gsl_vector_complex* AdvVectorComplexTool::createGslVectorComplex(const AdvVectorComplexTool::Part part, const vector<Double>& src) {
    string memberName = string("createGslVectorComplex(AdvVectorComplexTool::Part, const vector<Double>&)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    gsl_vector_complex* retval = gsl_vector_complex_alloc(src.size());
    this->initGslVectorComplex(part, src, retval);
    return retval;
}

gsl_vector_complex* AdvVectorComplexTool::createGslVectorComplex(const AdvVectorComplexTool::Part part, const gsl_vector* src) {
    string memberName = string("createGslVectorComplex(AdvVectorComplexTool::Part, const gsl_vector*)");

     if (! this->checkSrc(src)) {
         errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
         return NULL;
     }
     gsl_vector_complex* retval = gsl_vector_complex_alloc(src->size);
     this->initGslVectorComplex(part, src, retval);
     return retval;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AdvVectorComplexTool::initGslVectorComplex(const Double* real, const Double* imag, const UInt4 size, gsl_vector_complex* dest) {
    string memberName = string("initGslVectorComplex(const Double*, const Double*, UInt4, gsl_vector_complex*)");

    if (! this->checkSrc(real, imag, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    gsl_vector_view rr = gsl_vector_complex_real(dest);
    gsl_vector_view ri = gsl_vector_complex_imag(dest);
    for (UInt4 i=0; i<size; ++i) {
        gsl_vector_set(&rr.vector, i, real[i]);
        gsl_vector_set(&ri.vector, i, imag[i]);
    }
}

void AdvVectorComplexTool::initGslVectorComplex(const vector<Double>& real, const vector<Double>& imag, gsl_vector_complex* dest) {
    string memberName = string("initGslVectorComplex(vector<Double>&, vector<Double>&, gsl_vector_complex*)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, real.size())) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    gsl_vector_view rr = gsl_vector_complex_real(dest);
    gsl_vector_view ri = gsl_vector_complex_real(dest);
    for (UInt4 i=0; i<real.size(); ++i) {
        gsl_vector_set(&rr.vector, i, real.at(i));
        gsl_vector_set(&ri.vector, i, imag.at(i));
    }
}

void AdvVectorComplexTool::initGslVectorComplex(const gsl_vector* real, const gsl_vector* imag, gsl_vector_complex* dest) {
    string memberName = string("initGslVectorComplex(gsl_vector*, gsl_vector*, gsl_vector_complex*)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, real->size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    gsl_vector_view rr = gsl_vector_complex_real(dest);
    gsl_vector_view ri = gsl_vector_complex_imag(dest);
    gsl_vector_memcpy(&rr.vector, real);
    gsl_vector_memcpy(&ri.vector, imag);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

gsl_vector_complex* AdvVectorComplexTool::createGslVectorComplex(const Double* real, const Double* imag, const UInt4 size) {
    string memberName = string("createGslVectorComplex(const Double*, const Double*, UInt4)");

    if (! this->checkSrc(real, imag, size) ) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    gsl_vector_complex* retval = gsl_vector_complex_alloc(size);
    this->initGslVectorComplex(real, imag, size, retval);
    return retval;
}

gsl_vector_complex* AdvVectorComplexTool::createGslVectorComplex(const vector<Double>& real, const vector<Double>& imag) {
    string memberName = string("createGslVectorComplex(vector<Double>&, vector<Double>&)");

    if (! this->checkSrc(real, imag) ) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    gsl_vector_complex* retval = gsl_vector_complex_alloc(real.size());
    this->initGslVectorComplex(real, imag, retval);
    return retval;
}

gsl_vector_complex* AdvVectorComplexTool::createGslVectorComplex(const gsl_vector* real, const gsl_vector* imag) {
    string memberName = string("createGslVectorComplex(const gsl_vector*, const gsl_vector*)");

    if (! this->checkSrc(real, imag) ) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    gsl_vector_complex* retval = gsl_vector_complex_alloc(real->size);
    this->initGslVectorComplex(real, imag, retval);
    return retval;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AdvVectorComplexTool::initVectorComplex(const AdvVectorComplexTool::Part part, const Double* src, const UInt4 size, vector< complex<Double> >& dest) {
    string memberName = string("initVectorComplex(AdvVectorComplexTool::Part part, const Double*, UInt4, vector< complex<Double> >&)");

    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    switch (part) {
    case AdvVectorComplexTool::REAL:
        for (UInt4 i=0; i<size; ++i) {
            dest.at(i) = *(new complex<Double>(src[i], 0.0));
        }
        break;
    case AdvVectorComplexTool::IMAG:
        for (UInt4 i=0; i<size; ++i) {
            dest.at(i) = *(new complex<Double>(0.0, src[i]));
        }
        break;
    }
}

void AdvVectorComplexTool::initVectorComplex(const AdvVectorComplexTool::Part part, const vector<Double>& src, vector< complex<Double> >& dest) {
    string memberName = string("initVectorComplex(AdvVectorComplexTool::Part, vector<Double>&, vector< complex<Double> >&)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, src.size())) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    switch (part) {
    case AdvVectorComplexTool::REAL:
        for (UInt4 i=0; i<src.size(); ++i) {
            dest.at(i) = *(new complex<Double>(src.at(i), 0.0));
        }
        break;
    case AdvVectorComplexTool::IMAG:
        for (UInt4 i=0; i<src.size(); ++i) {
            dest.at(i) = *(new complex<Double>(0.0, src.at(i)));
        }
        break;
    }
}

void AdvVectorComplexTool::initVectorComplex(const AdvVectorComplexTool::Part part, const gsl_vector* src, vector< complex<Double> >& dest) {
    string memberName = string("initVectorComplex(AdvVectorComplexTool::Part, const gsl_vector*, vector< complex<Double> >&)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, src->size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    switch (part) {
    case AdvVectorComplexTool::REAL:
        for (UInt4 i=0; i<src->size; ++i) {
            dest.at(i) = *(new complex<Double>(gsl_vector_get(src, i), 0.0));
        }
        break;
    case AdvVectorComplexTool::IMAG:
        for (UInt4 i=0; i<src->size; ++i) {
            dest.at(i) = *(new complex<Double>(0.0, gsl_vector_get(src, i)));
        }
        break;
    }
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

vector< complex<Double> >* AdvVectorComplexTool::createVectorComplex(const AdvVectorComplexTool::Part part, const Double* src, const UInt4 size) {
    string memberName = string("createVectorComplex(AdvVectorComplexTool::Part part, const Double*, UInt4)");

    vector< complex<Double> >* retval = NULL;
    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    retval = new vector< complex<Double> >(size);
    this->initVectorComplex(part, src, size, *retval);
    return retval;
}

vector< complex<Double> >* AdvVectorComplexTool::createVectorComplex(const AdvVectorComplexTool::Part part, const vector<Double>& src) {
    string memberName = string("initVectorComplex(AdvVectorComplexTool::Part, vector<Double>&)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    vector< complex<Double> >* retval = new vector< complex<Double> >(src.size());
    this->initVectorComplex(part, src, *retval);
    return retval;
}

vector< complex<Double> >* AdvVectorComplexTool::createVectorComplex(const AdvVectorComplexTool::Part part, const gsl_vector* src) {
    string memberName = string("initVectorComplex(AdvVectorComplexTool::Part, const gsl_vector*)");

    vector< complex<Double> >* retval = NULL;
    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    retval = new vector< complex<Double> >(src->size);
    this->initVectorComplex(part, src, *retval);
    return retval;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AdvVectorComplexTool::initVectorComplex(const Double* real, const Double* imag, const UInt4 size, vector< complex<Double> >& dest) {
    string memberName = string("initVectorComplex(const Double*, const Double*, UInt4, vector< complex<Double> >&)");

    if (! this->checkSrc(real, imag, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    for (UInt4 i=0; i<size; ++i) {
        dest.at(i) = *(new complex<Double>(real[i], imag[i]));
    }
}

void AdvVectorComplexTool::initVectorComplex(const vector<Double>& real, const vector<Double>& imag, vector< complex<Double> >& dest) {
    string memberName = string("initVectorComplex(vector<Double>&, vector<Double>&, vector< complex<Double> >&)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, real.size())) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    for (UInt4 i=0; i<real.size(); ++i) {
        dest.at(i) = *(new complex<Double>(real.at(i), imag.at(i)));
    }
}

void AdvVectorComplexTool::initVectorComplex(const gsl_vector* real, const gsl_vector* imag, vector< complex<Double> >& dest) {
    string memberName = string("initVectorComplex(const gsl_vector*, const gsl_vector*, vector< complex<Double> >&)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, real->size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    for (UInt4 i=0; i<real->size; ++i) {
        dest.at(i) = *(new complex<Double>(gsl_vector_get(real, i), gsl_vector_get(imag, i)));
    }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

vector< complex<Double> >* AdvVectorComplexTool::createVectorComplex(const Double* real, const Double *imag, const UInt4 size) {
    string memberName = string("initVectorComplex(const Double*, const Double*, UInt4)");

    if (! this->checkSrc(real, imag, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    vector< complex<Double> >* retval = new vector< complex<Double> >(size);
    this->initVectorComplex(real, imag, size, *retval);
    return retval;
}

vector< complex<Double> >* AdvVectorComplexTool::createVectorComplex(const vector<Double>& real, const vector<Double>& imag) {
    string memberName = string("initVectorComplex(vector<Double>&, vector<Double>&)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    vector< complex<Double> >* retval = new vector< complex<Double> >(real.size());
    this->initVectorComplex(real, imag, *retval);
    return retval;
}

vector< complex<Double> >* AdvVectorComplexTool::createVectorComplex(const gsl_vector* real, const gsl_vector* imag) {
    string memberName = string("initVectorComplex(const gsl_vector*, const gsl_vector*)");

    if (! this->checkSrc(real, imag )) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid surce\n");
        return NULL;
    }
    vector< complex<Double> >* retval = new vector< complex<Double> >(real->size);
    this->initVectorComplex(real, imag, *retval);
    return retval;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AdvVectorComplexTool::initArrayComplex(const AdvVectorComplexTool::Part part, const Double* src, const UInt4 size, complex<Double>* dest) {
    string memberName = string("initArrayComplex(AdvVectorComplexTool::Part, const Double*, UINt4, complex<Double>*)");

    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    switch (part) {
    case AdvVectorComplexTool::REAL:
        for (UInt4 i=0; i<size; ++i) {
            dest[i] = *(new complex<Double>(src[i], 0.0));
        }
        break;
    case AdvVectorComplexTool::IMAG:
        for (UInt4 i=0; i<size; ++i) {
            dest[i] = *(new complex<Double>(0.0, src[i]));
        }
        break;
    }
}

void AdvVectorComplexTool::initArrayComplex(const AdvVectorComplexTool::Part part, const vector<Double>& src, complex<Double>* dest) {
    string memberName = string("initArrayComplex(AdvVectorComplexTool::Part, vector<Double>&, complex<Double>*)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, src.size())) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    switch (part) {
    case AdvVectorComplexTool::REAL:
        for (UInt4 i=0; i<src.size(); ++i) {
            dest[i] = *(new complex<Double>(src[i], 0.0));
        }
        break;
    case AdvVectorComplexTool::IMAG:
        for (UInt4 i=0; i<src.size(); ++i) {
            dest[i] = *(new complex<Double>(0.0, src[i]));
        }
        break;
    }
}

void AdvVectorComplexTool::initArrayComplex(const AdvVectorComplexTool::Part part, const gsl_vector* src, complex<Double>* dest) {
    string memberName = string("initArrayComplex(AdvVectorComplexTool::Part, const gsl_vector*, complex<Double>*)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, src->size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    switch (part) {
    case AdvVectorComplexTool::REAL:
        for (UInt4 i=0; i<src->size; ++i) {
            dest[i] = *(new complex<Double>(gsl_vector_get(src, i), 0.0));
        }
        break;
    case AdvVectorComplexTool::IMAG:
        for (UInt4 i=0; i<src->size; ++i) {
            dest[i] = *(new complex<Double>(0.0, gsl_vector_get(src, i)));
        }
        break;
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

complex<Double>* AdvVectorComplexTool::createArrayComplex(const AdvVectorComplexTool::Part part, const Double* src, UInt4 size) {
    string memberName = string("createArrayComplex(AdvVectorComplexTool::Part, const Double*, UINt4)");

    if ( this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    complex<Double>* retval = new complex<Double>[size];
    this->initArrayComplex(part, src, size, retval);
    return retval;
}

complex<Double>* AdvVectorComplexTool::createArrayComplex(const AdvVectorComplexTool::Part part, const vector<Double>& src) {
    string memberName = string("createArrayComplex(AdvVectorComplexTool::Part, vector<Double>&*)");

    if (! this->checkSrc(src) ) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    complex<Double>* retval = new complex<Double>[src.size()];
    this->initArrayComplex(part, src, retval);
    return retval;
}

complex<Double>* AdvVectorComplexTool::createArrayComplex(const AdvVectorComplexTool::Part part, const gsl_vector* src) {
    string memberName = string("createArrayComplex(AdvVectorComplexTool::Part, const gsl_vector*)");

    if (! this->checkSrc(src) ) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    complex<Double>* retval = new complex<Double>[src->size];
    this->initArrayComplex(part, src, retval);
    return retval;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void AdvVectorComplexTool::initArrayComplex(const Double* real, const Double* imag, const UInt4 size, complex<Double>* dest) {
    string memberName = string("initArrayComplex(const Double*, const Double*, UInt4, complex<Double>*");

    if (! this->checkSrc(real, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    for (UInt4 i=0; i<size; ++i) {
        dest[i] = *(new complex<Double>(real[i], imag[i]));
    }
}

void AdvVectorComplexTool::initArrayComplex(const vector<Double>& real, const vector<Double>& imag, complex<Double>* dest) {
    string memberName = string("initArrayComplex(vector<Double>&, vector<Double>&, complex<Double>*)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, real.size())) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    for (UInt4 i=0; i<real.size(); ++i) {
        dest[i] = *(new complex<Double>(real.at(i), imag.at(i)));
    }
}

void AdvVectorComplexTool::initArrayComplex(const gsl_vector* real, const gsl_vector* imag, complex<Double>* dest) {
    string memberName = string("initArrayComplex(const gsl_vector* real, const gsl_vector* imag, complex<Double>*)");

    if (! this->checkSrc(real, imag) ) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return;
    }
    if (! this->checkDest(dest, real->size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid destination\n");
        return;
    }

    for (UInt4 i=0; i<real->size; ++i) {
        dest[i] = *(new complex<Double>(gsl_vector_get(real, i), gsl_vector_get(imag, i)));
    }
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

complex<Double>* AdvVectorComplexTool::createArrayComplex(const Double* real, const Double* imag, const UInt4 size) {
    string memberName = string("createArrayComplex(const Double* real, const Double* imag)");

    if (! this->checkSrc(real, imag, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    complex<Double>* retval = new complex<Double>[size];
    this->initArrayComplex(real, imag, size, retval);
    return retval;
}

complex<Double>* AdvVectorComplexTool::createArrayComplex(const vector<Double>& real, const vector<Double>& imag) {
    string memberName = string("createArrayComplex(vector<Double>&, vector<Double>&)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    complex<Double>* retval = new complex<Double>[real.size()];
    this->initArrayComplex(real, imag, retval);
    return retval;
}

complex<Double>* AdvVectorComplexTool::createArrayComplex(const gsl_vector* real, const gsl_vector* imag) {
    string memberName = string("createArrayComplex(const gsl_vector*, const gsl_vector*)");

    if (! this->checkSrc(real, imag)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    complex<Double> *retval = new complex<Double>[real->size];
    this->initArrayComplex(real, imag, retval);
    return retval;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

gsl_vector_complex* AdvVectorComplexTool::VectorComplexToGslVectorComplex(const vector< complex<Double> >& v) {
    string memberName = string("VectorComplexToGslVectorComplex(vector< complex<Double> >&)");

    if (! this->checkSrc(v)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    gsl_vector_complex *retval = gsl_vector_complex_alloc(v.size());
    gsl_vector_view rr = gsl_vector_complex_real(retval);
    gsl_vector_view ri = gsl_vector_complex_imag(retval);
    for (UInt4 i=0; i<v.size(); ++i) {
        gsl_vector_set(&rr.vector, i, v.at(i).real());
        gsl_vector_set(&ri.vector, i, v.at(i).imag());
    }
    return retval;
}

gsl_vector_complex* AdvVectorComplexTool::ArrayComplexToGslVectorComplex(const complex<Double>* v, UInt4 size) {
    string memberName = string("ArrayComplexToGslVectorComplex(const complex<Double>*, UInt4)");

    if (! this->checkSrc(v, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    gsl_vector_complex *retval = gsl_vector_complex_alloc(size);
    gsl_vector_view rr = gsl_vector_complex_real(retval);
    gsl_vector_view ri = gsl_vector_complex_imag(retval);
    for (UInt4 i=0; i<size; ++i) {
        gsl_vector_set(&rr.vector, i, v[i].real());
        gsl_vector_set(&ri.vector, i, v[i].imag());
    }
    return retval;
}

vector< complex<Double> >* AdvVectorComplexTool::GslVectorComplexToVectorComplex(const gsl_vector_complex* v) {
    string memberName = string("GslVectorComplexToVectorComplex(const gsl_vector_complex* v)");

    if (! this->checkSrc(v)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    vector< complex<Double> >* retval = new vector< complex<Double> >(v->size);
    for (UInt4 i=0; i<v->size; ++i) {
        const gsl_complex *zp = gsl_vector_complex_const_ptr(v, i);
        retval->at(i) = *(new complex<Double>(GSL_REAL(*zp), GSL_IMAG(*zp)));
    }
    return retval;
}

vector< complex<Double> >* AdvVectorComplexTool::ArrayComplexToVectorComplex(const complex<Double>* v, UInt4 size) {
    string memberName = string("ArrayComplexToVectorComplex(const complex<Double>*, UInt4)");

    if (! this->checkSrc(v, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    vector< complex<Double> >* retval = new vector< complex<Double> >(size);
    for (UInt4 i=0; i<size; ++i) {
        retval->at(i) = v[i];
    }
    return retval;
}

complex<Double>* AdvVectorComplexTool::VectorComplexToArrayComplex(const vector< complex<Double> >& v) {
    string memberName = string("VectorComplexToArrayComplex(vector< complex<Double> >& v)");

    if (! this->checkSrc(v)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    complex<Double>* retval = new complex<Double>[v.size()];
    for (UInt4 i=0; i<v.size(); ++i) {
        retval[i] = v.at(i);
    }
    return retval;
}

complex<Double>* AdvVectorComplexTool::GslVectorComplexToArrayComplex(const gsl_vector_complex* v) {
    string memberName = string("GslVectorComplexToArrayComplex(const gsl_vector_complex* v)");

    if (! this->checkSrc(v)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    complex<Double>* retval = new complex<Double>[v->size];
    for (UInt4 i=0; i < v->size; ++i) {
        const gsl_complex *zp = gsl_vector_complex_const_ptr(v, i);
        retval[i] = *(new complex<Double>(GSL_REAL(*zp), GSL_IMAG(*zp)));
    }
    return retval;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

gsl_vector* AdvVectorComplexTool::getPartAsGslVector(const AdvVectorComplexTool::Part part, const gsl_vector_complex* src) {
    string memberName = string("getPartAsGslVector(AdvVectorComplexTool::Part, const gsl_vector_complex*)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }
    gsl_vector* retval = gsl_vector_alloc(src->size);
    gsl_vector_const_view view = (part == AdvVectorComplexTool::REAL ? gsl_vector_complex_const_real(src) : gsl_vector_complex_const_imag(src));
    gsl_vector_memcpy(retval, &view.vector);
    return retval;
}

gsl_vector* AdvVectorComplexTool::getPartAsGslVector(const AdvVectorComplexTool::Part part, complex<Double>* src, const UInt4 size) {
    string memberName = string("getPartAsGslVector(AdvVectorComplexTool::Part, complex<Double>*, UInt4)");

    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    Double& (complex<Double>::*fp)();
    switch (part) {
    case AdvVectorComplexTool::REAL:
        fp = &complex<Double>::real;
        break;
    case AdvVectorComplexTool::IMAG:
        fp = &complex<Double>::imag;
        break;
    }

    gsl_vector* retval = gsl_vector_alloc(size);
    for (UInt4 i=0; i<size; ++i) {
        gsl_vector_set(retval, i, (src[i].*fp)());
    }
    return retval;
}

gsl_vector* AdvVectorComplexTool::getPartAsGslVector(const AdvVectorComplexTool::Part part, vector< complex<Double> >& src) {
    string memberName = string("getPartAsGslVector(AdvVectorComplexTool::Part, vector< complex<Double> >&)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    Double& (complex<Double>::*fp)();
    switch (part) {
    case AdvVectorComplexTool::REAL:
        fp = &complex<Double>::real;
        break;
    case AdvVectorComplexTool::IMAG:
        fp = &complex<Double>::imag;
        break;
    }

    gsl_vector *retval = gsl_vector_alloc(src.size());
    for (UInt4 i=0; i<src.size(); ++i) {
        gsl_vector_set(retval, i, (src.at(i).*fp)());
    }
    return retval;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

Double* AdvVectorComplexTool::getPartAsArray(const AdvVectorComplexTool::Part part, const gsl_vector_complex* src) {
    string memberName = string("getPartAsArray(AdvVectorComplexTool::Part, const gsl_vector_complex*)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    Double *retval = new Double[src->size];
    gsl_vector_const_view view = ((part==AdvVectorComplexTool::REAL) ? gsl_vector_complex_const_real(src) : gsl_vector_complex_const_imag(src));
    for (UInt4 i=0; i<src->size; ++i) {
        retval[i] = gsl_vector_get(&view.vector, i);
    }
    return retval;
}

Double* AdvVectorComplexTool::getPartAsArray(const AdvVectorComplexTool::Part part, complex<Double>* src, const UInt4 size) {
    string memberName = string("getPartAsArray(AdvVectorComplexTool::Part, complex<Double>*src, UInt4)");

    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    Double& (complex<Double>::*fp)();
    switch (part) {
    case AdvVectorComplexTool::REAL:
        fp = &complex<Double>::real;
        break;
    case AdvVectorComplexTool::IMAG:
        fp = &complex<Double>::imag;
        break;
    }

    Double *retval = new Double[size];
    for (UInt4 i=0; i<size; ++i) {
        retval[i] = (src[i].*fp)();
    }
    return retval;
}

Double* AdvVectorComplexTool::getPartAsArray(const AdvVectorComplexTool::Part part, vector< complex<Double> >& src) {
    string memberName = string("getPartAsArray(AdvVectorComplexTool::Part, vector< complex<Double> >&)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    Double& (complex<Double>::*fp)();
    switch (part) {
    case AdvVectorComplexTool::REAL:
        fp = &complex<Double>::real;
        break;
    case AdvVectorComplexTool::IMAG:
        fp = &complex<Double>::imag;
        break;
    }

    Double *retval = new Double[src.size()];
    for (UInt4 i=0; i<src.size(); ++i) {
        retval[i] = (src.at(i).*fp)();
    }
    return retval;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

vector<Double>* AdvVectorComplexTool::getPartAsVector(const AdvVectorComplexTool::Part part, const gsl_vector_complex* src) {
    string memberName = string("getPartAsVector(AdvVectorComplexTool::Part, const gsl_vector_complex*)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    vector<Double> *retval=new vector<Double>(src->size);
    gsl_vector_const_view view = (part==AdvVectorComplexTool::REAL ? gsl_vector_complex_const_real(src) : gsl_vector_complex_const_imag(src));
    for (UInt4 i=0; i<src->size; ++i) {
        retval->at(i)=gsl_vector_get(&view.vector, i);
    }
    return retval;
}

vector<Double>* AdvVectorComplexTool::getPartAsVector(const AdvVectorComplexTool::Part part, complex<Double>* src, const UInt4 size) {
    string memberName = string("getPartAsVector(AdvVectorComplexTool::Part, complex<Double>*, UInt4)");

    if (! this->checkSrc(src, size)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    Double& (complex<Double>::*fp)();
    switch (part) {
    case AdvVectorComplexTool::REAL:
        fp = &complex<Double>::real;
        break;
    case AdvVectorComplexTool::IMAG:
        fp = &complex<Double>::imag;
        break;
    }

    vector<Double> *retval = new vector<Double>(size);
    for (UInt4 i=0; i<size; ++i) {
        retval->at(i) = (src[i].*fp)();
    }
    return retval;
}

vector<Double>* AdvVectorComplexTool::getPartAsVector(const AdvVectorComplexTool::Part part, vector< complex<Double> >& src) {
    string memberName = string("getPartAsVector(AdvVectorComplexTool::Part, vector< complex<Double> >&)");

    if (! this->checkSrc(src)) {
        errorMessage(className, memberName, __FILE__, __LINE__, "invalid source\n");
        return NULL;
    }

    Double& (complex<Double>::*fp)();
    switch (part) {
    case AdvVectorComplexTool::REAL:
        fp = &complex<Double>::real;
        break;
    case AdvVectorComplexTool::IMAG:
        fp = &complex<Double>::imag;
        break;
    }

    vector<Double> *retval = new vector<Double>(src.size());
    for (UInt4 i=0; i<src.size(); ++i) {
        retval->at(i) = (src.at(i).*fp)();
    }
    return retval;
}
