#ifndef CONTAINER_FOR_MATRIX
#define CONTAINER_FOR_MATRIX

#include "Header.hh"
#include "StringTools.hh"

#include "ContainerBaseTemplate.hh"
#include "ContainerForVector.hh"

template<typename _SCALER_TYPE >
struct ValueColumnSizeComp {
    Bool operator() (std::pair<const string, _SCALER_TYPE >& item1, std::pair<const string, _SCALER_TYPE >& item2) {
        return item1.second.at(0).size() < item2.second.at(0).size();
    };
};


template<typename _ELEMENT_TYPE >
class AdvContainerForMatrix : public ContainerForScaler< vector< vector< _ELEMENT_TYPE > > >{

    public:
        /** constructor */
        AdvContainerForMatrix< _ELEMENT_TYPE >() { this->clear(); };

        /** destructor */
        ~AdvContainerForMatrix< _ELEMENT_TYPE >() { this->clear(); };

        /** get the row of the matrix with the specified key.
         *  @param[in] key  the key for an entry
         *  @param[in] i    the index of the row
         */
        vector< _ELEMENT_TYPE > getRow(const string& key, const UInt4 i) throw(invalid_argument, out_of_range);

        /** get the row of the matrix with the specified key.
         *  @param[in] key  the key for an entry
         *  @param[in] j    the index of the column
         */
        vector< _ELEMENT_TYPE > getCol(const string& key, const UInt4 j) throw(invalid_argument, out_of_range);

        /** get the (i, j) element of the matrix with the specified key.
         *  @param[in] key  the key for an entry
         *  @param[in] i    the row index
         *  @param[in] j    the column index
         */
        _ELEMENT_TYPE get(const string& key, const UInt4 i, const UInt4 j) throw(invalid_argument, out_of_range) ;

        /** replace the specified row to a new row.
         *  @param[in] key  the key for an entry
         *  @param[in] i    the row index
         *  @param[in] v    the new value
         */
        void replaceRow(const string& key, const UInt4 i, const vector< _ELEMENT_TYPE >& v) ;

        /** replace the specified column to a new column.
         *  @param[in] key  the key for an entry
         *  @param[in] i    the column index
         *  @param[in] v    the new value
         */
        void replaceCol(const string& key, const UInt4 j, const vector< _ELEMENT_TYPE >& v) ;

        /** replace the value of the specified element to a new value.
         *  @param[in] key  the key for an entry
         *  @param[in] i    the column index
         *  @param[in] i    the column index
         *  @param[in] v    the new value
         */
        void replace(const string& key, const UInt4 i, const UInt4 j, const _ELEMENT_TYPE v) ;

        /** replace the specified cloumn to a new column.
         *  @param[in] key  the key for an entry
         *  @param[in] i    the row index
         *  @param[in] j    the column index
         */

        /** */
        void dump(const UInt4 indent=0, const UInt4 indentDepth=4, const string& keyTitle="key", const string& sizeTitle="size", const string& valueTitle="value");
};

template<typename _ELEMENT_TYPE >
vector< _ELEMENT_TYPE > AdvContainerForMatrix< _ELEMENT_TYPE >::getRow(const string& key, const UInt4 i) throw(invalid_argument, out_of_range) {
    if (! this->contain(key)) {
        std::cerr << "Error: AdvContainerForMatrix::get: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"." << endl;
        throw invalid_argument(string("not found the entry with the specified key \"") + key + string("\""));
    }
    if ( i >= this->cmap[key].size() ) {
        StringTools tools;
        std::cerr << "Error: AdvContainerForMatrix::get: " << __FILE__ << ":" << __LINE__ << ": the row index " << i << " is out of range: key="<< key << endl;
        throw out_of_range(string("the row index ") + tools.UInt4ToString(i) + string(" is out of range, key=\"") + key + string("\""));
    }
    return this->cmap[key].at(i);
};

template<typename _ELEMENT_TYPE >
vector< _ELEMENT_TYPE > AdvContainerForMatrix< _ELEMENT_TYPE >::getCol(const string& key, const UInt4 j) throw(invalid_argument, out_of_range) {
    if (! this->contain(key)) {
        std::cerr << "Error: AdvContainerForMatrix::get: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"." << endl;
        throw invalid_argument(string("not found the entry with the specified key \"") + key + string("\""));
    }
    if (j >= this->cmap[key].at(0).size()) {
        StringTools tools;
        std::cerr << "Error: AdvContainerForMatrix::get: " << __FILE__ << ":" << __LINE__ << ": the column index " << j << " is out of range: key=\""<< key << "\"" << endl;
        throw out_of_range(string("the column index ") + tools.UInt4ToString(j) + string(" is out of range, key=\"") + key + string("\""));
    }

    vector< _ELEMENT_TYPE > *col = new vector< _ELEMENT_TYPE >(this->cmap[key].size());
    for (UInt4 i=0; i<this->cmap[key].size(); ++i) {
        col->at(i) = this->cmap[key].at(i).at(j);
    }
    return *col;
}

template<typename _ELEMENT_TYPE >
_ELEMENT_TYPE AdvContainerForMatrix< _ELEMENT_TYPE >::get(const string& key, const UInt4 i, const UInt4 j) throw(invalid_argument, out_of_range) {
    if (! this->contain(key)) {
        std::cerr << "Error: AdvContainerForMatrix::get: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"." << endl;
        throw invalid_argument(string("not found the entry with the specified key \"") + key + string("\""));
    }
    if (i >= this->cmap[key].size()) {
        StringTools tools;
        std::cerr << "Error: AdvContainerForMatrix::get: " << __FILE__ << ":" << __LINE__ << ": the row index " << i << " is out of range: key="<< key << endl;
        throw out_of_range(string("the row index ") + tools.UInt4ToString(i) + string(" is out of range, key=\"") + key + string("\""));
    }
    if (j >= this->cmap[key].at(0).size()) {
        StringTools tools;
        std::cerr << "Error: AdvContainerForMatrix::get: " << __FILE__ << ":" << __LINE__ << ": the column index " << j << " is out of range: key=\"" << key << "\"" << endl;
        throw out_of_range(string("the column index ") + tools.UInt4ToString(j) + string(" is out of range, key=\"") + key + string("\""));
    }
    return this->cmap[key].at(i).at(j);
};

template<typename _ELEMENT_TYPE>
void AdvContainerForMatrix< _ELEMENT_TYPE >::replaceRow(const string& key, const UInt4 i, const vector< _ELEMENT_TYPE >& v) {
    if (! this->contain(key)) {
        std::cerr << "Warning: AdvContainerForMatrix::replaceRow: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"." << endl;
    } else if (i >= this->cmap[key].size()) {
        std::cerr << "Warning: AdvContainerForMatrix::replaceRow: " << __FILE__ << ":" << __LINE__ << ": the row index " << i << " is out of range: key=\""<< key << "\"" << endl;
    } else if ( v.size() != this->cmap[key].at(i).size() ) {
        std::cerr << "Warning: AdvContainerForMatrix::replaceRow: " << __FILE__ << ":" << __LINE__ << ": the size of given new row is not equal to the size of the specified row." << endl;
    } else {
        this->cmap[key].at(i) = v;
    }
};

template<typename _ELEMENT_TYPE>
void AdvContainerForMatrix< _ELEMENT_TYPE >::replaceCol(const string& key, const UInt4 j, const vector< _ELEMENT_TYPE >& v) {
    if (! this->contain(key)) {
        std::cerr << "Warning: AdvContainerForMatrix::replaceCol: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"." << endl;
    } else if (j >= this->cmap[key].size()) {
        std::cerr << "Warning: AdvContainerForMatrix::replaceCol: " << __FILE__ << ":" << __LINE__ << ": the column index " << j << " is out of range: key=\""<< key << "\"" << endl;
    } else if ( v.size() != this->cmap[key].size() ) {
        std::cerr << "Warning: AdvContainerForMatrix::replaceCol: " << __FILE__ << ":" << __LINE__ << ": the size of the given new column is not equal to the size of the specified column." << endl;
    } else {
        for (UInt4 i=0; i<this->cmap[key].size(); ++i) {
            this->cmap[key].at(i).at(j) = v.at(i);
        }
    }
};

template<typename _ELEMENT_TYPE>
void AdvContainerForMatrix< _ELEMENT_TYPE >::replace(const string& key, const UInt4 i, const UInt4 j, const _ELEMENT_TYPE v) {
    if (! this->contain(key)) {
        std::cerr << "Warning: AdvContainerForMatrix::replace: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"." << endl;
    } else if (i >= this->cmap[key].size()) {
        std::cerr << "Warning: AdvContainerForMatrix::replace: " << __FILE__ << ":" << __LINE__ << ": the row index " << i << " is out of range: key=\""<< key << "\"" << endl;
    } else if (j >= this->cmap[key].size()) {
        std::cerr << "Warning: AdvContainerForMatrix::replace: " << __FILE__ << ":" << __LINE__ << ": the row index " << j << " is out of range: key=\""<< key << "\"" << endl;
    } else {
        this->cmap[key].at(i).at(j) = v;
    }
};


template<typename _ELEMENT_TYPE >
void AdvContainerForMatrix< _ELEMENT_TYPE >::dump(const UInt4 indent, const UInt4 indentDepth, const string& keyTitle, const string& sizeTitle, const string& valueTitle) {

    if (! this->cmap.empty()) {
        size_t maxKeyWidth = max(keyTitle.size(), max_element(this->cmap.begin(), this->cmap.end(), KeyLengthComp< vector< vector< _ELEMENT_TYPE > > >())->first.size());

        size_t maxRowWidth = max_element(this->cmap.begin(), this->cmap.end(), ValueSizeComp<       vector< vector< _ELEMENT_TYPE > > >())->second.size();
        size_t maxColWidth = max_element(this->cmap.begin(), this->cmap.end(), ValueColumnSizeComp< vector< vector< _ELEMENT_TYPE > > >())->second.at(0).size();
        maxRowWidth = maxRowWidth > 0 ? static_cast<size_t>(floor(log10(static_cast<Double>(maxRowWidth)))+1) : 1;
        maxColWidth = maxColWidth > 0 ? static_cast<size_t>(floor(log10(static_cast<Double>(maxColWidth)))+1) : 1;

        OutputTypeTitle ott;
        ott.title(indentDepth*indent, typeid(vector< vector< _ELEMENT_TYPE > >));
        ott.header(indentDepth*(indent+1), maxKeyWidth, keyTitle, maxRowWidth+maxColWidth+1, sizeTitle, valueTitle);

        for_each(this->cmap.begin(), this->cmap.end(), OutputMatrixEntry< _ELEMENT_TYPE >(indentDepth*(indent+1), maxKeyWidth, maxRowWidth, maxColWidth));
        std::cout << endl;
    }
};

#endif // CONTAINER_FOR_MATRIX
