#include <cassert>
#include <cmath>
/* gnu scientific library */
#include "gsl/gsl_sf_gamma.h"
#include "gsl/gsl_sf_psi.h"
/* Manyo lib */
#include "AdvAugmentedLorentzian.hh"

/**
 *  constructor
 */
AdvAugmentedLorentzian::AdvAugmentedLorentzian() : AdvFuncBase(string("augmented lorentzian"), string("al"), NumberOfParamsForAugmentedLorentzian) {
}

/**
 *  constructor
 */
AdvAugmentedLorentzian::~AdvAugmentedLorentzian() {
}

/**
 *  normarisation factor of the function
 *
 *  \param[in] p the power number
 *
 *  the value of the numarisation factor is the inverse number of Beta function B(p-0.5, 0.5).
 *  the Beta function can be represented using the Ganna Function; B(p, q) = G(p)G(q)/G(p+q).
 */
Double AdvAugmentedLorentzian::N(const Double p) {
    return gsl_sf_gamma(p)/(gsl_sf_gamma(p-0.5)*gsl_sf_gamma(0.5));
}

// /**
//  *  denominator of the function
//  *
//  *  \param[in] x  argument
//  *  \param[in] c  the center of the function
//  *  \param[in] w  the width of the function
//  *  \param[in] p  the power number of the 
//  *  \param[in] m  the order of diffrentiation
//  */
// Double AdvAugmentedLorentzian::denominator(const Double x, const Double c, const Double w, const Double p, const Double m) {
//     assert( w != 0.0 );
//     Double t=(x-c)/w;
//     return pow(w, 1.0+m)*pow(t*t + 1.0, p+m);
// }

/**
 *  evaluate the value of the function
 *
 *  \param[in] x  argument
 *  \param[in] h  the height of the function
 *  \param[in] c  the center of the function
 *  \param[in] w  the width of the function
 *  \param[in] p  the power number of the 
 */
Double AdvAugmentedLorentzian::eval(const Double x, const Double h, const Double c, const Double w, const Double p) {
    assert( w != 0.0 );
    //return h*N(p)/denominator(x, c, w, p, 0.0);
    Double t=(x-c)/w;
    return h*N(p)/(w*pow(t*t+1.0, p));
}

#ifdef HAVE_DIFFERENTIAL_MEMBER
/**
 *  evaluate the value of the 1st. differentiation coefficient of the function, dL/dx
 *
 *  \param[in] x  argument
 *  \param[in] h  the height of the function
 *  \param[in] c  the center of the function
 *  \param[in] w  the width of the function
 *  \param[in] p  the power number of the 
 */
Double AdvAugmentedLorentzian::der1st(const Double x, const Double h, const Double c, const Double w, const Double p) {
    assert( w != 0.0 );
    Double t=(x-c)/w;
    //return -2.0*h*p*N(p)*t*denominator(x, c, w, p, 2.0);
    return -2.0*((p-1.0/2.0)/w)*t*(this->eval(x, h, c, w, p+1.0));
}

/**
 *  evaluate the value of the 2nd. differentiation coefficient of the function, d^2L/dx^2
 *
 *  \param[in] x  argument
 *  \param[in] h  the height of the function
 *  \param[in] c  the center of the function
 *  \param[in] w  the HWHM of the function
 *  \param[in] p  the power number of the 
 */
Double AdvAugmentedLorentzian::der2nd(const Double x, const Double h, const Double c, const Double w, const Double p) {
    assert( w != 0.0 );
    Double t=(x-c)/w;
    //return 2.0*h*p*N(p)*((2.0*p+1.0)*t*t-1.0)*denominator(x, c, w, p, 2.0);
    return 2.0*((p-1.0/2.0)*(p+1.0-1.0/2.0)/(p+1.0))/(w*w)*((2.0*p+1.0)*t*t-1.0)*(this->eval(x, h, c, w, p+2.0));
}

/**
 *  evaluate the value of the differentiation coefficient for w, dL/dw
 *
 *  \param[in] x  argument
 *  \param[in] h  the height of the function
 *  \param[in] c  the center of the function
 *  \param[in] w  the HWHM of the function
 *  \param[in] p  the power number of the 
 */
Double AdvAugmentedLorentzian::derW(const Double x, const Double h, const Double c, const Double w, const Double p) {
    assert( w != 0.0 );
    Double t=(x-c)/w;
    //return N(p)*h*((2.0*p-1.0)*t*t - 1.0)/denominator(x, c, w, p, 1.0);
    return ((p-1.0/2.0)/p)*((1.0-t*t)/w)*(this->eval(x, h, c, w, p));
}

/**
 *  evaluate the value of the differentiation coefficient for p, dL/dp
 *
 *  \param[in] x argument
 *  \param[in] h the height of the function
 *  \param[in] c the center of the function
 *  \param[in] w the HWHM of the function
 *  \param[in] p the power number of the 
 */
Double AdvAugmentedLorentzian::derP(const Double x, const Double h, const Double c, const Double w, const Double p) {
    assert( w != 0.0 );
    Double t=(x-c)/w;
    return eval(x, h, c, w, p)*(gsl_sf_psi(p)-gsl_sf_psi(p-0.5)-log(t*t+1.0));
}

#endif // HAVE_DIFFERENTIAL_MEMBER

#ifdef USE_POINTER
Double AdvAugmentedLorentzian::eval(const Double x, const Double *p) {
    return eval(x, p[0], p[1], p[2], p[3]);
}

#ifdef HAVE_DIFFERENTIAL_MEMBER
Double AdvAugmentedLorentzian::der1st(const Double x, const Double *p) {
    return der1st(x, p[0], p[1], p[2], p[3]);
}

Double AdvAugmentedLorentzian::der2nd(const Double x, const Double *p) {
    assert( p[2] != 0.0 );
    return der2nd(x, p[0], p[1], p[2], p[3]);
}

Double *AdvAugmentedLorentzian::gradient(const Double x, const Double *p) {
    assert( p[2] != 0.0 );
    Double *g=new Double[NumberOfParamsForAugmentedLorentzian];

    g[0] =        eval(x, 1.0,  p[1], p[2], p[3]);
    g[1] = -1.0*der1st(x, p[0], p[1], p[2], p[3]);
    g[2] =        derW(x, p[0], p[1], p[2], p[3]);
    g[3] =        derP(x, p[0], p[1], p[2], p[3]);

    return g;
}
#endif // HAVE_DIFFERENTIAL_MEMBER
#endif // USE_POINTER

#ifdef USE_VECTOR
Double AdvAugmentedLorentzian::eval(const Double x, const vector<Double> &p) {
    return eval(x, p[0], p[1], p[2], p[3]);
}

#ifdef HAVE_DIFFERENTIAL_MEMBER
Double AdvAugmentedLorentzian::der1st(const Double x, const vector<Double> &p) {
    return der1st(x, p[0], p[1], p[2], p[3]);
}

Double AdvAugmentedLorentzian::der2nd(const Double x, const vector<Double> &p) {
    assert( p[2] != 0.0 );
    return der2nd(x, p[0], p[1], p[2], p[3]);
}

vector<Double> AdvAugmentedLorentzian::gradient(const Double x, const vector<Double> &p) {
    assert( p[2] != 0.0 );
    vector<Double> *g=new vector<Double>() ;

    g->push_back(        eval(x, 1.0,  p[1], p[2], p[3]) );
    g->push_back( -1.0*der1st(x, p[0], p[1], p[2], p[3]) );
    g->push_back(        derW(x, p[0], p[1], p[2], p[2]) );
    g->push_back(        derP(x, p[0], p[1], p[2], p[3]) );

    return *g;
}
#endif // HAVE_DIFFERENTIAL_MEMBER
#endif // USE_VECTOR
