#ifndef VECTOR_COMPLEX_TOOL
#define VECTOR_COMPLEX_TOOL

#include <complex>

#include "gsl/gsl_complex.h"
#include "gsl/gsl_complex_math.h"
#include "gsl/gsl_vector.h"
#include "gsl/gsl_vector_double.h"
#include "gsl/gsl_vector_complex.h"
#include "gsl/gsl_vector_complex_double.h"

#include "Header.hh"

#include "AdvMessage.hh"

/** Tools for std::complex-type vectors.
 *  creation, initialzation, type conversion
 *
 *  @author TANIMORI Souichirou, AdvanceSoft Corp.
 *  @version 0.0
 *  @since   0.8
 */
class AdvVectorComplexTool : public AdvMessage {
    private:
        static const std::string className; // = std::string("AdvVectorComplexTool");

    public:
        enum Part {
            REAL, /** the symbol for real      part of std::complex */
            IMAG, /** the symbol for imagenary prat of std::complex */
        };

    private:
        /** check source.
         *  @param[in] src  the pointer to the source array
         *  @param[in] size the size of the source array
         *  @retval true  when src &ne; NULL and size &gt; 0
         *  @retval false otherwize
         */
        Bool checkSrc(const Double* src, const UInt4 size);

        /** check source.
         *  @param[in] src  the reference to a source std::vector
         *  @retval true  if src is not empty.
         *  @retval false otherwise
         */
        Bool checkSrc(const std::vector<Double>& src);

        /** check source.
         *  @param[in] src  the pointer to a gsl_vector.
         *  @retval true  if src &new; NULL and the size of src is lagerer than 0
         *  @retval false otherwize
         */
        Bool checkSrc(const gsl_vector* src);

        /** check source.
         *  @param[in] real the pointer to an array as the real      part
         *  @param[in] imag the pointer to an array as the imagenary part
         *  @retval true  if real &ne; NULL, imag &ne NULL, real&rarr;size > 0, imag&rarr;size and real&rarr;size &eq; imag&rarr;size
         *  @retval false otherwize
         */
        Bool checkSrc(const Double* real, const Double* imag, const UInt4 size);

        /** check source.
         *  @param[in] real the reference to an array as the real      part of std::complex numbers
         *  @param[in] imag the reference to an array as the imagenary part of std::complex numbers
         *  @retval true  when ! real.empty() and ! image.empty() and real.size() == image.size()
         *  @retval false otherwise
         */
        Bool checkSrc(const std::vector<Double>& real, const std::vector<Double>& imag);

        /** check source.
         *  @param[in] real  the pointer to a gs_vector as the real      part
         *  @param[in] imag  the pointer to a gs_vector as the imagenary part
         *  @retval true  when real &ne NULL and imag &ne;  and real&rarr;size == imag&rarr;size
         *  @retval false otherwise
         */
        Bool checkSrc(const gsl_vector* real, const gsl_vector* imag);

        /** check the destination.
         *  @param[in] dest    the pointer of a gsl_vector_complex
         *  @param[in] srcSize the size of the souce
         *  @retval true  if dest &ne; NULL and dest &rarr size &eq; srcSize
         *  @retval false otherwise
         */
        Bool checkDest(gsl_vector_complex* dest, const UInt4 srcSize);

        /** check the destirnation.
         *  @param[in] dest    the reference of a std::complex number type std::vector
         *  @param[in] srcSize the size of the souce
         *  @retval true  if the capacity of dest &eq; srcSize
         *  @retval false otherwise
         */
        Bool checkDest(std::vector< std::complex<Double> >& dest, const UInt4 srcSize);

        /** check the destirnation.
         *  @param[in] dest    the pointer to a comlex number type array
         *  @param[in] srcSize the size of the souce
         *  @retval true  if the capacity of dest &eq; srcSize
         *  @retval false otherwise
         */
        Bool checkDest(std::complex<Double>* dest, const UInt4 srcSize);

        Bool checkSrc(const gsl_vector_complex* src);
        Bool checkSrc(const std::complex<Double>* src, const UInt4 size);
        Bool checkSrc(const std::vector< std::complex<Double> >& src);

    public:
        /** constructor */
        AdvVectorComplexTool() {};

        /** constructor */
        virtual ~AdvVectorComplexTool() {};

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** initialise a vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  the pointer to a source array
         *  @param[in] size the size of the array
         *  @param[in] dest destination std::vector
         */
        void initGslVectorComplex(const enum Part part, const Double *src, const UInt4 size, gsl_vector_complex* dest);

        /** initialise a vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  a source std::vector
         *  @param[in] dest destination std::vector
         */
        void initGslVectorComplex(const enum Part part, const std::vector<Double>& src, gsl_vector_complex* dest);

        /** initialise a vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  the pointer to a source array
         *  @param[in] dest destination std::vector
         */
        void initGslVectorComplex(const enum Part part, const gsl_vector* src, gsl_vector_complex* dest);


        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** create a std::complex-type vactor from a Double array .
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  the pointer to an array
         *  @param[in] size the size of the array
         *  @retval the pointer to a new std::vector if src is not NULL and size is larger than 0
         *  @retval NULL, otherwise
         */
        gsl_vector_complex* createGslVectorComplex(const enum Part part, const Double* src, const UInt4 size);

        /** create a std::complex-type std::vector from a Double std::vector.
         *  @param[in] part the part of a std::complex std::vector to be initialized, REAL or IMAG
         *  @param[in] src  a real-type std::vector
         *  @retval the pointer to a new std::vector if src is not empty
         *  @retval NULL, otherwise
         */
        gsl_vector_complex* createGslVectorComplex(const enum Part part, const std::vector<Double>& src);

        /** create a std::complex-type std::vector from a Double std::vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  a real-type std::vector
         *  @retval the pointer to the created std::vector if src is not NULL and not empty
         *  @retval NULL, otherwise
         */
        gsl_vector_complex* createGslVectorComplex(const enum Part part, const gsl_vector* src);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** initialize a std::complex std::vector.
         *  @param[in]  real the pointer to an array as real part
         *  @param[in]  imag the pointer to an array as imagenary part
         *  @param[in]  size the size of the arrays
         *  @param[out] dest the pointer to a vector to be initializeed.
         */
        void initGslVectorComplex(const Double* real, const Double* imag, const UInt4 size, gsl_vector_complex* dest);

        /** initialize a std::complex std::vector.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @param[out] dest the pointer to a vector to be initializeed.
         */
        void initGslVectorComplex(const std::vector<Double>& real, const std::vector<Double>& imag, gsl_vector_complex* dest);

        /** initiallze a std::complex std::vector.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @param[out] dest the pointer to a vector to be initializeed.
         */
        void initGslVectorComplex(const gsl_vector* real, const gsl_vector* imag, gsl_vector_complex* dest);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** create std::complex-type std::vector from two Double array, one is the real part and another is the imagenary part.
         *  @param[in] real the pointer to an array as real part
         *  @param[in] imag the pointer to an array as imagenary part
         *  @param[in] size the size of the arrays
         *  @retval the pointer to a gsl_vector_complex type std::vector if the both of real and imag are not NULL and size is larger than 0
         *  @retval NULL, otherwise
         */
        gsl_vector_complex* createGslVectorComplex(const Double* real, const Double* imag, const UInt4 size);

        /** create a std::complex-type std::vector from two std::vector.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @retval the pointer to a gsl_vector_complex type std::vector if the both of real and imag are not empty
         *  @retval NULL, otherwise
         */
        gsl_vector_complex* createGslVectorComplex(const std::vector<Double>& real, const std::vector<Double>& imag);

        /** create a compex-type std::vector from two std::vector.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @retval the pointer to a std::complex std::vector if the both of real and imag are not NULL and not empty
         *  @retval NULL, otherwise
         */
        gsl_vector_complex* createGslVectorComplex(const gsl_vector* real, const gsl_vector* imag);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** initialize a std::complex std::vector.
         *  @param[in]  part part to be initialized, REAL or IMAG
         *  @param[in]  src  the pointer to an array
         *  @param[in]  size the size of the array
         *  @param[out] dest the pointer to a vector to be initiailzed
         */
        void initVectorComplex(const enum Part part, const Double* src, const UInt4 size, std::vector< std::complex<Double> >& dest);

        /** initialize a std::complex std::vector.
         *  @param[in]  part part to be initialized, REAL or IMAG
         *  @param[in]  src  a real-type std::vector
         *  @param[out] dest the pointer to a vector to be initiailzed
         */
        void initVectorComplex(const enum Part part, const std::vector<Double>& src, std::vector< std::complex<Double> >& dest);

        /** initialize a std::complex std::vector.
         *  @param[in]  part part to be initialized, REAL or IMAG
         *  @param[in]  src  a real-type std::vector
         *  @param[out] dest the pointer to a vector to be initiailzed
         */
        void initVectorComplex(const enum Part part, const gsl_vector* src, std::vector< std::complex<Double> >& dest);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** create a std::complex std::vector from a real-tyoe std::vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src    the pointer to an array
         *  @param[in] size the size of the array
         *  @retval the pointer to a vector< std::complex<Double> > type std::vector if src is not NULL and the size of src is larger than 0
         *  @retval NULL, otherwise
         */
        std::vector< std::complex<Double> >* createVectorComplex(const enum Part part, const Double* src, const UInt4 size);

        /** create a std::complex std::vector from a real-type std::vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src    a real-type std::vector
         *  @retval the pointer to a vector< std::complex<Double> > type std::vector if src is not empty
         *  @retval NULL, otherwise
         */
        std::vector< std::complex<Double> >* createVectorComplex(const enum Part part, const std::vector<Double>& src);

        /** create a std::complex std::vector from a real-type std::vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  a real-type std::vector
         *  @retval the pointer to a vector< std::complex<Double> > type std::vector if src is not NULL and not empty
         *  @retval NULL, otherwise
         */
        std::vector< std::complex<Double> >* createVectorComplex(const enum Part part, const gsl_vector* src);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** initialze a std::complex array.
         */
        void initVectorComplex(const Double* real, const Double* imag, const UInt4 size, std::vector< std::complex<Double> >& dest);
        void initVectorComplex(const std::vector<Double>& real, const std::vector<Double>& imag, std::vector< std::complex<Double> >& dest);
        void initVectorComplex(const gsl_vector* real, const gsl_vector* imag, std::vector< std::complex<Double> >& dest);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** create a std::complex std::vector from two real-type std::vector.
         *  @param[in] real the pointer to an array as real part
         *  @param[in] imag the pointer to an array as imagenary part
         *  @param[in] size the size of the arrays
         *  @retval the pointer to a vector< std::complex<Double> > type std::vector if the both of real and imag are not NULL and size is larger than 0
         *  @retval NULL, otherwize
         */
        std::vector< std::complex<Double> >* createVectorComplex(const Double* real, const Double* imag, const UInt4 size);

        /** create a std::complex std::vector from two real-type std::vector.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @retval the pointer to a vector< std::complex<Double> > type std::vector if the both of real and imag are not empty
         *  @retval NULL, otherwize
         */
        std::vector< std::complex<Double> >* createVectorComplex(const std::vector<Double>& real, const std::vector<Double>& imag);

        /** create a std::complex std::vector from two real-type std::vector.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @retval the pointer to a vector< std::complex<Double> > std::vector if real and imag are not NULL and not empty
         *  @retval NULL, otherwise
         */
        std::vector< std::complex<Double> >* createVectorComplex(const gsl_vector*     real, const gsl_vector*     imag);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        void initArrayComplex(const enum Part part, const Double* src, const UInt4 size, std::complex<Double>* dest);
        void initArrayComplex(const enum Part part, const std::vector<Double>& src, std::complex<Double>* dest);
        void initArrayComplex(const enum Part part, const gsl_vector* src, std::complex<Double>* dest);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** create a std::complex array from a real-type std::vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  the pointer to an array
         *  @param[in] size the size of the array
         *  @retval the pointer to std::complex<Double> array if src is not NULL and the size of src is larger than 0
         *  @retval NULL, otherwise
         */
        std::complex<Double>* createArrayComplex(const enum Part part, const Double* src, const UInt4 size);

        /** create a std::complex array from a real-type std::vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  a real-type std::vector
         *  @retval the pointer to a std::complex<Double> array if src is not empty
         *  @retval NULL, otherwise
         */
        std::complex<Double>* createArrayComplex(const enum Part part, const std::vector<Double>& src);

        /** create a std::complex array from a real-type std::vector.
         *  @param[in] part part to be initialized, REAL or IMAG
         *  @param[in] src  a real-type std::vector
         *  @retval the pointer to a std::complex<Double> array if src is not NULL and not empty
         *  @retval NULL, otherwise
         */
        std::complex<Double>* createArrayComplex(const enum Part part, const gsl_vector* src);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        void initArrayComplex(const Double* real, const Double* imag, const UInt4 size, std::complex<Double>* dest);
        void initArrayComplex(const std::vector<Double>& real, const std::vector<Double>& imag, std::complex<Double>* dest);
        void initArrayComplex(const gsl_vector* real, const gsl_vector* imag, std::complex<Double>* dest);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** create a std::complex array from two real-type std::vector.
         *  @param[in] real the pointer to an array as real part
         *  @param[in] imag the pointer to an array as imagenary part
         *  @param[in] size the size of the arrays
         *  @retval the pointer to a std::complex<Double> array if the both of real and imag are not NULL and the size is larger than 0
         */
        std::complex<Double>* createArrayComplex(const Double* real, const Double* imag, const UInt4 size);

        /** create a std::complex array from two real-tyoe std::vector.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @retval the pointer to a std::complex<Double> array if the both of real and imag are not empty
         *  @retval NULL, otherwize
         */
        std::complex<Double>* createArrayComplex(const std::vector<Double>& real, const std::vector<Double>& imag);

        /** create a std::complex array from two real-type array.
         *  @param[in] real a vector as the real part
         *  @param[in] imag a vector as the imagenary part
         *  @retval the pointer to a std::complex<Double> array if the both of real and imag are not NULL and not empty
         *  @retval NULL, otherwise
         */
        std::complex<Double>* createArrayComplex(const gsl_vector* real, const gsl_vector* imag);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        /** transform a vector to another type std::vector.
         *  @param[in] src a std::complex std::vector
         *  @retval the pointer to a gsl_vector_complex type std::vector if src is not empty
         *  @retval NULL, otherwise
         */
        gsl_vector_complex* VectorComplexToGslVectorComplex(const std::vector< std::complex<Double> >& src);

        /** transfotm a vector to other type std::vector.
         *  @param[in] src a gsl_vector_complex
         *  @retval the pointer to a std::complex std::vector if src is not NULL and not empty
         *  @retval NULL, otherwise
         */
        std::vector< std::complex<Double> >* GslVectorComplexToVectorComplex(const gsl_vector_complex* src);

        /** transform an array to a vector.
         *  @param[in] src    a pointer to an array
         *  @param[in] size the size of v
         *  @retval a pointer to a vector if src is not NULL and the size of src is larger than 0
         *  @param  NULL, otherwise
         */
        gsl_vector_complex* ArrayComplexToGslVectorComplex(const std::complex<Double>* src, UInt4 size);

        /** transform a vector to an arry.
         *  @param[in] src  the pointer to a vector
         *  @retval a pointer to an array if src is not NULL and not empty
         *  @retval NULL, otherwise
         */
        std::complex<Double>* GslVectorComplexToArrayComplex(const gsl_vector_complex* src);

        /** transform an array to std::vector.
         *  @param[in] src    the pointer to an array
         *  @param[in] size the size of v
         *  @retval the pointer to a vector if src is not NULL and the size of src is larger than 0
         *  @retval NULL, otherwize
         */
        std::vector< std::complex<Double> >* ArrayComplexToVectorComplex(const std::complex<Double>* src, UInt4 size);

        /** transform a vector to an array.
         *  @param[in] src a vector
         *  @retval the pointer to an array if src is not empty
         *  @retval NULL, otherwise
         */
        std::complex<Double>* VectorComplexToArrayComplex(const std::vector< std::complex<Double> >& src);

        //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

        gsl_vector* getPartAsGslVector(const enum Part part, const gsl_vector_complex* src);
        gsl_vector* getPartAsGslVector(const enum Part part, std::complex<Double>* src, const UInt4 size);
        gsl_vector* getPartAsGslVector(const enum Part part, std::vector< std::complex<Double> >& src);

        Double* getPartAsArray(const enum Part part, const gsl_vector_complex* src);
        Double* getPartAsArray(const enum Part part, std::complex<Double>* src, const UInt4 size);
        Double* getPartAsArray(const enum Part part, std::vector< std::complex<Double> >& src);

        std::vector<Double>* getPartAsVector(const enum Part part, const gsl_vector_complex* src);
        std::vector<Double>* getPartAsVector(const enum Part part, std::complex<Double>* src, const UInt4 size);
        std::vector<Double>* getPartAsVector(const enum Part part, std::vector< std::complex<Double> >& src);
};
#endif // VECTOR_COMPLEX_TOOL
