#include "AdvMultiDataLevmar.hh"
#include <unistd.h>
#include <sstream>
#include <ctime>

const std::string AdvMultiDataLevmar::className=std::string("AdvMultiDataLevmar");


/**
 *  default constructor
 */
AdvMultiDataLevmar::AdvMultiDataLevmar() {
    methodName=std::string("Multi Data Levenberg-Marquardt");
}


/**
 *  destructor
 */
AdvMultiDataLevmar::~AdvMultiDataLevmar() {
    //delete im;
    //delete stat;
    //delete history;
}

//AdvParamSet AdvMultiDataLevmar::setDefaultParam(ElementContainer &src) {
//    return *(new AdvParamSet());
//}

/**
 *  set default values of parameters for the method
 */
AdvParamSet AdvMultiDataLevmar::setDefaultParam(ElementContainerArray &src) {
    std::string memberName = std::string("setDefaultParam(ElementContainerArray &)");
    DebugMessage(className, memberName, "enter\n");

    AdvParamSet *param = new AdvParamSet();

    /* constrain type */
    param->add(CONSTRAIN,          DEFAULT_CONSTRAIN);

    /* numerical differentiation */
    param->add(USE_NUMERICAL_DIFF, DEFAULT_USE_NUMERICAL_DIFF);
    param->add(DIFF_METHOD,        DEFAULT_DIFF_METHOD);

    param->add(USE_DATA_WEIGHTS,   DEFAULT_USE_DATA_WEIGHTS);
    /* maximum of iterations */
    param->add(MAX_ITERATIONS,     DEFAULT_MAX_ITERATIONS);
    /* output interval */
    param->add(OUTPUT_INTERVAL,    DEFAULT_OUTPUT_INTERVAL);

    param->add(HISTORY_CAPACITY,   DEFAULT_HISTORY_CAPACITY);

    /* function base */
    //param->add(FUNCTIONS, *(new std::vector<FuncBase*>()));

    /* paramter and reference values */
    //param->add(PARAMETER_VALUES, *(new std::vector<Double>()));
    //param->add(REFERENCE_VALUES, *(new std::vector<Double>()));

    /* constrain */
    //param->add(LOWER_BOUNDS, *(new std::vector<Double>()));
    //param->add(UPPER_BOUNDS, *(new std::vector<Double>()));

    /* multi data weights */
    std::vector<Double> w=*(new std::vector<Double>(src.PutSize()));
    Double s=0.0;
    Int4 n;
    for (UInt4 i=0; i<src.PutSize(); ++i) {
        ElementContainer *elem=src.PutPointer(i);
        n=elem->PutSize(elem->PutYKey());
        s=1.0/n;
    }

    /* scaling factor */
    param->add(SCALING_FACTOR,     DEFAULT_SCALING_FACTOR);

    /* stopping thresholds, deprecated */
    param->add(TOLERANCE,          DEFAULT_TOLERANCE);
    param->add(RELATIVE_TOLERANCE, DEFAULT_RELATIVE_TOLERANCE);
    param->add(GRADIENT_TOLERANCE, DEFAULT_GRADIENT_TOLERANCE);

    /* stopping thresholds */
    param->add(GRADIENT_THRESH,   DEFAULT_GRADIENT_THRESH);
    param->add(PARAM_DIFF_THRESH, DEFAULT_PARAM_DIFF_THRESH);
    param->add(RESIDU_ERR_THRESH, DEFAULT_RESIDU_ERR_THRESH);

    param->add(DIFF_DELTA,         DEFAULT_DIFF_DELTA);

    //PeakSearch *peakSearch = new PeakSearch(src, new MovingAverage());

    DebugMessage(className, memberName, "exit\n");
    return *param;
}

//Bool AdvMultiDataLevmar::checkParam(ElementContainer &src, AdvDomain &domain, AdvParamSet &param) {
//    return false;
//}

Bool AdvMultiDataLevmar::checkParam(ElementContainerArray &src, std::vector<AdvDomain> &domainArray, AdvParamSet &param) {
    std::string memberName = std::string("checkParam(ElementContainerArray &, AdvDomain &, AdvParamSet &)");

    DebugMessage(className, memberName, "enter\n");
    Bool retval=true;
    retval = retval && this->im->checkParam(src, domainArray, param);

    DebugMessage(className, memberName, "exit\n");
    return retval;
}

//void AdvMultiDataLevmar::toInnerForm (ElementContainer &src, AdvDomain &domain, AdvParamSet &param) {
//}

void AdvMultiDataLevmar::toInnerForm (ElementContainerArray &src, std::vector<AdvDomain> &domainArray, AdvParamSet &param) {
    std::string memberName=std::string("toInnerForm(ElementContainerArray &, AdvDomain &, AdvParamSet &)");
    DebugMessage(className, memberName, "enter\n");

    this->resultBin=*(new std::vector< std::vector<Double> >(src.PutSize()));
    this->resultX  =*(new std::vector< std::vector<Double> >(src.PutSize()));
    for (UInt4 i=0; i<src.PutSize(); ++i) {
        this->resultBin.at(i)=*(domainArray.at(i).getBin());
        this->resultX.at(i)  =*(domainArray.at(i).createXC());
    }

    this->command = new AdvFitCommand();
    this->stat    = new AdvConvergenceStat();
    this->history = new AdvConvergenceStat();

    this->im=new AdvLevmarImmutables();
    this->im->toInnerForm(src, domainArray, param);
    this->im->output();

    DebugMessage(className, memberName, "exit\n");
}

/**
 *  start fitting
 */
void AdvMultiDataLevmar::fit() {
    std::string memberName=std::string("fit()");
    DebugMessage(className, memberName, "enter\n");

    AdvLevmarFit *obj = new AdvLevmarFit(this->im, this->stat, this->command);
    AdvReportConvergenceProcess *report = new AdvReportConvergenceProcess(this->im, this->stat, this->history);

    Int4 retval=obj->Start();
    DebugMessage(className, memberName, "LevamrFit Start: %d\n", retval);
    obj->Detach();
    DebugMessage(className, memberName, "LevamrFit Detached\n");

    
    //report->Run();
    report->Start();
    DebugMessage(className, memberName, "AdvReportConvergenceProcess Start: %d\n", retval);
    report->Detach();
    DebugMessage(className, memberName, "AdvReportConvergenceProcess Detached\n");

    /*
    AdvParamSet *p;
    p = this->history->referLatestStat();
    while (static_cast<AdvLevmarConsts::LevmarStat>(p->getInt4(AdvLevmarConsts::TERMINATION_STAT)) == AdvLevmarConsts::CONTINUE) {
        DebugMessage(className, memberName, "fit sleep\n");
        sleep(1);
        p = this->history->referLatestStat();
    }
    */

    DebugMessage(className, memberName, "exit\n");
}

/**
 *  \return true if fitting is running.
 */
Bool AdvMultiDataLevmar::isFitting() {
    Bool retval=false;
    AdvParamSet *pp;
    if ( this->history !=NULL ) {
        pp=this->history->referLatestStat();
        retval = static_cast<AdvLevmarConsts::LevmarStat>(pp->getInt4(AdvLevmarConsts::TERMINATION_STAT)) == AdvLevmarConsts::CONTINUE;
    }
    return retval;
}

/**
 *  command to stop fitting
 */
void AdvMultiDataLevmar::stopFit() {
    if (this->command != NULL) {
        this->command->setStop();
    }
}


void AdvMultiDataLevmar::eval() {
    std::string memberName = std::string("eval()");
    DebugMessage(className, memberName, "enter\n");

    //AdvParamSet *paramSet=this->stat->referLatestStat();
    AdvParamSet *paramSet=this->history->referLatestStat();

    for (UInt4 i=0; i<paramSet->getVectorSize(AdvMultiDataLevmar::PARAMETER_VALUES); ++i) {
        DebugMessage(className, memberName, "%u %e %e\n", i, paramSet->getDouble(AdvMultiDataLevmar::PARAMETER_VALUES, i), paramSet->getDouble(AdvMultiDataLevmar::PARAM_ERRORS, i));
    }

    this->resultY=*(new std::vector< std::vector<Double> >());
    this->resultE=*(new std::vector< std::vector<Double> >());

    if ( paramSet != NULL ) {
        DebugMessage(className, memberName, "the number of data series: %u\n", this->resultX.size());
        for (UInt4 i=0; i<this->resultX.size(); ++i) {
            AdvFuncComb f=*(new AdvFuncComb(this->im->args.funcList.at(i), paramSet->getVector(AdvMultiDataLevmar::PARAMETER_VALUES), paramSet->getVector(AdvMultiDataLevmar::PARAM_ERRORS)));
            this->resultY.push_back(f.eval(this->resultX.at(i)));
            this->resultE.push_back(f.evalError(this->resultX.at(i)));
            DebugMessage(className, memberName, "size of Bin: %u\n", this->resultBin.at(i).size());
            DebugMessage(className, memberName, "size of X  : %u\n", this->resultX.at(i).size()  );
            DebugMessage(className, memberName, "size of Y  : %u\n", this->resultY.at(i).size()  );
            DebugMessage(className, memberName, "size of E  : %u\n", this->resultE.at(i).size()  );
        }
    }
    DebugMessage(className, memberName, "exit\n");
}


std::vector< std::vector<Double> > AdvMultiDataLevmar::getTrend(UInt4 i) {
    //AdvParamSet *paramSet=this->stat->referLatestStat();
    AdvParamSet *paramSet=this->history->referLatestStat();

    std::vector< std::vector<Double> > *retval= new std::vector< std::vector<Double> >();
    if ( paramSet != NULL ) {
        AdvFuncComb f=*(new AdvFuncComb(this->im->args.funcList.at(i), paramSet->getVector(AdvMultiDataLevmar::PARAMETER_VALUES), paramSet->getVector(AdvMultiDataLevmar::PARAM_ERRORS)));
        retval->push_back(f.eval(  this->resultX.at(i)));
        retval->push_back(f.der1st(this->resultX.at(i)));
        retval->push_back(f.der2nd(this->resultX.at(i)));
    }
    return *retval;
}

std::vector< std::vector<Double> > AdvMultiDataLevmar::getTrend() {
    std::vector< std::vector<Double> > *retval= new std::vector< std::vector<Double> >();
    for (UInt4 i=0; i<this->im->args.funcList.size(); ++i) {
        std::vector< std::vector<Double> > tmp = getTrend(i);
        retval->insert(retval->end(), tmp.begin(), tmp.end());
    }
    return *retval; 
}

void AdvMultiDataLevmar::addExpressionsToHeader(ElementContainer &dest, const std::string &keyBase, std::vector< std::vector<Double> > *expressions) const {
    Int4 ct=0;
    for (std::vector< std::vector<Double> >::const_iterator expr=expressions->begin(); expr !=expressions->end(); ++expr) {
        std::ostringstream ss;
        ss << keyBase << " " << ct;
        std::string key=ss.str();
        dest.AddToHeader(key, *expr);
        ++ct;
    }
}

void AdvMultiDataLevmar::toElementContainer(ElementContainerArray &src, UInt4 id, ElementContainer &dest) const {
    std::string memberName=std::string("toElementContainer(ElementContainerArray &, UInt4, ElementContainer &)");

    ElementContainer *srcElem=src.PutPointer(id);
    std::string xKey=srcElem->PutXKey();
    std::string yKey=srcElem->PutYKey();
    std::string eKey=srcElem->PutEKey();

    //DebugMessage(className, memberName, __FILE__, __LINE__, "i=%d %s %s %s\n", i, xKey.c_str(), yKey.c_str(), eKey.c_str());
    dest.Add(xKey, this->resultBin.at(id), srcElem->PutUnit(xKey));
    dest.Add(yKey, this->resultY.at(id),   srcElem->PutUnit(yKey));
    dest.Add(eKey, this->resultE.at(id),   srcElem->PutUnit(eKey));
    dest.SetKeys(xKey, yKey, eKey);

    //AdvParamSet *p=this->stat->referLatestStat();
    AdvParamSet *p=this->history->referLatestStat();
    // method
    dest.AddToHeader("method",                         this->getMethodName());
    // control
    dest.AddToHeader(AdvLevmarConsts::CONSTRAIN,          AdvLevmarConsts::CONSTRAIN_STR[this->im->control.constrain]);
    dest.AddToHeader(AdvLevmarConsts::USE_DATA_WEIGHTS,   this->im->control.useDataWeights);
    dest.AddToHeader(AdvLevmarConsts::USE_NUMERICAL_DIFF, this->im->control.useNumericalDiff);
    if (this->im->control.useNumericalDiff) {
        dest.AddToHeader(AdvLevmarConsts::DIFF_METHOD,    this->im->control.diffMethod);
    }

    // argments
    dest.AddToHeader(AdvLevmarConsts::FUNCTIONS,            *(this->im->args.getFunctionStrForMultiData(id)));
    dest.AddToHeader(AdvLevmarConsts::INITIAL_PARAM_VALUES, *(this->im->args.getInitialParam()));
//    switch (this->im->control.constrain) {
//        case AdvLevmarConsts::BOX:
//#ifdef HAVE_LAPACK
//        case AdvLevmarConsts::BLEC:
//        case AdvLevmarConsts::BLIC:
//        case AdvLevmarConsts::BLEIC:
//#endif // HAVE_LAPACK
//            if (this->im->args.lb != NULL) {
//                dest.AddToHeader(AdvLevmarConsts::LOWER_BOUNDS, *(this->im->args.getLowerBounds()));
//            }
//            if (this->im->args.ub != NULL) {
//                dest.AddToHeader(AdvLevmarConsts::UPPER_BOUNDS, *(this->im->args.getUpperBounds()));
//            }
//#ifdef HAVE_LAPACK
//            if (this->im->control.constrain == AdvLevmarConsts::BLEC) {
//                if (this->im->args.wghts != NULL) {
//                    dest.AddToHeader(AdvLevmarConsts::BOX_WEIGHTS, *(this->im->args.getBoxWeights()));
//                }
//            }
//#endif // HAVE_LAPACK
//            break;
//        default:
//            break;
//    }
//#ifdef HAVE_LAPACK
//    switch (this->im->control.constrain) {
//        case AdvLevmarConsts::LEC:
//        case AdvLevmarConsts::BLEC:
//        case AdvLevmarConsts::LEIC:
//        case AdvLevmarConsts::BLEIC:
//            this->addExpressionsToHeader(dest, AdvLevmarConsts::EQUATIONS, this->im->args.getEqMatrix());
//            break;
//        default:
//            break;
//    }
//    switch (this->im->control.constrain) {
//        case AdvLevmarConsts::LIC:
//        case AdvLevmarConsts::BLIC:
//        case AdvLevmarConsts::LEIC:
//        case AdvLevmarConsts::BLEIC:
//            this->addExpressionsToHeader(dest, AdvLevmarConsts::INEQUALITIES, this->im->args.getIneqMatrix());
//            break;
//        default:
//            break;
//    }
//#endif // HAVE_LAPACK

    //dest.AddToHeader(AdvLevmarConsts::SCALING_FACTOR,    this->im->args.opts[0]);
    //dest.AddToHeader(AdvLevmarConsts::RESIDU_ERR_THRESH, this->im->args.opts[1]);
    //dest.AddToHeader(AdvLevmarConsts::GRADIENT_THRESH,   this->im->args.opts[2]);
    //dest.AddToHeader(AdvLevmarConsts::PARAM_DIFF_THRESH, this->im->args.opts[3]);

    //if (this->im->control.useNumericalDiff) {
    //    dest.AddToHeader(AdvLevmarConsts::DIFF_DELTA,     this->im->args.opts[4]);
    //}

    std::vector<Double> *v=new std::vector<Double>();
    v=this->im->args.additionalData->getDomainBounds(id);
    /*
    v->push_back(*(this->im->args.additionalData->getX().front()));
    v->push_back(*(this->im->args.additionalData->.getX().back()) );
    v->push_back(*(this->im->args.additionalData->x->begin()));
    v->push_back(*(this->im->args.additionalData->x->end()-1));
    */
    dest.AddToHeader("domain", *v);

    // results
    //dest.AddToHeader(AdvLevmarConsts::ITERATION_COUNT,    p->getInt4(AdvLevmarConsts::ITERATION_COUNT));
    //dest.AddToHeader(AdvLevmarConsts::TERMINATION_STAT,   AdvLevmarConsts::TERMINATION_REASON[p->getInt4(AdvLevmarConsts::TERMINATION_STAT)]);
    dest.AddToHeader(AdvLevmarConsts::PARAMETER_VALUES,   p->getVector(AdvLevmarConsts::PARAMETER_VALUES));
    dest.AddToHeader(AdvLevmarConsts::PARAM_ERRORS,       p->getVector(AdvLevmarConsts::PARAM_ERRORS));

    //dest.AddToHeader(AdvLevmarConsts::R_FACTOR,           p->getDouble(AdvLevmarConsts::R_FACTOR));
    //dest.AddToHeader(AdvLevmarConsts::RESIDUAL_ERR_NORM,  p->getDouble(AdvLevmarConsts::RESIDUAL_ERR_NORM));
    //dest.AddToHeader(AdvLevmarConsts::GRADIENT_NORM,      p->getDouble(AdvLevmarConsts::GRADIENT_NORM));
    //dest.AddToHeader(AdvLevmarConsts::PARAM_DIFF_NORM,    p->getDouble(AdvLevmarConsts::PARAM_DIFF_NORM));
}

//void AdvMultiDataLevmar::toElementContainer(ElementContainer &src, ElementContainer &dest) const {
//    AdvMultiDataLevmar::toElementContainerElement(src, 0, dest);
//}

void AdvMultiDataLevmar::toElementContainerArray(ElementContainerArray &src, ElementContainerArray &dest) const {

    //AdvParamSet *p=this->stat->referLatestStat();
    //AdvParamSet *p=this->history->referLatestStat();

    time_t tmp=std::time(NULL);

    dest.AddToHeader("date", std::asctime(std::localtime(&tmp)));

    dest.AddToHeader(AdvLevmarConsts::CONSTRAIN,          AdvLevmarConsts::CONSTRAIN_STR[this->im->control.constrain]);
    dest.AddToHeader(AdvLevmarConsts::USE_DATA_WEIGHTS,   this->im->control.useDataWeights);
    dest.AddToHeader(AdvLevmarConsts::USE_NUMERICAL_DIFF, this->im->control.useNumericalDiff);
    if (this->im->control.useNumericalDiff) {
        dest.AddToHeader(AdvLevmarConsts::DIFF_METHOD,    this->im->control.diffMethod);
    }

    for (UInt4 i=0; i<src.PutSize(); ++i) {
        ElementContainer r = *(new ElementContainer());
        this->toElementContainer(src, i, r);
        dest.Add(r);
    }

}


AdvParamSet AdvMultiDataLevmar::getFittedParam() const {
    AdvParamSet *fittedParam = new AdvParamSet();
    //AdvParamSet *latestStat  = this->stat->referLatestStat();
    AdvParamSet *latestStat  = this->history->referLatestStat();
    std::vector<Double> v;

    fittedParam->add(AdvMultiDataLevmar::CONSTRAIN,          this->im->control.constrain);
    fittedParam->add(AdvMultiDataLevmar::USE_NUMERICAL_DIFF, this->im->control.useNumericalDiff);
    fittedParam->add(AdvMultiDataLevmar::DIFF_METHOD,        this->im->control.diffMethod);
    fittedParam->add(AdvMultiDataLevmar::USE_DATA_WEIGHTS,   this->im->control.useDataWeights);

    fittedParam->add(AdvMultiDataLevmar::MAX_ITERATIONS,     this->im->control.maxIterations );
    fittedParam->add(AdvMultiDataLevmar::OUTPUT_INTERVAL,    this->im->control.outputInterval);

    fittedParam->add(AdvMultiDataLevmar::FUNCTIONS, this->im->args.funcList);

    fittedParam->add(AdvMultiDataLevmar::INITIAL_PARAM_VALUES, *(this->im->args.getInitialParam()));
    v= latestStat->getVector(AdvMultiDataLevmar::PARAMETER_VALUES);
    fittedParam->add(AdvMultiDataLevmar::PARAMETER_VALUES, v);

    switch ( this->im->control.constrain) {
        case AdvMultiDataLevmar::BOX:
#ifdef HAVE_LAPACK
        case AdvMultiDataLevmar::BLEC:
        case AdvMultiDataLevmar::BLIC:
        case AdvMultiDataLevmar::BLEIC:
#endif
            if (this->im->args.lb != NULL) {
                fittedParam->add(AdvMultiDataLevmar::LOWER_BOUNDS, *(this->im->args.getLowerBounds()));
            }
            if (this->im->args.ub != NULL) {
                fittedParam->add(AdvMultiDataLevmar::UPPER_BOUNDS, *(this->im->args.getUpperBounds()));
            }
#ifdef HAVE_LAPACK
            if (this->im->control.constrain == AdvLevmarConsts::BLEC) {
                if (this->im->args.wghts != NULL) {
                    fittedParam->add(AdvLevmarConsts::BOX_WEIGHTS, *(this->im->args.getBoxWeights()));
                }
            }
#endif
            break;
        default:
            break;
    }
#ifdef HAVE_LAPACK
    switch ( this->im->control.constrain) {
        case AdvMultiDataLevmar::LEC:
        case AdvMultiDataLevmar::BLEC:
        case AdvMultiDataLevmar::LEIC:
        case AdvMultiDataLevmar::BLEIC:
            fittedParam->add(AdvMultiDataLevmar::EQUATIONS, *(this->im->args.getEqMatrix()));
            break;
        default:
            break;
    }
    switch ( this->im->control.constrain) {
        case AdvMultiDataLevmar::LIC:
        case AdvMultiDataLevmar::BLIC:
        case AdvMultiDataLevmar::LEIC:
        case AdvMultiDataLevmar::BLEIC:
            fittedParam->add(AdvMultiDataLevmar::INEQUALITIES, *(this->im->args.getEqMatrix()));
            break;
        default:
            break;
    }
#endif

    fittedParam->add(AdvMultiDataLevmar::GRADIENT_TOLERANCE, this->im->args.opts[1]);
    fittedParam->add(AdvMultiDataLevmar::TOLERANCE,          this->im->args.opts[2]);
    fittedParam->add(AdvMultiDataLevmar::RELATIVE_TOLERANCE, this->im->args.opts[3]);

    fittedParam->add(AdvMultiDataLevmar::GRADIENT_THRESH,    this->im->args.opts[1]);
    fittedParam->add(AdvMultiDataLevmar::PARAM_DIFF_THRESH,  this->im->args.opts[2]);
    fittedParam->add(AdvMultiDataLevmar::RESIDU_ERR_THRESH,  this->im->args.opts[3]);
    if (this->im->control.useNumericalDiff) {
        fittedParam->add(AdvMultiDataLevmar::DIFF_DELTA, this->im->args.opts[4]);
    }

    v=latestStat->getVector(AdvMultiDataLevmar::PARAM_ERRORS);
    fittedParam->add(AdvMultiDataLevmar::PARAM_ERRORS, v);
    fittedParam->add(AdvMultiDataLevmar::COVARIANCE_MATRIX, latestStat->getMatrix(AdvMultiDataLevmar::COVARIANCE_MATRIX));

    fittedParam->add(AdvMultiDataLevmar::FUNCTION_EVALUATIONS,  latestStat->getInt4(AdvMultiDataLevmar::FUNCTION_EVALUATIONS));
    fittedParam->add(AdvMultiDataLevmar::JACOBIAN_EVALUATIONS,  latestStat->getInt4(AdvMultiDataLevmar::JACOBIAN_EVALUATIONS));
    fittedParam->add(AdvMultiDataLevmar::LINEAR_SYSTEMS_SOLVED, latestStat->getInt4(AdvMultiDataLevmar::LINEAR_SYSTEMS_SOLVED));
    
    return *fittedParam;
}


Double AdvMultiDataLevmar::likehood() {
    return 0.0;
}

AdvParamSet *AdvMultiDataLevmar::getLatestConvergenceStat() const {
    return this->history->referLatestStat();
}


/**
 *
 */
std::deque<AdvParamSet*> AdvMultiDataLevmar::getConvergenceHistory() const {
    std::deque<AdvParamSet*> *retval=new std::deque<AdvParamSet*>();
    for (UInt4 i=0; i<this->history->size(); ++i) {
        retval->push_back(this->history->at(i));
    }
    return *retval;
}

/**
 *  get the size (depth) of convergence history
 */
UInt4 AdvMultiDataLevmar::getHistorySize() const {
    return this->history->size();
}

/**
 *  get a histrical list of Int4 type value
 */
std::vector<Int4> AdvMultiDataLevmar::getConvergenceHistoryForInt4(const std::string &key) const {
    std::vector<Int4> *retval=new std::vector<Int4>();
    for (UInt4 i=0; i<this->history->size(); ++i) {
        retval->push_back(this->history->at(i)->getInt4(key));
    }
    return *retval;
}

/**
 *  get a histrical list of Double type value
 */
std::vector<Double> AdvMultiDataLevmar::getConvergenceHistoryForDouble(const std::string &key) const {
    std::vector<Double> *retval=new std::vector<Double>();
    for (UInt4 i=0; i<this->history->size(); ++i) {
        retval->push_back(this->history->at(i)->getDouble(key));
    }
    return *retval;
}
