#!/usr/bin/python3
# -*- coding: utf-8 -*-

from __future__ import print_function

import utsusemi.SAS.ana.Reduction.CorrBaseCommands as CBC

# special reserved word for commands history
HISTORY = None

# reserved words for return value in this.
DAT = None
ecs = None
EC = None
ret = None
PyObj = None
Iq = None
Tr = None
EMPTYDAT = None

########################################
def NormByMonCounts( dat=DAT, monitor=EC, key="Lamb", XRange="0.6 7.6", factor=1.0 ):
    """
    Normalize data by monitor counts
    @param  dat      (ElementContainerMatrix, ElementContainer) target data
    @param  monitor  (ElementContainer) Monitor data from GetMonHistSAS
    @param  Xrange   (string)  "min-value max-value" for the range to sumalize intensities.
    @param  factor   (float)   factor for normalization I=(Iorg/mon)*factor
    @return None
    """
    CBC.NormByMonCounts( dat, monitor, key, XRange, factor)

########################################
def CorrectDetEffi( dat=DAT, filename="detectorEffiTBL.dat" ):
    """
    Does the correction of detector efficiency for lambda
    CorrectDetEffi( dat, filename )
    
    @param  dat       (ElementContainerMatrix)
    @param  filename  (string)
    @retval None
    """
    print("#######")
    CBC.CorrectDetEffi( dat, filename)

#########################################
def CorrectDetEffiRPMT( dat=DAT, alpha=0.10516, beta=0.04838 ):
    """
    Does the correction of detector efficiency for lambda
    CorrectDetEffi( dat, filename )
    
    @param  dat   (ElementContainerMatrix)
    @param  alpha (float) parameter of lambda dependency for detect efficiency of RPMT
    @param  beta  (float) parameter of lambda dependency for detect efficiency of RPMT
    @retval None
    """
    CBC.CorrectDetEffiRPMT( dat, alpha, beta )

########################################
def CorrectDetEffiMulti( dat=DAT, BankDefault="SC,SM,SD,SU,MU,MD,ML,MR,HU,HD,HL", Bank20atm="SCC,BW", DefaultData="SUS304-3He-PSD-SAS-6atm.dat", Atm20Data="SUS304-3He-PSD-SAS-20atm.dat" ):
    """
    Does the correction of detector efficiency for lambda
    CorrectDetEffi( dat, filename )

    @param  dat[Def:DAT] (ElementContainerMatrix)
    @param  BankDefault (string)
    @param  Bank20atm   (string)
    @param  DefaultData (string)
    @param  Atm20Data   (string)
    @retval None
    """
    CBC.CorrectDetEffiMulti( dat, BankDefault, Bank20atm, DefaultData, Atm20Data )

#########################################
def I0LambdaCorrection(dat=DAT,I0Lamb=EC,PolData="None"):
    """
    I0(lambda) correction
    @param dat [Def:DAT]     (ElementContainerMatrix)
    @param I0Lamb [Def:EC]   (ElementContainer)
    @param PolData  (str) file path to text data for the polarization vs lambda
    """
    CBC.I0LambdaCorrection(dat, I0Lamb, PolData)

#########################################
def P0LambdaCorrection(dat1=DAT,dat2=DAT,PolData="BL15_pol_vs_lambda.dat"):
    """
    P0(lambda dependency of poralization) correction
    @param dat1 [Def:DAT1]  (ElementContainerMatrix) I^{+}_obs
    @param dat2 [Def:DAT2]  (ElementContainerMatrix) I^{-}_obs
    @param PolData  (str) file path to text data for the polarization vs lambda
    @retval RET list of ElementContainerMatrix [I^{+}, I^{-}]
    """
    return CBC.P0LambdaCorrection(dat1, dat2, PolData)

#########################################
def NormByKickers(dat=DAT):
    """
    Normalize Intensity by the number of kickers
    @param dat   (ElementContainerMatrix)
    @retval None
    """
    CBC.NormByKickers(dat)

#########################################
def MakeDataListForSIB(dataList="A,B",A=DAT,B=DAT,C=DAT,D=DAT,E=DAT,F=DAT,G=DAT,H=DAT,I=DAT,J=DAT,K=DAT,L=DAT,M=DAT):
    """
    Make Data list for ScaleIntensityBanks
    @param dataList (string) list contents using A,B,... and M
    @param A [Def:EMPTYDAT] Data for a bank
    @param B [Def:EMPTYDAT] Data for a bank
    @param C [Def:EMPTYDAT] Data for a bank
    @param D [Def:EMPTYDAT] Data for a bank
    @param E [Def:EMPTYDAT] Data for a bank
    @param F [Def:EMPTYDAT] Data for a bank
    @param G [Def:EMPTYDAT] Data for a bank
    @param H [Def:EMPTYDAT] Data for a bank
    @param I [Def:EMPTYDAT] Data for a bank
    @param J [Def:EMPTYDAT] Data for a bank
    @param K [Def:EMPTYDAT] Data for a bank
    @param L [Def:EMPTYDAT] Data for a bank
    @param M [Def:EMPTYDAT] Data for a bank
    @retval listCBB
    """
    return CBC.MakeDataListForSIB(dataList,A,B,C,D,E,F,G,H,I,J,K,L,M)

#########################################
def ScaleIntensityBanks(datalist=DAT, SCC=1.0, SC=1.0, SM=1.0, SU=1.0, SD=1.0, MU=1.0, MD=1.0, MR=1.0, ML=1.0, HU=1.0, HD=1.0, HL=1.0, BW=1.0):
    """
    Scaling the intensitiy for each bank by given factor
    @param datalist [Def:listCBB]  (ElementContainerMatrix of list of ElementContainerMatrix)
    @param SCC (float) Scale factor for SCC bank
    @param SC  (float) Scale factor for SC bank
    @param SM  (float) Scale factor for SM bank
    @param SU  (float) Scale factor for SU bank
    @param SD  (float) Scale factor for SD bank
    @param MU  (float) Scale factor for MU bank
    @param MD  (float) Scale factor for MD bank
    @param MR  (float) Scale factor for MR bank
    @param ML  (float) Scale factor for ML bank
    @param HU  (float) Scale factor for HU bank
    @param HD  (float) Scale factor for HD bank
    @param HL  (float) Scale factor for HL bank
    @param BW  (float) Scale factor for BW bank
    @retval None
    """
    CBC.ScaleIntensityBanks(datalist, SCC, SC, SM, SU, SD, MU, MD, MR, ML, HU, HD, HL, BW)

#########################################
def LambdaDependCorrect(data=DAT, P1="ALL:lam_dep.txt", P2="None", P3="None", P4="None", P5="None", P6="None", P7="None", P8="None", P9="None"):
    """
    Correction for lambda dependency on intensity by given text file.
    Params format : <Bank(s)>:<FactorFile path>
    <Bank(s)> can include several banks with comma (,) like "SM,SC"

    @param data [Def:DAT] (ElementContainerMatrix or list of it)
    @param P1 (string)
    @param P2 (string)
    @param P3 (string)
    @param P4 (string)
    @param P5 (string)
    @param P6 (string)
    @param P7 (string)
    @param P8 (string)
    @param P9 (string)
    @retval None
    """
    CBC.LambdaDependCorrect(data, P1, P2, P3, P4, P5, P6, P7, P8, P9)

# Dictionary for entry the name of functions
_functions = {
              "NormByMonCounts":NormByMonCounts,
              "CorrectDetEffiMulti":CorrectDetEffiMulti,
              "CorrectDetEffiRPMT":CorrectDetEffiRPMT,
              "CorrectDetEffi":CorrectDetEffi,
              "I0LambdaCorrection":I0LambdaCorrection,
              "P0LambdaCorrection":P0LambdaCorrection,
              "NormByKickers":NormByKickers,
              "MakeDataListForSIB":MakeDataListForSIB,
              "ScaleIntensityBanks":ScaleIntensityBanks,
              "LambdaDependCorrect":LambdaDependCorrect
}

_functionsOrder = [ "I0LambdaCorrection",
                    "CorrectDetEffiMulti",
                    "CorrectDetEffi",
                    "CorrectDetEffiRPMT",
                    "P0LambdaCorrection",
                    "NormByMonCounts",
                    "NormByKickers",
                    "MakeDataListForSIB",
                    "ScaleIntensityBanks",
                    "LambdaDependCorrect"
                    ]
