#ifndef PARAMSET_PARTS_TEMPLATE
#define PARAMSET_PARTS_TEMPLATE

#include <map>
#include <string>
#include <vector>

#include <cstdlib>
#include <stdexcept>

#include <algorithm>
#include <functional>

#include <ios>
#include <iostream>
#include <iomanip>
#include <typeinfo>

#include "Header.hh"

#include "Message.hh"
#include "ContainerDumpTool.hh"

/** function object that extract the key from a entry
 *
 *  @author TANIMORI Souichirou, AdvanceSoft Corp.
 *  @version 0.0
 *  @since   0.9
 */
template<typename _SCALER_TYPE >
struct ExtractKey {
    string operator() ( std::pair<const string, _SCALER_TYPE >& item) { return item.first; };
};

/** compare  the length of two keys.
 *  @return true when the key (the first element) of item1 is less than the key of item2.
 *          less type functional
 *
 *  @author TANIMORI Souichirou, AdvanceSoft Corp.
 *  @version 0.0
 *  @since   0.9
 */
template<typename _SCALER_TYPE >
struct KeyLengthComp : binary_function< std::pair<const string, _SCALER_TYPE >&, std::pair<const string, _SCALER_TYPE >&, Bool > {
    Bool operator() (std::pair<const string, _SCALER_TYPE >& item1, std::pair<const string, _SCALER_TYPE >& item2) {
        return item1.first.size() < item2.first.size();
    };
};

/** template for a container of parameters for user specified type.
 *     - empty   check whether the container has no entry or not
 *     - size    the number of entries that container has
 *     - contain check whether the container has the specified key or not
 *     - add     add the value with the specified key to the container
 *     - get     get the value with the specified key. the method throw exceptions when fail to get a value.
 *     - replace replace the value with the specified key to a new value
 *     - erase   erase the entry with the specified key
 *     - clear   clear all entries
 *     - dump    print all entries to the standard output
 *
 *  @author TANIMORI souichirou, Advancesoft Corp.
 *  @version 0.0
 *  @sice    0.9
 */
template<typename _SCALER_TYPE >
class ContainerForScaler {

    protected:
        map<string, _SCALER_TYPE > cmap;
        
    public:
        /** constructor */
        ContainerForScaler< _SCALER_TYPE >() { this->cmap.clear(); };

        /** destructor */
        ~ContainerForScaler< _SCALER_TYPE >() { this->cmap.clear(); };

        /** return true if the container has no entry.
         */
        Bool empty() const { return this->cmap.empty(); };

        /** return the number of entries.
         */
        UInt4 size() const { return this->cmap.size(); };

        /** return true when the container contains the entry with the sepcified key.
         *  @param[in] key
         */
        Bool contain(const string& key) const { return this->cmap.find(key) != this->cmap.end(); };

        /** clear all entries.
         */
        void clear() { this->cmap.clear(); };

        /** add the value with the specified key to the container.
         *  the method do nothing when the container already has the entry with the specified key.
         *  @param[in] key
         *  @param[in] value
         */
        void add(const string& key, _SCALER_TYPE value);

        /** get the value of the entry with the sepcified key.
         *  @param[in] key
         *  @exception invalid_argument
         */
        _SCALER_TYPE get(const string& key) throw(invalid_argument);


        /** erase the entry with the specified key.
         *  the method do nothing when the container does not have the entry with the specified key.
         *  @param[in] key
         */
        void erase(const string& key);

        /** replace old value with key to new value.
         *  the method do nothing when the container does not have the entry with the specified key.
         *  @param[in] key
         *  @param[in] newValue
         */
        void replace(const string& key, _SCALER_TYPE value);

        /** get a list of keys for values.
         *  return an empty list when the container has no entry.
         */
        vector<string> getKeys() ;
       
        /** print all entry to the standard output
         *  @param[in] indent
         */
        void dump(const UInt4 indent=0, const UInt4 indentDepth=4, const string& keyTitile="key", const string& sizeTitle="size", const string& valueTitle="value");

};

template<typename _SCALER_TYPE>
void ContainerForScaler< _SCALER_TYPE>::add(const string& key, _SCALER_TYPE value) {

    if ( this->cmap.find(key) != this->cmap.end() ) {
        std::cerr << "Warning: ContainerForScale::add: " << __FILE__ << ":" << __LINE__ << ": already found the value with key \"" << key << "\"" << endl;
    } else {
        this->cmap.insert(std::pair<string, _SCALER_TYPE >(key, value));
    }
};

template<typename _SCALER_TYPE >
_SCALER_TYPE ContainerForScaler< _SCALER_TYPE >::get(const string& key) throw(invalid_argument) {
    if (this->cmap.find(key) == this->cmap.end()) {
        std::cerr << "Error: ContainerForScaler::get: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"" << endl;
        throw invalid_argument(string("ContainerForScaler::get: not found the entry with the specified key \"") + key + "\"\n");
    }
    return this->cmap[key];
}


template<typename _SCALER_TYPE >
void ContainerForScaler< _SCALER_TYPE >::erase(const string& key) {

    if ( this->cmap.find(key) == this->cmap.end() ) {
        std::cerr << "Warning: ContainerForScaler::erase: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"" << endl;
    } else {
        this->cmap.erase(this->cmap.find(key));
    }
};

template<typename _SCALER_TYPE >
void ContainerForScaler< _SCALER_TYPE >::replace(const string& key, _SCALER_TYPE newValue) {

    if ( this->cmap.find(key) == this->cmap.end() ) {
        std::cerr << "Warning: ContainerForScaler::replace: " << __FILE__ << ":" << __LINE__ << ": not found the value with the key \"" << key << "\"" << endl;
    } else {
        this->cmap[key]=newValue;
    }
};

template<typename _SCALER_TYPE >
vector<string> ContainerForScaler< _SCALER_TYPE >::getKeys() {

    vector<string> *retval = new vector<string>();
    if (! this->cmap.empty() ) {
        retval->resize(this->cmap.size());
        transform(this->cmap.begin(), this->cmap.end(), retval->begin(), ExtractKey< _SCALER_TYPE >());
    }
    return *retval;
    
};

template<typename _SCALER_TYPE>
void ContainerForScaler< _SCALER_TYPE>::dump(const UInt4 indent, const UInt4 indentDepth,
                                             const string& keyTitle, const string& sizeTitle, const string& valueTitle) {

    if (! this->cmap.empty() ) {
        size_t maxKeyWidth=0;
        maxKeyWidth = max(keyTitle.size(), max_element(this->cmap.begin(), this->cmap.end(), KeyLengthComp< _SCALER_TYPE >())->first.size());

        OutputTypeTitle ott; // size column is 0
        ott.title(indentDepth*indent, typeid( _SCALER_TYPE));  // subtitle
        ott.header(indentDepth*(indent+1), maxKeyWidth, keyTitle, 0, sizeTitle, valueTitle); // header

        for_each (this->cmap.begin(), this->cmap.end(), OutputScalerEntry< _SCALER_TYPE >(indentDepth*(indent+1), maxKeyWidth)); // key and values
        std::cout << endl;
    }
};

#endif // PARAMSET_PARTS_TEMPLATE
