/*
$Id: NeXusFileIO.hh 2104 2010-04-07 04:18:13Z jisuzuki $
 */

#ifndef NEXUSFILEIO
#define NEXUSFILEIO

#include "ReadNeXusFile.hh"
#include "WriteNeXusFile.hh"

//! NeXus file interface
/*!
  Data containers defined in Manyo Library can be stored in NeXus format
  with this class. This class is a wrapper of ReadNeXusFile and WriteNeXusFile.
 */
class NeXusFileIO{
public:

  /*!
    An data object of ElementContainerMatrix is stored in NeXus format.
    The input data object will be copied in this method. If you want to avoid it,
    use WriteP(ElementContainerMatrix *M, std::string FileName, std::string UserName, UInt4 CompMode=1).

    @param M input data container
    @param FileName output file name
    @param UserName Your name will be recorded in the output file.
    @param CompMode The default value is 1, but the speed of data writing is very slow with the default value.
    We recommend that CompMode=0 is set and compress it with tar or zip command after writing the file.
   */
  void Write( ElementContainerMatrix M, std::string FileName,
    std::string UserName, UInt4 CompMode=1 );

  /*!
    An data object of ElementContainerArray is stored in NeXus format.
    The input data object will be copied in this method. If you want to avoid it,
    use WriteP(ElementContainerArray *A, std::string FileName, std::string UserName, UInt4 CompMode=1).

    @param A input data container
    @param FileName output file name
    @param UserName Your name will be recorded in the output file.
    @param CompMode The default value is 1, but the speed of data writing is very slow with the default value.
     We recommend that CompMode=0 is set and compress it with tar or zip command after writing the file.
   */
  void Write( ElementContainerArray A, std::string FileName,
    std::string UserName, UInt4 CompMode=1 );

  /*!
    An data object of ElementContainer is stored in NeXus format.
    The input data object will be copied in this method. If you want to avoid it,
    use WriteP(ElementContainer *C, std::string FileName, std::string UserName, UInt4 CompMode=1).

    @param C input data container
    @param FileName output file name
    @param UserName Your name will be recorded in the output file.
    @param CompMode The default value is 1, but the speed of data writing is very slow with the default value.
     We recommend that CompMode=0 is set and compress it with tar or zip command after writing the file.
   */
  void Write( ElementContainer C, std::string FileName,
    std::string UserName, UInt4 CompMode=1 );

  /*!
    ElementContainerMatrix in NeXus file is obtained in this method.
    This method can be read NeXus files produced with both of CompMode=1 and 0.

    @param FileName Data file name read in this method.
   */
  ElementContainerMatrix ReadElementContainerMatrix( std::string FileName );

  /*!
     ElementContainerArray in NeXus file is obtained in this method.
    This method can be read NeXus files produced with both of CompMode=1 and 0.
   */
  ElementContainerArray ReadElementContainerArray( std::string FileName );

  /*!
     ElementContainer in NeXus file is obtained in this method.
    This method can be read NeXus files produced with both of CompMode=1 and 0.
   */
  ElementContainer ReadElementContainer( std::string FileName );

  /*!
     ElementContainer in NeXus file is obtained in this method.
     This method can be read NeXus files produced with both of CompMode=1 and 0.

     @return a pointer of ElmentContainer read in this method.
   */
  ElementContainer       *ReadElementContainerP( std::string FileName );

  /*!
     ElementContainerArray in NeXus file is obtained in this method.
     This method can be read NeXus files produced with both of CompMode=1 and 0.

     @return a pointer of ElmentContainerArray read in this method.
   */
  ElementContainerArray  *ReadElementContainerArrayP( std::string FileName );

  /*!
     ElementContainerMatrix in NeXus file is obtained in this method.
     This method can be read NeXus files produced with both of CompMode=1 and 0.

     @return a pointer of ElmentContainerMatrix read in this method.
   */
  ElementContainerMatrix *ReadElementContainerMatrixP( std::string FileName );

  /*!
    An data object of ElementContainerMatrix is stored in NeXus format.

    @param M input data container
    @param FileName output file name
    @param UserName Your name will be recorded in the output file.
    @param CompMode The default value is 1, but the speed of data writing is very slow with the default value.
     We recommend that CompMode=0 is set and compress it with tar or zip command after writing the file.
   */
  void WriteP( ElementContainerMatrix *M, std::string FileName,
               std::string UserName, UInt4 CompMode=1 );

  /*!
    An data object of ElementContainerArray is stored in NeXus format.

    @param A input data container
    @param FileName output file name
    @param UserName Your name will be recorded in the output file.
    @param CompMode The default value is 1, but the speed of data writing is very slow with the default value.
     We recommend that CompMode=0 is set and compress it with tar or zip command after writing the file.
   */
  void WriteP( ElementContainerArray *A, std::string FileName,
               std::string UserName, UInt4 CompMode=1 );

  /*!
    An data object of ElementContainer is stored in NeXus format.

    @param E input data container
    @param FileName output file name
    @param UserName Your name will be recorded in the output file.
    @param CompMode The default value is 1, but the speed of data writing is very slow with the default value.
     We recommend that CompMode=0 is set and compress it with tar or zip command after writing the file.
   */
  void WriteP( ElementContainer *E, std::string FileName,
               std::string UserName, UInt4 CompMode=1  );
};


#endif
