#ifndef DETECTORINFOEDITORREADOUT2D
#define DETECTORINFOEDITORREADOUT2D

#include "DetectorInfoEditorBase.hh"
//////////////////////////////////
// DetectorInfoEditorReadout2d
/////////////////////////////////

//!
/*!
 *
 */
class DetectorInfoEditorReadout2d:
    public DetectorInfoEditorBase
{
private:

    std::string _MessageTag;

protected:
    void Initialize();
    double CalcPixelSolidAngle( std::vector<Double>* pv, std::vector<Double>* v1 );
        //!< Calculate solid angle
        /*!< pv = pixel position std::vector
         *   v1[0] = the number of pixels on a PSD
         *   v1[1] to v1[8] = PositionInfoVect ( read from XML file )
         * ( v1[1],v1[2],v1[3] ) = P0 vec ( direction std::vector of detector )
         * ( v1[4],v1[5],v1[6] ) = L std::vector (PSD direction and length)
         *   v1[7] = L0 : length between posi of P0 and origin of L vec
         *   v1[8] = diameter of PSD
         *
         *   @param pv (std::vector<Double>*) position std::vector
         *   @param v1 (std::vector<Double>*) parameters
         *   @return None
         */
    std::vector<Double> CrossVectorNorm( std::vector<Double> A, std::vector<Double> B );
        //!< calculates Cross between vectors
        /*!<
         *   @param A (std::vector<Double>)
         *   @param B (std::vector<Double>)
         *   @return A x B with unit length
         */
public:

    DetectorInfoEditorReadout2d();
        //!< Constructor
        /*!<
         */
    DetectorInfoEditorReadout2d( std::string dfile, bool workAsReader=true );
        //!< Constructor
        /*!<
         *   @param dfile (std::string) DetectorInfo file path to be read
         *   @param workAsReader (bool)
         */
    ~DetectorInfoEditorReadout2d();
        //!< Destructor
        /*!<
         */
    //////////////////////////////////////////////////////////////////////////
    // public variables
    //////////////////////////////////////////////////////////////////////////
    std::vector< std::vector<Double>* >* detPositionInfoVect;  /**< detector axis vectors*/

    //////////////////////////////////////////////////////////////////////////
    // public functions
    //////////////////////////////////////////////////////////////////////////
    void Clear( UInt4 index=0 );
        //!< Clears stored parameters
        /*!<
         *   This deletes information by given index.
         *    index = 0 : all (default)
         *    index = 1 : InstrumentInfo
         *    index = 2 : TfpInfo
         *    index = 3 : TfpCalcParams
         *    index = 4 : PositionInfo
         *    index = 5 : BankInfo
         *
         *   @param index (UInt4)
         *   @return None
         */
    bool Read( std::string arg );
        //!< Reads DetectorInfo.xml and import information
        /*!<
         *   @param arg (std::string) path/to/DetectorInfo.xml or DetectorInfo as std::string
         *   @retval true : succeeded
         *   @retval false: failed
         */
    bool Write( std::string filepath );
        //!< Writes DetectorInfo.xml
        /*!<
         *   @param filepath (std::string)
         *   @retval true : succeeded
         *   @retval false: failed
         */
    std::string OutXml( bool withIndent=false );
        //!< Returns DetectorInfo as std::string
        /*!<
         *   @param withIndent(bool) false : output text without an indent format
         *   @return std::string
         */
    bool _makeOutputXmlReadout2d();
        //!< Makes XML format from stored information
        /*!<
         *   @param  None
         *   @retval true    succeded
         *   @retval false   failed
         */
    bool SetInfoAsReader();
        //!< Converts WiringInfo into the format to be read from histogram creation methods
        /*!<
         *   @param  None
         *   @retval true    succeded
         *   @retval false   failed
         */
    void ClearReader( UInt4 index=0 );
        //!< Clears the WiringInfo data to be read from histogram creation methods
        /*!<
         *   @param  index (UInt4)   index of variables to be cleared.
         *   @retval None
         */
    bool CalcReadout2dPixelPosition(UInt4 det_id, UInt4 x_size, UInt4 y_size, std::vector< std::vector<Double>* >* posi_vec );
        //!< Calculate pixel position
        /*!< This calculates pixel position by given detector ID, Pixel size
         *
         *   @param det_id    (UInt4) detector ID
         *   @param x_size    (UInt4) x-axis size
         *   @param y_size    (UInt4) y-axis size
         *   @param posi_vec  (std::vector< std::vector<Double> >) Positions [pixelNo][axes]  pixelNo= x_size*py + px
         *   @retval true             No Error
         *   @retval false            Error
         */
    std::vector<Double> CalcReadout2dPixelSolidAngle( UInt4 det_id, UInt4 x_size, UInt4 y_size, std::vector< std::vector<Double>* >* posi_vec );
        //!< Calculates Solid Angle by given detId and Pixel size
        /*!<
         *   @param det_id (UInt4)
         *   @param x_size    (UInt4) x-axis size
         *   @param y_size    (UInt4) y-axis size
         *   @param posi_vec  (std::vector< std::vector<Double> >) Positions [pixelNo][axes]  pixelNo= x_size*py + px
         *   @return std::vector of solid angle [pixelNo]
         */
};
#endif
