#include <iostream>
#include <istream>
#include <ostream>
#include <cstdio>
#include <cmath>
#include <cstdlib>

#include "Header.hh"
#include "ElementContainer.hh"
#include "ElementContainerArray.hh"

#include "Gaussian.hh"
#include "Lorentzian.hh"
#include "Domain.hh"
#include "ParamSet.hh"
#include "PeakData.hh"
#include "MovingAverage.hh"
#include "Levmar.hh"
//#include "CommonParamKeys.hh"
#include "PeakSearch.hh"


Double dist(const Double x, const UInt4 nPeaks, const Double a[], const Double c[], const Double w[]) {

    Double sum=0.0;
    for (UInt4 i=0; i<nPeaks; ++i) {
        sum += a[i]*exp(-1.0*pow((x-c[i])/w[i], 2.0));
    }
    return sum;
}

ElementContainer init_src(const Double xmin, const Double xmax, const UInt4 N) {

    Double a[]={ 30000.0, 100000.0, 100000.0, 30000.0 };
    Double c[]={     3.0,      5.0,     10.0,    13.0 };
    Double w[]={     1.0,      2.5,      2.5,     1.0 };
    const UInt4 N_PEAKS=sizeof(a)/sizeof(a[0]);


    vector<Double> bin = *(new vector<Double>());
    Double delta=(xmax-xmin)/N;
    for (UInt4 i=0; i<=N; ++i) {
        bin.push_back(xmin + static_cast<Double>(i)*delta);
    }

    vector<Double> y = *(new vector<Double>());
    vector<Double> e = *(new vector<Double>());
    Double x, p, q, s, t;
    for (UInt4 i=0; i<N; ++i) {
        x=(bin.at(i) + bin.at(i+1))/2.0;
        p=dist(x, N_PEAKS, a, c, w);
        t=static_cast<Double>(rand())/static_cast<Double>(RAND_MAX);
        s=static_cast<Double>(rand())/static_cast<Double>(RAND_MAX)-0.5;
        q=(s > 0.0 ? 1.0 : (s==0.0 ? 0.0: -1.0))*(p/10.0)*sqrt(-1.0*log(t));
        y.push_back(floor(p+q));
        e.push_back(p/10.0);
    }

    ElementContainer ec = *(new ElementContainer());
    ec.AddToHeader("run number", 1);
    ec.AddToHeader("run level",  1);
    ec.AddToHeader("Inst.", "manyo");
    ec.Add("TOF",       bin, "sec."  );
    ec.Add("Intensity", y,   "count.");
    ec.Add("Error",     e,   "count.");
    ec.SetKeys("TOF", "Intensity", "Error");

    return ec;
}

Int4 main(Int4 argc, Char *argv[]) {

    Double xmin;
    Double xmax;
    UInt4  nDiv;

    std::cout << "xmin= "; std::cin >> xmin;
    std::cout << "xmax= "; std::cin >> xmax;
    std::cout << "nDiv= "; std::cin >> nDiv;
    ElementContainer src=init_src(xmin, xmax, nDiv);
    src.Dump();

    PeakSearch peakSearch = *(new PeakSearch(&src, new MovingAverage));
    peakSearch.setParam(MovingAverage::WINDOW_WIDTH, 201U);
    peakSearch.setParam(MovingAverage::WINDOW_UPPER, 100U);
    peakSearch.execute();
    PeakData peakData = peakSearch.getPeaks();
    peakData.Dump();

    Levmar levmar = *(new Levmar());

    /* domain */
    Domain domain;
    domain.setSource(src);
    domain.setRange(0, src.PutSize(src.PutYKey())-1);

    /* functions */
    vector<FuncBase*> funcList = *(new vector<FuncBase*>());
    funcList.push_back(new Gaussian());
    funcList.push_back(new Lorentzian());

    /* default values for fittinng parameters */
    vector<Double> fittingParams = peakData.toVector();
    vector<Double> referenceData = *(new vector<Double>());
    vector<Double> lowerBounds   = *(new vector<Double>());
    vector<Double> upperBounds   = *(new vector<Double>());

    string yKey=src.PutYKey();
    string eKey=src.PutEKey();
    for (UInt4 i=domain.getLowerBoundID(); i<=domain.getUpperBoundID(); ++i) {
        referenceData.push_back(src.Put(yKey, i)/src.Put(eKey, i));
    }

    for (UInt4 i=0; i < fittingParams.size(); ++i) {
        lowerBounds.push_back(fittingParams.at(i)*0.9);
        upperBounds.push_back(fittingParams.at(i)*1.1);
    }

    ParamSet param=levmar.setDefaultParam(src);
    param.add(Levmar::FUNCTIONS, funcList);
    param.add(Levmar::REFERENCE_VALUES, referenceData);
    param.add(Levmar::PARAMETER_VALUES, fittingParams);
    param.replace(Levmar::USE_DATA_WEIGHTS, true);

    param.add(Levmar::LOWER_BOUNDS, lowerBounds);
    param.add(Levmar::UPPER_BOUNDS, upperBounds);
    param.replace(Levmar::OUTPUT_INTERVAL, 50U);
    //param.add(PEAK_DATA, peakData);
    param.dump();

    std::cerr<< "check param: " << levmar.checkParam(src, domain, param) << endl;

    levmar.toInnerForm(src, domain, param);
    levmar.fit();
    levmar.eval();

    ElementContainer result;
    levmar.toElementContainer(src, result);
    ParamSet fittedParam = levmar.getFittedParam();
    fittedParam.dump();
    //ElementContainerArray ecArray=levmar.getResultComponents(src);
    //ElementContainer fc=ecArray.Put(1);
    /*
    */
}
