#include "AdvOperationBase.hh"


/** 
 *  Operation Base
 *
 *  @author TANIMORI Souichirou, AdvanceSoft Corp.
 *  @version 0.0
 */

//const string AdvOperationBase::DOMAIN=string("domain");

const string AdvOperationBase::className=string("AdvOperationBase");

/** 
 *  constructor
 */
AdvOperationBase::AdvOperationBase() {

    string memberName=string("AdvOperationBase()");

    this->source=NULL;
    DebugMessage(className, memberName, "initialize source to Null\n");
    this->method=NULL;
    DebugMessage(className, memberName, "initialize method to Null\n");
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}


/**
 *  constructor
 *
 * \param[in] src an element conatier as the source data
 */
AdvOperationBase::AdvOperationBase(ElementContainer *src) {
    string memberName=string("AdvOperationBase(ElementContainer *)");

    this->setSource(src);
    this->method=NULL;
    DebugMessage(className, memberName, "initialize method to Null\n");
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

/**
 *  constructor
 *
 *  \param[in] src     an element container as the source data
 *  \param[in] method  the method for an operation
 */
AdvOperationBase::AdvOperationBase(ElementContainer *src, AdvMethod *method) {
    string memberName=string("AdvOperationBase(ElementContainer *, AdvMethod *)");

    this->setSource(src);
    this->setMethod(method);
    this->setDefaultParam();
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

/**
 *  constructor
 *
 *  \param[in] src        an element container as the source data
 *  \param[in] methodtype the method type for an operation
 */
AdvOperationBase::AdvOperationBase(ElementContainer *src, const AdvMethodType &methodType) {
    string memberName=string("AdvOperationBase(ElementContainer *, const AdvMethodtype *)");

    this->setSource(src);
    this->setMethod(methodType);
    this->setDefaultParam();
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

/**
 *  constructor
 *
 *  \param[in] src        an element container as the source data
 *  \param[in] methodtype the method type for an operation
 */
AdvOperationBase::AdvOperationBase(ElementContainer *src, const string &methodName) {
    string memberName=string("AdvOperationBase(ElementContainer *, const AdvMethodtype *)");

    this->setSource(src);
    this->setMethod(methodName);
    this->setDefaultParam();
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

/**
 *  constructor
 *
 *  \param[in] src     an element container as the source data
 *  \param[in] method  the method for an operation
 *  \param[in] xLower  the lower bound of the domain for the operation
 *  \param[in] xUpper  the upper bound of the domain for the operation
 */
AdvOperationBase::AdvOperationBase(ElementContainer *src, AdvMethod *method, const Double xLower, const Double xUpper) {
    string memberName=string("AdvOperationBase(ElementContainer *, AdvMethod *, const Double, const Double)");

    this->setSource(src);
    this->setMethod(method);
    this->setDefaultParam();
    this->setDomain(xLower, xUpper);
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

AdvOperationBase::AdvOperationBase(ElementContainer *src, const AdvMethodType &methodType, const Double xLower, const Double xUpper) {
    string memberName=string("AdvOperationBase(ElementContainer *, const AdvMethodtype&, const Double, const Double)");

    this->setSource(src);
    this->setMethod(methodType);
    this->setDefaultParam();
    this->setDomain(xLower, xUpper);
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

AdvOperationBase::AdvOperationBase(ElementContainer *src, const string &methodName, const Double xLower, const Double xUpper) {
    string memberName=string("AdvOperationBase(ElementContainer *, const AdvMethodtype&, const Double, const Double)");

    this->setSource(src);
    this->setMethod(methodName);
    this->setDefaultParam();
    this->setDomain(xLower, xUpper);
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

/**
 *  constructor
 *
 *  \param[in] src     an element container as the source data
 *  \param[in] method  the method for an operation
 *  \param[in] lower  the index for the lower bound of the domain for the operation
 *  \param[in] upper  the index for the upper bound of the domain for the operation
 */
AdvOperationBase::AdvOperationBase(ElementContainer *src, AdvMethod *method, const UInt4 lower, const UInt4 upper) {
    string memberName=string("AdvOperationBase(ElementContainer *, AdvMethod *, const UInt4, const UInt4)");

    this->setSource(src);
    this->setMethod(method);
    this->setDefaultParam();
    this->setDomain(lower, upper);
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

AdvOperationBase::AdvOperationBase(ElementContainer *src, const AdvMethodType &methodType, const UInt4 lower, const UInt4 upper) {

    string memberName=string("AdvOperationBase(ElementContainer *, const AdvMethodType &, const UInt4, const UInt4)");

    this->setSource(src);
    this->setMethod(methodType);
    this->setDefaultParam();
    this->setDomain(lower, upper);
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

AdvOperationBase::AdvOperationBase(ElementContainer *src, const string &methodName, const UInt4 lower, const UInt4 upper) {
    string memberName=string("AdvOperationBase(ElementContainer *, const estring &, const UInt4, const UInt4)");

    this->setSource(src);
    this->setMethod(methodName);
    this->setDefaultParam();
    this->setDomain(lower, upper);
    this->result=new ElementContainer();
    DebugMessage(className, memberName, "create ElementContainer for the result\n");
}

/** 
 *  constructor with determing a AdvDomain for the operation
 *
 *  \deprecated
 *  \param[in] src     an element container as the source data
 *  \param[in] xLower  the lower bound of the domain for the operation
 *  \param[in] xUpper  the upper bound of the domain for the operation
 */
AdvOperationBase::AdvOperationBase(ElementContainer *src, const Double xLower, const Double xUpper) {
    this->setSource(src);
    this->domain.setType(AdvDomain::CC);
    this->domain.setRange(xLower, xUpper);
    this->result=new ElementContainer(src->PutHeader());
}

/**
 *  destructor
 */
AdvOperationBase::~AdvOperationBase() {
    string memberName=string("~AdvOperationBase()");
    DebugMessage(className, memberName, "enter\n");
    this->result=static_cast<ElementContainer*>(NULL);
    this->method=static_cast<AdvMethod*>(NULL);
    this->source=static_cast<ElementContainer*>(NULL);
    DebugMessage(className, memberName, "exit\n");
};


/**
 *  set an element container as source data for the operation
 *
 *  \param[in] src  an element container as the source data
 */
void AdvOperationBase::setSource(ElementContainer *src) {
    string memberName=string("setSource");

    assert(src != static_cast<ElementContainer*>(NULL));
    assert(src->PutSize(src->PutXKey()) >= 2);
    this->source=src;
    DebugMessage(className, memberName, "\n");
    this->domain.setSource(*src);
    DebugMessage(className, memberName, "initialize domain (set source)\n");
}

/**
 *  set an algorithm for the operation
 *
 *  \param[in] methodc an instanse of the class implementsd algorithm for the operation
 */
void AdvOperationBase::setMethod(AdvMethod *method) {
    string memberName=string("setMethod(AdvMethod *)");

    assert(method != static_cast<AdvMethod*>(NULL));
    this->method = method;
    DebugMessage(className, memberName, "\n");
}

/**
 *  set an algorithm for the operation
 *
 *  \param[in] type method type
 */
void AdvOperationBase::setMethod(const AdvMethodType &type) {
    string memberName=string("setMethod(const AdvMethodType &)");
    DebugMessage(className, memberName, "enter\n");
    DebugMessage(className, memberName, __FILE__, __LINE__, "id=%d name=%s symbol=%s\n", type.value, type.name.c_str(), type.symbol.c_str());
    DebugMessage(className, memberName, __FILE__, __LINE__, "id=%d name=%s symbol=%s\n", UNKNOWN_METHOD_TYPE.value, UNKNOWN_METHOD_TYPE.name.c_str(), UNKNOWN_METHOD_TYPE.symbol.c_str());

    assert( type.value != UNKNOWN_METHOD_TYPE.value);
    AdvMethodFactory *factory = AdvMethodFactory::getInstance();
    this->method = factory->createMethod(type);
    DebugMessage(className, memberName, "exit\n");
}


/**
 *  set an algorithm for the operation
 *
 * \param[in] methodName method name
 */
void AdvOperationBase::setMethod(const string &methodName) {
    string memberName=string("setMethod(const string &)");

    DebugMessage(className, memberName, "enter\n");
//    assert( methodName == BSPLINE.name   || methodName == MOVING_AVERAGE.name   || methodName == LEVMAR.name ||
//            methodName == BSPLINE.symbol || methodName == MOVING_AVERAGE.symbol || methodName == LEVMAR.symbol); 

    AdvMethodFactory *factory =          AdvMethodFactory::getInstance();
    this->method = factory->createMethod(methodName);
    DebugMessage(className, memberName, "exit\n");
}


/** 
 *  set member variables by given values
 *
 *  \deprecated
 *  \param[in] src     an element container as the source data
 *  \param[in] xLower  the lower bound of the domain for the operation
 *  \param[in] xUpper  the upper bound of the domain for the operation
 */
void AdvOperationBase::setDomain(ElementContainer *src, const Double xLower, const Double xUpper) {

    assert(src->PutSize( src->PutXKey()) >= 2 );
    assert(this->result->PutHeaderPointer() != static_cast<HeaderBase*>(NULL));

    this->setSource(src);
    this->domain.setRange(xLower, xUpper);
}


/** 
 *  set the domain domain by giving the both bounds
 *
 *  \param[in] xLower  the lower bound of the domain for the operation
 *  \param[in] xUpper  the upper bound of the domain for the operation
 */
void AdvOperationBase::setDomain(const Double xLower, const Double xUpper) {
    string memberName=string("setDomain(const Double, const Double)");

    DebugMessage(className, memberName, "enter\n");
    this->domain.setRange(xLower, xUpper);
    DebugMessage(className, memberName, "exit\n");
}


/**
 *  set the domain by the indices for the both bounds
 *
 *  \param[in] lower  the index for the lower bound of the domain for the operation
 *  \param[in] upper  the index for the upper bound of the domain for the operation
 */
void AdvOperationBase::setDomain(const UInt4 lower, const UInt4 upper) {
    string memberName=string("setDomain(const UInt4, const UInt4)");

    DebugMessage(className, memberName, "enter\n");
    this->domain.setRange(lower, upper);
    DebugMessage(className, memberName, "exit\n");
}

void AdvOperationBase::setDomainByBinBoundID(const UInt4 lower, const UInt4 upper) {
    string memberName=string("setDomainByBinBoundID(const UInt4, const UInt4)");

    DebugMessage(className, memberName, "enter\n");
    this->domain.setBinBoundRange(lower, upper);
    DebugMessage(className, memberName, "exit\n");
}

void AdvOperationBase::setDomainByBinID(const UInt4 lower, const UInt4 upper) {
    string memberName=string("setDomainByBinID(const UInt4, const UInt4)");

    DebugMessage(className, memberName, "enter\n");
    this->domain.setBinRange(lower, upper);
    DebugMessage(className, memberName, "exit\n");
}


/**
 *  set the types of the domain by bin bounds
 */
void AdvOperationBase::setDomain(const AdvDomain::BoundsType type) {
    this->domain.setType(type);
}


/**
 *  \return the domain for the operation
 */
AdvDomain AdvOperationBase::getDomain() {
    return this->domain;
}

/**
 *  \return the lower bound of the domain
 */
Double AdvOperationBase::getLowerBound() {
    return this->domain.getLowerBound();
}

/**
 *  \return the upper bound of the domain
 */
Double AdvOperationBase::getUpperBound() {
    return this->domain.getUpperBound();
}

/**
 *  \return the index of the lower bound of the domain
 */
UInt4 AdvOperationBase::getLowerBoundID() {
    return this->domain.getLowerBoundID();
}

/**
 *  \return the index of the upper bound of the domain
 */
UInt4 AdvOperationBase::getUpperBoundID() {
    return this->domain.getUpperBoundID();
}


/**
 *  set the parameters for the operation
 *
 *  \param[in] param  a set of parameters for the operaion
 */
void AdvOperationBase::setParam(AdvParamSet param) {
    this->param=param;
}

/**
 *  set the parameters for the operation to default values
 */
void AdvOperationBase::setDefaultParam() {
    string memberName=string("setDefaultParam");
    DebugMessage(className, memberName, "enter\n");
    this->param=this->method->setDefaultParam(*(this->source));
    DebugMessage(className, memberName, "set a set of parames to default values\n");
    //this->domain.setRange(static_cast<UInt4>(0), this->source->PutSize( this->source->PutYKey() )-1U );
    this->domain.setBinBoundRange(static_cast<UInt4>(0), this->source->PutSize( this->source->PutXKey() )-1U );
    DebugMessage(className, memberName, "set range of the domain\n");
    DebugMessage(className, memberName, "exit\n");
}

/**
 *  \return a set of parameters as result of the operation
 */
AdvParamSet AdvOperationBase::getParam() {
    return this->param;
}

/**
 *  \return a set of fitted parameters
 */
AdvParamSet AdvOperationBase::getFittedParam() {
    return this->method->getFittedParam();
}

#include <typeinfo>
/**
 *  set a boolean parameter with key to the given value
 *
 *  \param[in] key   the key for the boolean parameter
 *  \param[in] value the value of the boolean parameter
 */
void AdvOperationBase::setParam(string key, const Bool value) {
    string memberName=string("setParam(string, Bool");
    DebugMessage(className, memberName, "key \"%s\", value's type; %s\n", key.c_str(), typeid(value).name());

    if (! this->param.contain(key) ) {
        this->param.add(key, value);
    } else {
        this->param.replace(key, value);
    }
}

/**
 *  set a Int4 type parameter with key to the given value
 *
 *  \param[in] key   the key for the Int4 type parameter
 *  \param[in] value the value of the Int4 type parameter
 */
void AdvOperationBase::setParam(string key, const Int4 value) {
    string memberName=string("setParam(string, Int4)");
    DebugMessage(className, memberName, "key \"%s\", value's type; %s\n", key.c_str(), typeid(value).name());

    if (! this->param.contain(key)) {
         this->param.add(key, value); 
    } else {
         this->param.replace(key, value); 
    }
}

/**
 *  set a UInt4 type parameter with key to the given value
 *
 *  \param[in] key   the key for the UInt4 type parameter
 *  \param[in] value the value of the UInt4 type parameter
 */
void AdvOperationBase::setParam(string key, const UInt4 value) {
    string memberName=string("setParam(string, UInt4)");
    DebugMessage(className, memberName, "key \"%s\", value's type; %s\n", key.c_str(), typeid(value).name());

    if ( ! this->param.contain(key) ) {
        this->param.add(key, value);

    } else {
        this->param.replace(key, value); 
    }
}

/**
 *  set a Double type parameter with key to the given value
 *
 *  \param[in] key   the key for the Double type parameter
 *  \param[in] value the value of the Double type parameter
 */
void AdvOperationBase::setParam(string key, const Double value) {
    string memberName=string("setParam(string, Double)");
    DebugMessage(className, memberName, "value's type; %s\n", typeid(value).name());

    if ( ! this->param.contain(key) ) {
        this->param.add(key, value);
    } else {
        this->param.replace(key, value);
    }
}


/**
 *  set a Double type vector with key
 *
 *  \param[in] key   the key for the Double type vector
 *  \param[in] value the Double type vector
 */
void AdvOperationBase::setParam(string key, vector<Double> value) {

    if ( ! this->param.contain(key) ) {
        this->param.add(key, value);
    } else {
        this->param.replace(key, value); 
    }
}

/**
 *  set the i-th component of the Double type vector with key
 *
 *  \param[in] key   the key for the Double type vector
 *  \param[in] i     the incex for the compoment of the vector
 *  \param[in] value the value of the compoment
 */
void AdvOperationBase::setParam(string key, const UInt4 i, const Double value) {

    string memberName=string("setParam(string, UInt4, Double)");

    if ( ! this->param.contain(key) ) {
        errorMessage(className, memberName, "no data with key %s", key.c_str());
    } else {
        this->param.replace(key, i, value);
    }
}

/**
 *  set a Double type matrix with key
 *
 *  \param[in] key   the key for the Double type matrix
 *  \param[in] value the Double type matrix
 */
void AdvOperationBase::setParam(string key, vector< vector<Double> > value) {

    if ( ! this->param.contain(key) ) {
        this->param.add(key, value);
    } else {
        this->param.replace(key, value); 
    }
}

/**
 *  set the (i, j) component of the Double type matrix with key
 *
 *  \param[in] key   the key for the Double type matrix
 *  \param[in] i     the row    incex for the compoment of the matrix
 *  \param[in] j     the column incex for the compoment of the matrix
 *  \param[in] value the value of the compoment
 */
void AdvOperationBase::setParam(string key, const UInt4 i, const UInt4 j, const Double value) {

    string memberName=string("setParam(string&, UInt4, UInt4, Double)");

    if ( ! this->param.contain(key) ) {
        errorMessage(className, memberName, "no data with key %s", key.c_str());
    } else {
        this->param.replace(key, i, j, value);
    }
}


/**
 *  set a list of peaks with key
 *
 *  \param[in] key   the key for a list of peaks
 *  \param[in] value a list of peaks
 */
void AdvOperationBase::setParam(string key, AdvPeakData value) {

    if ( ! this->param.contain(key) ) {
        this->param.add(key, value);
    } else {
        this->param.replace(key, value);
    }
}

/**
 *  set a list of functions as a function
 *
 *  \param[in] key   the key for a function list
 *  \param[in] value a list of functions
 */
void AdvOperationBase::setParam(string key, vector<AdvFuncBase*> value) {
    if ( ! this->param.contain(key) ) {
        this->param.add(key, value);
    } else {
        this->param.replace(key, value);
    }
}

/**
 *  set expr as a list of functions
 *
 *  \param[in] key  the key for a expression
 *  \paran[in] expr an expression for a list of functions
 */
void AdvOperationBase::setParam(string key, string expr) {
    string memberName = string("setParam(string, string)");

    AdvFuncParser *parser = new AdvFuncParser(expr);
    vector<AdvFuncBase*> value=parser->parse();
    if (value.empty()) {
        errorMessage(className, memberName, "fail to craete function list: %s\n", expr.c_str());
    } else {
        if ( ! this->param.contain(key) ){
            this->param.add(key, value);
        } else {
            this->param.replace(key, value);
        }
    }
}


void AdvOperationBase::setParam(string key, vector< vector<AdvFuncBase*> > value) {
    if ( ! this->param.contain(key) ) {
        this->param.add(key, value);
    } else {
        this->param.replace(key, value);
    }
}

void AdvOperationBase::setParam(string key, vector<string> expr) {
    string memberName = string("setParam(string, vector<string>)");

    vector< vector<AdvFuncBase*> > funcMatrix = *(new vector< vector<AdvFuncBase*> >());
    for (std::vector<string>:: const_iterator elem=expr.begin(); elem != expr.end(); ++elem) {
        AdvFuncParser *parser = new AdvFuncParser(*elem);
        vector<AdvFuncBase*> funcList=parser->parse();
        if (funcList.empty()) {
            errorMessage(className, memberName, "fail to craete function list: %s\n", (*elem).c_str());
            return;
        }
        funcMatrix.push_back(funcList);
    }

    if ( ! this->param.contain(key) ) {
        this->param.add(key, funcMatrix);
    } else {
        this->param.add(key, funcMatrix);
    }
}

/**
 *  set python object as Double type vector
 *
 *  \param[in] key   the key for a vector
 *  \param[in] value a pyhon object as a Double type vecotr
 */
void AdvOperationBase::setParam(string key, PyObject *value) {
    if ( ! this->param.contain(key) ) {
       this->param.add(key, value);
    } else {
       this->param.replace(key, value);
    }
}

/**
 *  check consistency of paremters for the operation
 *
 *  \retval true if paramters are consistent
 */
Bool AdvOperationBase::checkParam() {
    string memberName=string("checkParam");

    DebugMessage(className, memberName, "enter\n");
    return this->method->checkParam(*(this->source), this->domain, this->param);
}

/**
 *  \return an element container as the result of operarion
 */
ElementContainer AdvOperationBase::getResult() {
    string memberName = string("getResult()");

    DebugMessage(className, memberName, "enter\n");
    ElementContainer *result=new ElementContainer();
    this->method->toElementContainer(*(this->source), *result);
    DebugMessage(className, memberName, "exit\n");
    return *result;
}
